// range.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// An extremely useful class for storing parameters that involve a minimum and
// maximum value.

#ifndef RANGE_H
#define RANGE_H

#include "localdefs.h"

class Range {
public:
	Range() : minval(0), maxval(0) {}
	Range(double mn, double mx) : minval(mn), maxval(mx) {}
	Range(float mn, float mx) : minval(mn), maxval(mx) {}
	Range(int mn, int mx) : minval(mn), maxval(mx) {}
	Range(const Range &r) : minval(r.minval), maxval(r.maxval) {}
	~Range() {}
	Range & set(double mn, double mx) {
		minval = mn; maxval = mx; return *this; } 
	double min() const { return minval; }
	double max() const { return maxval; }
	int intMin() const { return int(minval); }
	int intMax() const { return int(maxval); }
	double absoluteMax() const {
		return ::max(abs(maxval), abs(minval));
	}
	double spread() const { return abs(maxval - minval); }
	int size() const { return int(spread()) + 1; }
	double mean() const { return (maxval + minval) / 2.0; }
	int intMean() const { return int(mean()); }
	int includesZero() const { return (maxval >= 0.0 && minval <= 0.0); }
	int isPositive() const { return (maxval > 0.0 && minval >= 0.0); }
	int isNegative() const { return (maxval <= 0.0 && minval < 0.0); }
	int includes(double value) const {
		return (value >= minval && value <= maxval);
	}
	int includes(int value) const { return includes(double(value)); }
	int includes(const Range &r) const {
		return (r.minval >= minval && r.maxval <= maxval);
	}
	int overlaps(const Range &r) const {
		return (includes(r.minval) || includes(r.maxval));
	}
	Range & limit(double newmin, double newmax) {
		minval = ::max(::min(minval, newmax), newmin);
		maxval = ::min(::max(maxval, newmin), newmax);
		return *this;
	}
	Range & expandBy(double amount) {
		minval -= amount; maxval += amount; return *this;
	}
	Range & expandBy(int amount) { return expandBy(double(amount)); }
	Range & check() {
		if(spread() == 0.0) expandBy(1);
		return *this;
	}
	Range & operator *= (double x) {
		minval *= x; maxval *= x; return *this;
	} 
	Range & operator += (int x) {
		minval += x; maxval += x; return *this;
	} 
	Range & operator -= (int x) {
		minval -= x; maxval -= x; return *this;
	} 
	boolean operator == (const Range& r) const {
		return minval == r.minval &&  maxval == r.maxval;
	} 
	boolean operator != (const Range& r) const {
		return minval != r.minval ||  maxval != r.maxval;
	}
	Range & scaleBy(double factor) {
		if(min() == 0.0 || max() == 0.0)
			*this *= factor;
		else
			expandBy(spread() * (factor - 1.0) / 2.0);
		return *this;
	}
private:
	friend boolean operator > (double, const Range&);
	friend boolean operator < (double, const Range&);
	friend boolean operator >= (double, const Range&);
	friend boolean operator <= (double, const Range&);
	double minval, maxval;
};

inline boolean operator > (double val, const Range& r) {
	return val > r.maxval && val > r.minval;
}

inline boolean operator < (double val, const Range& r) {
	return val < r.minval && val < r.maxval;
}

inline boolean operator >= (double val, const Range& r) {
	return val > r || r.includes(val);
}

inline boolean operator <= (double val, const Range& r) {
	return val < r || r.includes(val);
}

inline Range operator + (const Range& x, int y) {
	return Range(x.min() + y, x.max() + y);
}

inline Range operator - (const Range& x, int y) {
	return Range(x.min() - y, x.max() - y);
}

inline Range operator * (const Range& x, double y) {
	return Range(x.min() * y, x.max() * y);
}

inline Range operator / (const Range& x, double y) {
	return Range(x.min() / y, x.max() / y);
}

#endif
