// soundheader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <sys/time.h>
#include "localdefs.h"
#include "application.h"
#include "comment.h"
#include "datafile.h"
#include "soundheader.h"
#include "aiff_header.h"
#include "wavheader.h"
#include "typeconvert.h"

SoundHeader::Type SoundHeader::default_HeaderType = HEADER_DEFAULT_TYPE;

// create Sound Header class instance appropriate for the magic number found
// in the file.  Default case is to create a HybridHeader and let it handle
// all other possibilities (such as raw data)

SoundHeader *
SoundHeader::create(DataFile* file, DataType type, int srate, int chans, double peak) {
	Type headerType = defaultHeaderType();	// default is reset if magic read
	if(file && file->dataSize() > 0) {			// check for existing file
		int magic = Header::readMagicNumber(file);
		headerType = (magic == SF_MAGIC) ? Ircam
				   : (magic == FORM) ? Aifc
				   : (magic == RIFF || magic == FFIR) ? Wav
				   : Hybrid;
	}
	return create(headerType, type, srate, chans, peak);
}

SoundHeader *
SoundHeader::create(
	SoundHeader::Type htype, DataType type, int srate, int chans, double peak
) {
	SoundHeader* header = nil;
	switch(htype) {
	case Ircam:		// old 1024-byte BSD header
		header = new IrcamHeader(type, srate, chans, peak);
		break;
	case Aifc:		// AIF-C header for Macs and SGIs
		header = new AIFFSoundHeader(type, srate, chans, peak);
		break;
	case Wav:		// WAVE format used on Intel machines (MS Windows)
		header = new WAVSoundHeader(type, srate, chans);
		break;
	case Snd:		// 28 byte NeXT or 24 byte SPARC au header
		header = new SndHeader(type, srate, chans);
		break;
	case None:
	case Hybrid: 	// Lansky's BSD header with Snd header at the beginning
	default:
		header = new HybridHeader(type, srate, chans, peak);
		break;
	}
	if(htype == None) header->setRaw();	// writing headerless file
	return header;
}

const char*
SoundHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Unknown or invalid soundfile magic number:  %d", magic());
	return msg;
}

int
SoundHeader::checkHeader() {
	char msg[64];
	msg[0] = '\0';	// null for later check
	int retcode = 0;
	if(nchans != 1 && nchans != 2 && nchans != 4)
		sprintf(msg, "Bad header: nchans= %d", nchans);
	else if(samprate < 4000 || samprate > 64000)
		sprintf(msg, "Bad header: samp rate= %d", samprate);
	else if(data_type == NoData)
		sprintf(msg, "Bad header: unknown data type");
	else if(!isRaw() && !isValid(data_type))	// dont check on raw read
		sprintf(msg, "Illegal sample format for this header type.");
	else
		retcode = 1;
	if(strlen(msg))
		Application::alert(msg);
	return retcode;
}

// this is the bytes/sec conversion needed to determine offsets in time

int
SoundHeader::bytesPerSecond() {
	return sampleRate() * nChans() * sampleSize();
}

//********

boolean
SndOrIrcamSoundHeader::isMagic() {
	return (magic() == SND_MAGIC || magic() == SF_MAGIC);
}

int
SndOrIrcamSoundHeader::readInfo(DataFile *file) {
	int status = false;
	switch(magic()) {
	case SND_MAGIC:	// native or hybrid header
		status = readSndInfo(file);
		break;
	case SF_MAGIC:		// Ircam header
		status = readIrcamInfo(file);
		break;
	default:			// we should never get here, since checked earlier
		data_type = NoData;
		nchans = 0;
		status = false;
	}
	return status && file->good();
}

int
SndOrIrcamSoundHeader::readSndInfo(DataFile *file) {
	SF_Header hdr;
	int status = true;
	if(file->read((void *) &hdr, sizeof(hdr)).good()) {
		header_type = (hdr.sfinfo.sf_magic == SF_MAGIC) ? Hybrid : Snd;
		U_SND_Header *sh = &hdr.sfinfo.sndheader;
		setDataOffset(sh->data_loc);
		setDataSize(sh->data_size);
		data_type = format_to_type(sh->format);
		nchans = sh->nchans;
		samprate = sh->samp_rate;
		if((status = checkHeader()) == true) {
			char msg[64];
			const char* msg1 = nil;
			const char* msg2 = nil;
			// correct in event of missing or corrupt offset information
			int minOffset = (header_type == Snd) ?
				sizeof(SND_Header) : sizeof(SF_Header);
			if(dataOffset() < minOffset) {
				sprintf(msg, "Incorrect header data offset (%d < %d)",
					dataOffset(), minOffset);
				msg1 = msg;
				setDataOffset(minOffset);
			}
			int trueSize = file->dataSize() - dataOffset();
			if(dataSize() != trueSize) {
				msg2 = "Incorrect data size in header -- adjusting.";
				setDataSize(trueSize);
			}
			if(msg1 || msg2)
				Application::alert("Warning:",
					msg1? msg1 : msg2, msg1? msg2 : nil);
			if(header_type == Hybrid) {
				extractPeak(&hdr.sfinfo.sf_codes);
				extractComment(&hdr.sfinfo.sf_codes);
			}
			else if(dataOffset() > sizeof(SND_Header))
				setComment((char*) sh + sizeof(SND_Header) - INFO_SIZE);
		}
	}
	else {
		Application::alert("SoundHeader::readSndInfo:  read error.");
		status = false;
	}
	return status;
}

int
SndOrIrcamSoundHeader::readIrcamInfo(DataFile *file) {
	IRCAM_Header hdr;
	int status = true;
	if(file->read((void *) &hdr, sizeof(hdr)).good()) {
		header_type = Ircam;
		setDataOffset(sizeof(IRCAM_Header));
		data_type = sampsize_to_type(hdr.sfinfo.sf_packmode);
		nchans = hdr.sfinfo.sf_chans;
		samprate = int(hdr.sfinfo.sf_srate);
		if((status = checkHeader()) == true) {
			extractPeak(&hdr.sfinfo.sf_codes);
			extractComment(&hdr.sfinfo.sf_codes);
		}
	}
	else {
		Application::alert("SoundHeader::readIrcamInfo: read error.");
		status = false;
	}
	return status;
}

int
SndOrIrcamSoundHeader::readComment(DataFile *f) {
	// do nothing here;  comment extracted from header, not read directly
	return f->good();
}

int
SndOrIrcamSoundHeader::writeComment(DataFile *f) {
	// do nothing here;  comment written into header, not directly
	return f->good();
}

const char *
SndOrIrcamSoundHeader::getSFCode(const char *hd, int code) {
	const char *sfc = hd;
	const char *hdend = hd + diskHeaderCommentSize();
	while(sfc < hdend) {
		SFCode *sp = (SFCode *) sfc;
		if(sp->code == SF_END)
				break;
		/* Catch possible wrap around on stack from bad header */
		/* or a zero struct size from bad header */
		if(sp->bsize <= 0 || sfc + sp->bsize < hd)
				break;
		if(sp->code == code)
				return(sfc);
		sfc += sp->bsize;
	}
	return(NULL);
}

int
SndOrIrcamSoundHeader::extractPeak(const char *ptr) {
	const char *codeptr;
	peakamp = 0;
	if(ptr) {
		SFMaxamp sfm;
		if((codeptr = getSFCode(ptr, SF_MAXAMP)) != NULL) {
			bcopy(codeptr + sizeof(SFCode), (char *) &sfm,
				sizeof(SFMaxamp));
			for(int i = 0; i < nchans && i < MaxChannels; i++)
				if(sfm.value[i] > peakamp)
					peakamp = sfm.value[i];
			return 1;
		}
		else
			Application::inform("No peak amp information found.", true);
	}
	return 0;
}

int
SndOrIrcamSoundHeader::extractComment(const char *ptr) {
	SFCode *sfc;
	const char *codeptr;
	if((codeptr = getSFCode(ptr, SF_COMMENT)) != NULL) {
		SFComment sfcm;
		sfc = (SFCode *) codeptr;
		bcopy(codeptr + sizeof(SFCode) , (char *) &sfcm, 
			int(sfc->bsize) - sizeof(SFCode));
		setComment(sfcm.comment);
		return true;
	}
	return false;
}

int
SndOrIrcamSoundHeader::putSFCode(const char *hd, const char *ptr,
		const SndOrIrcamSoundHeader::SFCode *codeptr) {
	static const int codesize = sizeof(SFCode);
	static SFCode endcode = {
		SF_END,
		codesize
	};
	char *sfc = (char *) hd;	// casting away const...
	int wasendcode = false;
        const char *hdend = hd + diskHeaderCommentSize();
	while(sfc < hdend) {
		SFCode *sp = (SFCode *) sfc;
		if(sp->code == SF_END) {
			wasendcode = true;
			break;
		}
		/* Catch possible wrap around on stack from bad header */
		if(sp->bsize <= 0 || sfc + sp->bsize < hd) {
			sp->code = SF_END;	// Force an end
			sp->bsize = codesize;
			break;
		}
		if(sp->code == codeptr->code)
			break;
		sfc += sp->bsize;
	}
	
	/* No space left */
	if(sfc + codeptr->bsize > hdend)
		return false;
#if 0
	if(!wasendcode)		// if not first code added to this header
		if(codeptr->bsize != sp->bsize) /* Size changed */
			return false;
#endif

	// copy code struct into header starting at end of last code
	bcopy((char *) codeptr, sfc, codesize);
	// copy actual data struct into header right after that
	bcopy(ptr, sfc + codesize, codeptr->bsize - codesize);
	// stick an end-code struct on to mark the end
	if(wasendcode) 
		bcopy((char *) &endcode, sfc + codeptr->bsize, codesize);
	return true;
}

int
SndOrIrcamSoundHeader::loadPeak(const char *ptr) {
	static SFMaxamp sfm;
	static SFCode ampcode = {
		SF_MAXAMP,
		sizeof(SFMaxamp) + sizeof(SFCode)
	};
	for(int i = 0; i < MaxChannels; i++) {
		sfm.value[i] = peakamp;
		sfm.samploc[i] = 0;
	}
	struct timeval tp;
	struct timezone tzp;
	gettimeofday(&tp,&tzp);
	sfm.timetag = tp.tv_sec;
	return putSFCode(ptr, (char *) &sfm, &ampcode);
}

int
SndOrIrcamSoundHeader::loadComment(const char *ptr) {
	static SFComment sfcom;
	static SFCode comcode = {
		SF_COMMENT,
		sizeof(SFComment) + sizeof(SFCode)
	};
	const Comment *com = getComment();
	if(com != nil) {
		strcpy(sfcom.comment, *com);
		return putSFCode(ptr, (char *) &sfcom, &comcode);
	}
	else return(true);	// nil comment is not failure		
}

SndHeader::SndHeader(DataType format, int rate, int chans) 
		: SndOrIrcamSoundHeader(format, rate, chans, 0, SND_MAGIC) {
	header_type = Snd;
	data_offset = sizeof(SND_Header);
}

int
SndHeader::writeInfo(DataFile *file) {
	// if writing Snd style, set offset based on comment length (if any)
	int offset = (headerType() == Snd) ?
		diskHeaderInfoSize() + commentLength() : diskHeaderSize();
	setDataOffset(offset);
	SND_Header head(magic(), dataOffset(), dataSize(),
		type_to_format(dataType()), sampleRate(), nChans());
	// write out header minus the comment array
	file->write((void *) &head, SndHeader::diskHeaderInfoSize());
	return file->good();
}

int
SndHeader::writeComment(DataFile *file) {
	static char info[4];
	Comment* com = getComment();
	if(com != nil)
		file->write((void *) com->text(), commentLength());
	else 
		file->write((void *) info, diskHeaderCommentSize());
	return file->good();
}

int
SndHeader::commentLength() {
	return max(Header::commentLength(), diskHeaderCommentSize());
}

IrcamHeader::IrcamHeader(DataType format, int rate, int chans, double peak)
		: SndOrIrcamSoundHeader(format, rate, chans, peak, SF_MAGIC) {
	header_type = Ircam;
	data_offset = diskHeaderSize();
	peakamp = peak;
}

int
IrcamHeader::writeInfo(DataFile *file) {
	IRCAM_Header head(SF_MAGIC, float(samprate), nchans, sampleSize());
	if(!loadPeak(&head.sfinfo.sf_codes))
		Application::inform("Unable to load peak amp into header", true);
	if(!loadComment(&head.sfinfo.sf_codes))
		Application::inform("Unable to load comment into header", true);
	file->write((void *) &head, sizeof(IRCAM_Header));
	return file->good();
}

HybridHeader::HybridHeader(DataType format, int rate, int chans, double peak) 
		: SndHeader(format, rate, chans) {
	header_type = Hybrid;
	data_offset = diskHeaderSize();
	peakamp = peak;
}

int
HybridHeader::writeInfo(DataFile *file) {
	if(SndHeader::writeInfo(file)) {
		file->seek(sizeof(SND_Header)); // skip past local comment
		IRCAM_Header head(SF_MAGIC, float(samprate), nchans, sampleSize());
		if(!loadPeak(&head.sfinfo.sf_codes))
			Application::inform("Unable to load peak amp into header", true);
		if(!loadComment(&head.sfinfo.sf_codes))
			Application::inform("Unable to load comment into header", true);
		// write out header minus the length of the local portion
		file->write((void *) &head, sizeof(head) - sizeof(SND_Header));
	}
	return file->good();
}

int
HybridHeader::writeComment(DataFile *f) {
	return SndOrIrcamSoundHeader::writeComment(f);
}
