#include "stdio.h"
#ifndef mips
#include "stdlib.h"
#endif
#include "xlisp.h"
#include "sound.h"

#include "falloc.h"
#include "delay.h"

void delay_free();


typedef struct delay_susp_struct {
    snd_susp_node susp;
    long terminate_cnt;
    sound_type s;
    long s_cnt;
    sample_block_values_type s_ptr;

    double feedback;
    long delaylen;
    sample_type *delaybuf;
    sample_type *delayptr;
    sample_type *endptr;
} delay_susp_node, *delay_susp_type;


void delay_n_fetch(register delay_susp_type susp, snd_list_type snd_list)
{
    int cnt = 0; /* how many samples computed */
    int togo;
    int n;
    sample_block_type out;
    register sample_block_values_type out_ptr;

    register sample_block_values_type out_ptr_reg;

    register double feedback_reg;
    register sample_type * delayptr_reg;
    register sample_type * endptr_reg;
    register sample_block_values_type s_ptr_reg;
    falloc_sample_block(out, "delay_n_fetch");
    out_ptr = out->samples;
    snd_list->block = out;

    while (cnt < max_sample_block_len) { /* outer loop */
	/* first compute how many samples to generate in inner loop: */
	/* don't overflow the output sample block: */
	togo = max_sample_block_len - cnt;

	/* don't run past the s input sample block: */
	susp_check_term_samples(s, s_ptr, s_cnt);
	togo = min(togo, susp->s_cnt);

	/* don't run past terminate time */
	if (susp->terminate_cnt != UNKNOWN &&
	    susp->terminate_cnt <= susp->susp.current + cnt + togo) {
	    togo = susp->terminate_cnt - (susp->susp.current + cnt);
	    if (togo == 0) break;
	}

	n = togo;
	feedback_reg = susp->feedback;
	delayptr_reg = susp->delayptr;
	endptr_reg = susp->endptr;
	s_ptr_reg = susp->s_ptr;
	out_ptr_reg = out_ptr;
	if (n) do { /* the inner sample computation loop */
*out_ptr_reg++ = *delayptr_reg = *delayptr_reg * feedback_reg + *s_ptr_reg++;
             if (++delayptr_reg >= endptr_reg) delayptr_reg = susp->delaybuf;;
	} while (--n); /* inner loop */

	susp->delayptr = delayptr_reg;
	/* using s_ptr_reg is a bad idea on RS/6000: */
	susp->s_ptr += togo;
	out_ptr += togo;
	susp_took(s_cnt, togo);
	cnt += togo;
    } /* outer loop */

    /* test for termination */
    if (togo == 0 && cnt == 0) {
	snd_list_terminate(snd_list);
    } else {
	snd_list->block_len = cnt;
	susp->susp.current += cnt;
    }
} /* delay_n_fetch */


void delay_toss_fetch(susp, snd_list)
  register delay_susp_type susp;
  snd_list_type snd_list;
{
    long final_count = min(susp->susp.current + max_sample_block_len,
			   susp->susp.toss_cnt);
    time_type final_time = susp->susp.t0 + final_count / susp->susp.sr;
    long n;

    /* fetch samples from s up to final_time for this block of zeros */
    while (((long) ((final_time - susp->s->t0) * susp->s->sr + 0.5)) >=
	   susp->s->current)
	susp_get_samples(s, s_ptr, s_cnt);
    /* convert to normal processing when we hit final_count */
    /* we want each signal positioned at final_time */
    if (final_count == susp->susp.toss_cnt) {
	n = (final_time - susp->s->t0) * susp->s->sr -
	     (susp->s->current - susp->s_cnt) + 0.5;
	susp->s_ptr += n;
	susp_took(s_cnt, n);
	susp->susp.fetch = susp->susp.keep_fetch;
    }
    snd_list->block_len = final_count - susp->susp.current;
    susp->susp.current = final_count;
    snd_list->u.next = snd_list_create((snd_susp_type) susp);
    snd_list->block = internal_zero_block;
}


void delay_mark(delay_susp_type susp)
{
    sound_xlmark(susp->s);
}


void delay_free(delay_susp_type susp)
{
free(susp->delaybuf);    sound_unref(susp->s);
    ffree_generic(susp, sizeof(delay_susp_node), "delay_free");
}


void delay_print_tree(delay_susp_type susp, int n)
{
    indent(n);
    printf("s:");
    sound_print_tree_1(susp->s, n);
}


sound_type snd_make_delay(sound_type s, time_type delay, double feedback)
{
    register delay_susp_type susp;
    rate_type sr = s->sr;
    time_type t0 = 0.0;
    int interp_desc = 0;
    sample_type scale_factor = 1.0;
    time_type t0_min = t0;
    long togo;
    long lsc;

    /* combine scale factors of linear inputs (S) */
    scale_factor *= s->scale;
    s->scale = 1.0;

    /* try to push scale_factor back to a low sr input */
    if (s->sr < sr) { s->scale = scale_factor; scale_factor = 1.0; }

    falloc_generic(susp, delay_susp_node, "snd_make_delay");
    susp->feedback = feedback;
    susp->delaylen = max(1, s->sr * delay + 0.5);
    susp->delaybuf = (sample_type *) calloc (susp->delaylen, sizeof(sample_type));
    susp->delayptr = susp->delaybuf;
    susp->endptr = susp->delaybuf + susp->delaylen;
    susp->susp.fetch = delay_n_fetch;
    susp->terminate_cnt = UNKNOWN;
    /* handle unequal start times, if any */
    if (t0 < s->t0) sound_prepend_zeros(s, t0);
    /* minimum start time over all inputs: */
    t0_min = min(s->t0, t0);
    /* how many samples to toss before t0: */
    susp->susp.toss_cnt = (t0 - t0_min) * sr + 0.5;
    if (susp->susp.toss_cnt > 0) {
	susp->susp.keep_fetch = susp->susp.fetch;
	susp->susp.fetch = delay_toss_fetch;
	t0 = t0_min;
    }

    /* initialize susp state */
    susp->susp.free = delay_free;
    susp->susp.sr = sr;
    susp->susp.t0 = t0;
    susp->susp.mark = delay_mark;
    susp->susp.print_tree = delay_print_tree;
    susp->susp.name = "delay";
    susp->susp.log_stop_cnt = UNKNOWN;
    susp->susp.current = 0;
    susp->s = s;
    susp->s_cnt = 0;
    return sound_create((snd_susp_type)susp, t0, sr, scale_factor);
}


sound_type snd_delay(sound_type s, time_type delay, double feedback)
{
    sound_type s_copy = sound_copy(s);
    return snd_make_delay(s_copy, delay, feedback);
}
