/*************************************************************************
 *                                                                       *
 *               ROUTINES IN THIS FILE:                                  *
 *                                                                       *
 *                      get_data(): offline input routine, accepting     *
 *                              binary shorts, ascii, MAT or esps files; *
 *                              calls get_ascdata, get_bindata,          *
 *                              get_matdata or esps library routines     *
 *                                                                       *
 *                      get_ascdata(): offline input routine, reads      *
 *                              ascii and allocates as it goes           *
 *                                                                       *
 *                      get_bindata(): offline input routine, reads      *
 *                              binary shorts and allocates as it goes   *
 *                                                                       *
 *                      get_online_bindata(): online input routine,      *
 *                              reads binary shorts into rest of         *
 *                              analysis                                 *
 *                                                                       *
 *                      open_out(): opens non-esps output file for       *
 *                               writing                                 *
 *                                                                       *
 *                      write_out(): writes output file; calls esps      *
 *                               library routines or print_vec()         *
 *                               or bin_vec()                            *
 *                                                                       *
 *                      print_vec(): offline output routine, writes      *
 *                              ascii                                    *
 *                                                                       *
 *                      binout_vec(): offline output routine, writes     *
 *                              binary floats                            *
 *                                                                       *
 *                      fvec_HPfilter(): IIR highpass filter on waveform *
 *                                                                       *
 *                      load_history(): load history                     *
 *                                                                       *
 *                      save_history(): save history                     *
 *                                                                       *
 *                                                                       *
 ************************************************************************/


#include <stdio.h>
#include <math.h>
#include "rasta.h"
#include "functions.h"

#ifdef IO_ESPS
#include <esps/esps.h>
#include <esps/fea.h>
#include <esps/feasd.h>

/* pre-defined brain dead definition for ESPS libs */
int debug_level = 4;

static double Start_time = 0.0;
/* Global to this file only */
#endif

#ifdef IO_MAT
/* MATLAB */
#include <mat.h>
#endif

/* Here we read in ascii numbers, binary shorts, 
   or esps format files where
   blocks are allocated as necessary. */
struct fvec *get_data(struct param * pptr)
{
#ifdef IO_ESPS
  /* ESPS input vars */
  struct header *inhead;
  struct feasd *sdrec = (struct feasd *)NULL;
  struct fvec *get_espsdata( FILE *, struct fvec *, 
                            struct header * );
#endif

  /* locally called functions */
  struct fvec *get_ascdata( FILE *, struct fvec *);
  struct fvec *get_bindata( FILE *, struct fvec *, struct param *);

#ifdef IO_MAT
  /* MATLAB */
  struct fvec *get_matdata( MATFile *, struct fvec *);    
  MATFile *matfptr = (MATFile *)NULL;
#endif

  /* local variables */
  struct fvec *sptr;
  FILE *fptr = (FILE *)NULL;
  char *funcname;

  funcname = "get_data";

  /* Allocate space for structure */
  sptr = (struct fvec *) malloc(sizeof(struct fvec) );
  if(sptr == (struct fvec *)NULL)
    {
      fprintf(stderr,"Unable to allocate speech structure\n");
      exit(-1);
    }
        
  if(pptr->espsin == TRUE)
    {
#ifdef IO_ESPS
      eopen("rasta_plp", pptr->infname, "r", FT_FEA, FEA_SD,
            &inhead, &fptr );

      Start_time = (double) get_genhd_val( "start_time", inhead, 0.0);

      sptr->length = inhead->common.ndrec;
      if(sptr->length >= 1) 
        {
          sptr->values 
            = (float *)calloc( sptr->length, sizeof(float));
          sdrec = allo_feasd_recs( inhead, FLOAT, sptr->length, 
                                  (char *)sptr->values, NO);
        }
      else
        {
          /* Initial buffer allocation for float speech vector */
          sptr->values 
            = (float *) malloc(SPEECHBUFSIZE * sizeof(float));
          if(sptr->values == (float *)NULL)
            {
              fprintf(stderr,
                      "Can't allocate first speech buffer\n");
              exit(-1);
            }
        }
      if(sdrec == ( struct feasd *)NULL)
        {
          fprintf(stderr,
                  "Can't do ESPS input allocation\n");
          exit(-1);
        }
#endif
    }
  else if(pptr->matin == TRUE)
    {
#ifdef IO_MAT
      /* open MAT-file */
      matfptr = matOpen(pptr->infname, "r");
      if(matfptr == (MATFile *)NULL)
        {
          fprintf(stderr,"Error opening %s\n", pptr->infname);
          exit(-1);
        }
      /* Initial buffer allocation for float speech vector */
      sptr->values = (float *) malloc(SPEECHBUFSIZE * sizeof(float));
      if(sptr->values == (float *)NULL)
        {
          fprintf(stderr,"Can't allocate first speech buffer\n");
          exit(-1);
        }
#endif
    }
  else 
    {
      /* Initial buffer allocation for float speech vector */
      sptr->values = (float *) malloc(SPEECHBUFSIZE * sizeof(float));
      if(sptr->values == (float *)NULL)
        {
          fprintf(stderr,"Can't allocate first speech buffer\n");
          exit(-1);
        }
      if(strcmp(pptr->infname, "-") == 0)
        {
          fptr = stdin;
        }
      else
        {
          fptr = fopen(pptr->infname, "r");
          if (fptr == (FILE *)NULL)
            {
              fprintf(stderr,"Error opening %s\n", 
                      pptr->infname);
              exit(-1);
            }
        }
    }
  if(sptr->values == (float *)NULL)
    {
      fprintf(stderr,"Can't allocate first speech buffer\n");
      exit(-1);
    }


  /* here we get data */

  if(pptr->espsin == TRUE)
    {
#ifdef IO_ESPS
      if(sptr->length < 1 )
        {
          /* read and allocate a chunk at a time */
          sptr = get_espsdata( fptr, sptr, 
                              inhead );
        }
      else
        {
          /* If the length was stored, get it all. */
          (void) get_feasd_recs( sdrec, 0, sptr->length, 
                                inhead, fptr);
        }
      pptr->sampfreq = (int)get_genhd_val("record_freq", 
                                          inhead, (double)(pptr->sampfreq) );

      if(pptr->debug == TRUE)
        {
          fprintf(stderr,"Read sampfreq %d in esps\n",
                  pptr->sampfreq);
        }
#endif
    }
  else if(pptr->matin == TRUE)
    {
#ifdef IO_MAT
      sptr = get_matdata(matfptr, sptr);
#endif
    }
  else if(pptr->ascin == TRUE)
    {
      sptr = get_ascdata(fptr, sptr);
    }
  else
    {
      sptr = get_bindata(fptr, sptr, pptr);
    }

  return(sptr);
}

/* reads in ascii data into speech structure array, storing
   length information and allocating memory as necessary */
struct fvec *get_ascdata( FILE *fptr, struct fvec *sptr)
{
  int i, nread;
  char *funcname;
  int buflength = SPEECHBUFSIZE;

  funcname = "get_ascdata";

  sptr->values = (float *)realloc((char*)sptr->values, 
                                  buflength * sizeof(float) );
  if(sptr->values == (float *)NULL)
    {
      fprintf(stderr,
              "Unable to allocate %ld bytes for speech buffer\n",
              buflength * sizeof(float));
      exit(-1);
    }
  i = 0;
  while( (nread = fscanf(fptr, "%f", &(sptr->values[i])) )  == 1)
    {
      i++;                      /* On to the next sample */
      if(i >= buflength)
        {
          buflength += SPEECHBUFSIZE;
          sptr->values = (float *)realloc((char*)sptr->values, 
                                          buflength * sizeof(float) );
          if(sptr->values == (float *)NULL)
            {
              fprintf(stderr,"Can't allocate %ld byte buffer\n",
                      buflength * sizeof(float));
              exit(-1);
            }
        }
    }
  sptr->length = i;             /* Final value of index is the length */
  return ( sptr);
}

/* reads in binary short data into speech structure array, storing
   length information and allocating memory as necessary */
struct fvec *get_bindata(FILE *fptr, struct fvec *sptr, struct param *pptr)
{
  int i, nread, start;
  char *funcname;
  int totlength = 0;
  short buf[SPEECHBUFSIZE];
        
  funcname = "get_bindata";

  while( (nread = fread((char*)buf, sizeof(short), 
                        SPEECHBUFSIZE, fptr) ) == SPEECHBUFSIZE)
    {
      start = totlength;
      totlength += SPEECHBUFSIZE;
      sptr->values = (float *)realloc((char*)sptr->values, 
                                      totlength * sizeof(float) );
      if(sptr->values == (float *)NULL)
        {
          fprintf(stderr,"Can't allocate %ld byte buffer\n",
                  totlength * sizeof(float));
          exit(-1);
        }
      for(i=0; i<SPEECHBUFSIZE; i++)
        {
          if (pptr->swapbytes == TRUE)
            {
              sptr->values[start+i] =
                (float)(SWAP_BYTES(buf[i]));
            }
          else
            {
              sptr->values[start+i] = (float)buf[i];
            }
        }
    }
  /* now read in last bunch that are less than one buffer full */
  start = totlength;
  totlength += nread;
  sptr->values = (float *)realloc((char*)sptr->values, 
                                  totlength * sizeof(float) );
  if(sptr->values == (float *)NULL)
    {
      fprintf(stderr,"Unable to realloc %ld bytes for speech buffer\n",
              totlength * sizeof(float));
      exit(-1);
    }
  for(i=0; i<nread; i++)
    {
      if (pptr->swapbytes == TRUE)
        {
          sptr->values[start+i] = (float)(SWAP_BYTES(buf[i]));
        }
      else
        {
          sptr->values[start+i] = (float)buf[i];
        }
    }
  sptr->length = totlength;

  return (sptr );
}

#ifdef IO_MAT
/* reads in MATLAB data into speech structure array, storing
   length information and allocating memory as necessary */
struct fvec *get_matdata( MATFile *matfptr, struct fvec *sptr)
{
  Matrix *matm;
  double *matr;
  int totlength, i;
  char *funcname;
        
  funcname = "get_matdata";

  /* load MATLAB-matrix (one row or one column: vector) */
  matm = matGetNextMatrix(matfptr);
  if(mxGetM(matm) == 1)
    totlength = mxGetN(matm);
  else if(mxGetN(matm) == 1)
    totlength = mxGetM(matm);
  else
    {
      fprintf(stderr,"MAT-file input error: more than one row or column\n");
      exit(-1);
    }
  /* allocation for float speech vector */
  sptr->values = (float *)realloc((char*)sptr->values, 
                                  totlength * sizeof(float) );
  if(sptr->values == (float *)NULL)
    {
      fprintf(stderr,"Unable to realloc %ld bytes for speech buffer\n",
              totlength * sizeof(float));
      exit(-1);
    }
  /* get speech data from MATLAB-matrix */
  matr = mxGetPr(matm);
  for(i = 0; i<totlength; i++)
    sptr->values[i] = (float)matr[i];
  sptr->length = totlength;

  return (sptr );
}
#endif

/* reads in binary short data into speech structure array, 
   assuming data coming from stdin and only reading in
   enough to provide a full frame's worth of data for
        the analysis. This means a full frame's worth for
        the first frame, and one analysis step's worth
        on proceeding frames. 

        This routine is very similar to fill_frame(), which can be
        found in anal.c; it is here because it does data i/o
        (OK, just "i" ) 

        Speech samples are highpass filtered on waveform if
        option -F is used */
struct fvec *get_online_bindata( struct fhistory *hptr, struct param *pptr )
{
        static struct fvec *outbufptr = NULL;
        static struct fvec *inbufptr;
        static struct svec *sbufptr;
        static struct fvec *window;
        int i, overlap, nread;
        char *funcname;
        
        funcname = "get_online_bindata";
        
        if(outbufptr == (struct fvec *)NULL) /* If first frame in analysis */
        {
                outbufptr = alloc_fvec( pptr->winpts);
                inbufptr = alloc_fvec( pptr->winpts); 
                sbufptr = alloc_svec( pptr->winpts);
                
                window = get_win(pptr, outbufptr->length);
                
                nread = fread((char*)sbufptr->values, sizeof(short), 
                                          pptr->winpts, stdin);
                if(nread != pptr->winpts)
                {
                        fprintf(stderr,"Done with online input\n");
                        save_history( hptr, pptr );
                        exit(0);
                }
                for(i=0; i<pptr->winpts; i++)
                        inbufptr->values[i] = (float)sbufptr->values[i];
        }
        else
        {
                nread = fread(sbufptr->values, sizeof(short),
                                          pptr->steppts, stdin);
                if(nread != pptr->steppts)
                {
                        fprintf(stderr,"Done with online input\n");
                        save_history( hptr, pptr );
                        exit(0);
                }
                
                /* Shift down input values */
                for(i=pptr->steppts; i<pptr->winpts; i++)
                        inbufptr->values[i-pptr->steppts] = inbufptr->values[i];

                /* new values */
                overlap = pptr->winpts - pptr->steppts;
                for(i=overlap; i<pptr->winpts; i++)
                        inbufptr->values[i] = (float)sbufptr->values[i-overlap];
        }

        if(pptr->HPfilter == TRUE)
                fvec_HPfilter(pptr, inbufptr);

        for(i=0; i<outbufptr->length; i++)
                if (pptr->swapbytes == TRUE)
                {
                       outbufptr->values[i] = window->values[i] *
                         (float)SWAP_BYTES(((short)(inbufptr->values[i])));
                }
                else
                {
                       outbufptr->values[i] =
                         window->values[i] * (float)inbufptr->values[i];
                }

        return (outbufptr);
}

#ifdef IO_ESPS
/* reads in esps data into speech structure array, storing
        length information and allocating memory as necessary */
struct fvec *get_espsdata( FILE *fptr, struct fvec *sptr, 
        struct header *ihd )
{
        int i, nread;
        char *funcname;
        struct feasd *sdrec;
        float ftmp[SPEECHBUFSIZE]; 
        int start = 0;
        
        funcname = "get_espsdata";

        sptr->length = SPEECHBUFSIZE;
        sdrec = allo_feasd_recs( ihd, FLOAT, SPEECHBUFSIZE,
                (char *)ftmp, NO);

        /* length starts as what we have allocated space for */
        while( (nread = get_feasd_recs(sdrec, 0L, SPEECHBUFSIZE, 
                        ihd, fptr)) == SPEECHBUFSIZE)
        {

                for(i=0; i<SPEECHBUFSIZE; i++)
                {
                        sptr->values[start+i] = ftmp[i];
                }
                /* increment to next size we will need */
                sptr->length += SPEECHBUFSIZE;
                start += SPEECHBUFSIZE;

                sptr->values = (float *)realloc((char*)sptr->values, 
                          (unsigned)(sptr->length * sizeof(float)) );
                if(sptr->values == (float *)NULL)
                {
                        fprintf(stderr,"Can't allocate %ld byte buffer\n",
                                sptr->length * sizeof(float));
                        exit(-1);
                      }
                
              }
        
        /* now adjust length for last bunch that is less than one buffer full */
        
        sptr->length -= SPEECHBUFSIZE;
        sptr->length += nread;
        for(i=0; i<nread; i++)
          {
            sptr->values[start+i] = ftmp[i];
          }
        
        return (sptr );
      }
#endif

/* Opens output file unless writing ESPS file; in that case,
   file is opened when write_out is first called */
FILE *open_out(struct param *pptr)
{
  FILE *fptr;

  if(((pptr->espsout) == TRUE) || ((pptr->matout) == TRUE))
    {
      fptr = (FILE *)NULL;
    }
  else if(strcmp(pptr->outfname, "-") == 0)
    {
      fptr = stdout;
    }
  else
    {
      fptr = fopen(pptr->outfname, "w");
      if (fptr == (FILE *)NULL)
        {
          fprintf(stderr,"Error opening %s\n", pptr->outfname);
          exit(-1);
        }
    }
  return (fptr);
}

/* General calling routine to write out a vector */
void write_out( struct param *pptr, FILE *outfp, struct fvec *outvec )
{
#ifdef IO_ESPS
  /* ESPS variables */
  static struct header *outhead = NULL;
  static struct fea_data *outrec;
  static float *opdata;
  static double rec_freq;

  /* Ignore outfp if an ESPS file; file is opened an
     written to within this routine only. */
  static FILE *esps_fp;
#endif

#ifdef IO_MAT
  /* MATLAB variables */
  static MATFile *matfptr;
  static double *matr;
  static double *matr_start = (double *)NULL;
  static double *matr_end;
#endif

  int i;
  char *funcname;
  funcname = "write_out";

  if((pptr->espsout)   == TRUE)
    {
#ifdef IO_ESPS
      if(outhead == (struct header *)NULL)
        {
          rec_freq = 1.0 / (pptr->stepsize * .001);
          eopen("rasta_plp", pptr->outfname, "w", NONE, NONE,
                (struct header **)NULL, &esps_fp);              
          outhead = new_header(FT_FEA);
          add_fea_fld( "rasta_plp", (long)outvec->length, 
                      1, NULL, FLOAT, NULL, outhead);
          *add_genhd_d("start_time", NULL,1,outhead) = Start_time;
          *add_genhd_d("record_freq", NULL,1,outhead) = rec_freq;
          write_header( outhead, esps_fp);
          outrec = allo_fea_rec( outhead );
          opdata = (float *)get_fea_ptr( outrec, "rasta_plp", 
                                        outhead);
                          
          /* ESPS seems to handle its own error
             checking (pretty much) */
        }
      for(i=0; i<outvec->length; i++)
        {
          opdata[i] = outvec->values[i];
        }
      put_fea_rec(outrec, outhead, esps_fp);
#endif
    }
  else if((pptr->matout) == TRUE )
    {
#ifdef IO_MAT
      if(matr_start == (double *)NULL)
        {
          matr = (double *) malloc((pptr->nframes * outvec->length) * sizeof(double));
          if(matr == (double *)NULL)
            {
              fprintf(stderr,"Can't allocate output buffer\n");
              exit(-1);
            }
          matr_start = matr;
          matr_end   = matr + (pptr->nframes * outvec->length);
        }
      for(i=0; i<outvec->length; i++, matr++)
        {
          *matr = (double)outvec->values[i];
        }       
      if( matr == matr_end )
        {
          /* open MAT-file */
          matfptr = matOpen(pptr->outfname,"w");
          if(matfptr == (MATFile *)NULL)
            {
              fprintf(stderr,"Error opening %s\n", pptr->outfname);
              exit(-1);
            }
          /* save output as MAT-file */
          if(matPutFull(matfptr, "rasta_out", outvec->length, pptr->nframes, matr_start, NULL) != 0)
            {
              fprintf(stderr, "Error saving %s\n", pptr->outfname);
              exit(-1);
            }                               
          matClose(matfptr);
          mxFree;
        }
#endif
    }
  else if((pptr->ascout) == TRUE || (pptr->crbout) == TRUE ||
          (pptr->comcrbout) == TRUE)
    {
      print_vec(pptr, outfp, outvec, pptr->nout);
    }
  else
    {
      binout_vec( outfp, outvec );
    }
}

/* Print out ascii for float vector with specified width (n columns) */
void print_vec(const struct param *pptr, FILE *fp, struct fvec *fptr, int width)
{
  int i;
  char *funcname;
  int lastfilt;

  funcname = "print_vec";
  if ((pptr->crbout == FALSE) && (pptr->comcrbout == FALSE))
    {
        
      for(i=0; i<fptr->length; i++)
        {
          fprintf(fp, "%g ", fptr->values[i]);
          if((i+1)%width == 0)
            {
              fprintf(fp, "\n");
            }
        }
    }
  else
    {
      lastfilt = pptr->nfilts - pptr->first_good;
      for (i= pptr->first_good; i<lastfilt; i++)
        {
          fprintf(fp, "%g ", fptr->values[i]);
        }
      fprintf(fp, "\n");
    }

}

/* Send float vector values to output stream */
void binout_vec( FILE *fp, struct fvec *fptr )
{
  char *funcname;

  funcname = "binout_vec";

  fwrite((char *)fptr->values, sizeof(float), fptr->length, fp);
}

/* digital IIR highpass filter on waveform to remove DC offset
   H(z) = (0.993076-0.993076*pow(z,-1))/(1-0.986152*pow(z,-1)) 
   offline and online version, inplace calculating */
void fvec_HPfilter(struct param *pptr, struct fvec *fptr)
{
  static double coefB,coefA;
  double d,c,p2;
  static int first_call = 1;
  static float old;
  static int start;
  float tmp;
  int i;
        

  if(first_call)
    {
      /* computing filter coefficients */
      d = pow(10.,-3./10.);     /* 3dB passband attenuation at frequency f_p */
      c = cos(2.*M_PI* 44.7598 /(double)pptr->sampfreq); /* frequency f_p about 45 Hz */
      p2 = (1.-c+2.*d*c)/(1.-c-2.*d);
      coefA = floor((-p2-sqrt(p2*p2-1.))*1000000.) / 1000000.; /* to conform to */
      coefB = floor((1.+coefA)*500000.) / 1000000.; /* older version */

      start = 1;
      old = fptr->values[0];
    }
        
  for(i=start; i<fptr->length; i++)
    {
      tmp = fptr->values[i];
      fptr->values[i] =  coefB * (fptr->values[i] - old) + coefA * fptr->values[i-1];
      old = tmp;
    }

  if(first_call)
    {
      start = fptr->length - pptr->steppts;
      if(start < 1)
        {
          fprintf(stderr,"step size >= window size -> online highpass filter not available");
          exit(-1);
        }
      first_call = 0;
    }
}


/* load history from file */
void load_history(struct fhistory *hptr, const struct param *pptr)
{
  int head[3];
  int i;
  FILE *fptr;
        

  /* open file */
  fptr = fopen(pptr->hist_fname,"r");
  if(fptr == (FILE *)NULL)
    {
      fprintf(stderr,"Warning: Cannot open %s, using normal initialization\n", pptr->hist_fname);
      return;
    }

  /* read header */
  if(fread(head,sizeof(int),3,fptr) != 3)
    {
      fprintf(stderr,"Warning: Cannot read history file header, using normal initialization\n");
      return;
    }
  if((head[0] != pptr->nfilts) || (head[1] != FIR_COEF_NUM) || (head[2] != IIR_COEF_NUM))
    {
      fprintf(stderr,"Warning: History file is incompatible (header differs), using normal initialization\n");
      return;
    }

  /* read stored noise estimation level */
  if(fread(hptr->noiseOLD, sizeof(float), head[0], fptr) != head[0])
    {
      fprintf(stderr,"Warning: History file failure, using normal initialization\n");
      return;
    }

  /* read stored RASTA filter input buffer */
  for(i=0; i<head[0]; i++)
    {
      if(fread(hptr->filtIN[i], sizeof(float), head[1], fptr) != head[1])
        {
          fprintf(stderr,"Warning: History file failure, using normal initialization\n");
          return;
        }
    }
        
  /* read stored RASTA filter input buffer */
  for(i=0; i<head[0]; i++)
    {
      if(fread(hptr->filtOUT[i], sizeof(float), head[2], fptr) != head[2])
        {
          fprintf(stderr,"Warning: History file failure, using normal initialization\n");
          return;
        }
    }
        
  /* use history values instead of normal initialization */
  hptr->normInit = FALSE;
        
  fclose(fptr);
}


/* save history to file */
void save_history(struct fhistory *hptr, const struct param *pptr)
{
  int head[3];
  int i;
  FILE *fptr;
        

  /* open file */
  fptr = fopen(pptr->hist_fname,"w");
  if(fptr == (FILE *)NULL)
    {
      fprintf(stderr,"Error opening %s\n", pptr->outfname);
      exit(-1);
    }

  /* write header */
  head[0] = pptr->nfilts;
  head[1] = FIR_COEF_NUM;
  head[2] = IIR_COEF_NUM;
  fwrite(head, sizeof(int), 3, fptr);
        
  /* write noise level estimation */
  fwrite(hptr->noiseOLD, sizeof(float), head[0], fptr);

  /* write RASTA filter input buffer */
  for(i=0; i<head[0]; i++)
    fwrite(hptr->filtIN[i], sizeof(float), head[1], fptr);

  /* write RASTA filter output buffer */
  for(i=0; i<head[0]; i++)
    fwrite(hptr->filtOUT[i], sizeof(float), head[2], fptr);

  fclose(fptr);
}
