
/*
 * FILE: makefilter.c
 *   BY: Christopher Lee Fraley (cf0v@spice.cmu.edu or cf0v@andrew.cmu.edu)
 * DESC: Makes a Kaiser-windowed low-pass filter.
 * DATE: 7-JUN-88
 */

char makefilterVERSION[] = "2.0  (17-JUN-88  3:00pm)";

#include <stdio.h>
#include <math.h>
#include "stdefs.h"
#include "resample.h"
#include "filterkit.h"

/* LIBRARIES needed:
 *
 * 1. filterkit
 *       makeFilter()  - designs a Kaiser-windowed low-pass filter
 *       writeFilter() - writes a filter to a standard filter file
 *       GetUShort()   - prompt user for a UHWORD with help
 *       GetDouble()   - prompt user for a double with help
 *
 * 2. math
 */



char NmultHelp[] =
  "\n   Nmult is the length of the symmetric FIR lowpass filter used\
   \n   by the sampling rate converter. It must be odd.\
   \n   This is the number of multiplies per output sample for\
   \n   up-conversions (Factor>1), and is the number of multiplies\
   \n   per input sample for down-conversions (Factor<1). Thus if\
   \n   the rate conversion is Srate2 = Factor*Srate1, then you have\
   \n   Nmult*Srate1*MAXof(Factor,1) multiplies per second of real time.\
   \n   Naturally, higher Nmult gives better lowpass-filtering at the\
   \n   expense of longer compute times. Nmult should be odd because\
   \n   it is the length of a symmetric FIR filter, and the current\
   \n   implementation requires a coefficient at the time origin.\n";      

char FrollHelp[] =
  "\n   Froll determines the frequency at which the lowpass filter begins to\
   \n   roll-off. If Froll=1, then there is no 'guard zone' and the filter\
   \n   roll-off region will be aliased. If Froll is 0.85, for example, then\
   \n   the filter begins rolling off at 0.85*Srate/2, so that by Srate/2,\
   \n   the filter is well down and aliasing is reduced.  Since aliasing\
   \n   distortion is worse by far than loss of the high-frequency spectral\
   \n   amplitude, Froll<1 is highly recommended. The default of 0.85\
   \n   sacrifices the upper 15% of the spectrum as an anti-aliasing guard\
   \n   zone.\n";

char BetaHelp[] =
  "\n   Beta trades the rejection of the lowpass filter against the\
   \n   transition width from passband to stopband. Larger Beta means\
   \n   a slower transition and greater stopband rejection. See Rabiner\
   \n   and Gold (Th. and App. of DSP) under Kaiser windows for more about\
   \n   Beta. The following table from Rabiner and Gold gives some feel\
   \n   for the effect of Beta:\
   \n\
   \nAll ripples in dB, width of transition band =D*N, where N= window length\
   \n\
   \n               BETA    D       PB RIP   SB RIP\
   \n               2.120   1.50  +-0.27      -30\
   \n               3.384   2.23    0.0864    -40\
   \n               4.538   2.93    0.0274    -50\
   \n               5.658   3.62    0.00868   -60\
   \n               6.764   4.32    0.00275   -70\
   \n               7.865   5.0     0.000868  -80\
   \n               8.960   5.7     0.000275  -90\
   \n               10.056  6.4     0.000087  -100\n";



main()
{
   HWORD Imp[MAXNWING];               /* Filter coefficients */
   HWORD ImpD[MAXNWING];              /* ImpD[i] = ImpD[i+1] - ImpD[i] */
   double Froll, Beta;
   UHWORD Nmult, Nwing, LpScl;
   int err;

   printf("\nKaiser-windowed FIR Filter Design\n");
   printf("Written by:  Chritopher Lee Fraley\n");
   printf("Version %s\n", makefilterVERSION);

   Nmult = 13;
   Froll = 0.425;
   Beta  = 5.7;
   while (TRUE)
      {
      Nmult = GetUHWORD("(Odd) Filter length 'Nmult'", Nmult, NmultHelp);
      Nwing = Npc*(Nmult+1)/2;     /* # of filter coeffs in right wing */
      Nwing += Npc/2 + 1;          /* This prevents just missing last coeff */
                                   /*   for integer conversion factors  */
      Froll = GetDouble("Normalized Roll-off freq (0<Froll<=1)",
         Froll, FrollHelp);
      Beta = GetDouble("Beta", Beta, BetaHelp);
      printf("\n");
      if (!(Nmult % 2))
         printf("Error: Nmult must be odd and greater than zero\n");
      else if (Nwing > MAXNWING)
         printf("Error: Nmult too large for current MAXNWING\n");
      else if ((Froll<=0) || (Froll>1))
         printf("Error: Roll-off freq must be 0<Froll<=1\n");
      else if (Beta < 1)
         printf("Error: Beta must be greater or equal to 1\n");
      else if (err = makeFilter(Imp, ImpD, &LpScl, Nwing, Froll, Beta))
         printf("Error: Unable to make filter, err=%d\n", err);
      else if (err = writeFilter(Imp, ImpD, LpScl, Nmult, Nwing))
         printf("Error: Unable to write filter, err=%d\n", err);
      else
         break;
      }
}

