
/*
 * FILE: warp.c
 *   BY: Christopher Lee Fraley (cf0v@spice.cs.cmu.edu)
 * NOTE: Based upon SAIL program by Julius O. Smith (CCRMA, Stanford U)
 * DATE: 17-JUN-88
 */

char warpVERSION[] = "1.0  24-JUN-88, 4:20pm";

/*
 *      The original SAIL program on which this is based may be found in
 *      /../s/usr/mhs/resample  or  /../x/usr/rbd/src/resample
 *
 *      Implement dynamic sampling-rate changes; uses a second file to
 *      indicate where periods should fall.  This program may be used to add
 *      or remove vibrato and micro pitch variations from a sound sample.
 *      Based on SRCONV.SAI[SYS,MUS] with new algorithm by JOS&PG.
 *      The program internally uses 16-bit data and 16-bit filter
 *      coefficients.
 *
 *      Reference: "A Flexible Sampling-Rate Conversion Method,"
 *      J. O. Smith and P. Gossett, ICASSP, San Diego, 1984, Pgs 19.4.
 *
 *      * Need overflow detection in Filter() routines.  Also, we want
 *        to saturate instead of wrap-around and report number of clipped
 *        samples.
 */

/* CHANGES from original SAIL program:
 *
 * 1. SrcUD() - Switched order of making guard bits (v>>Nhg) and
 *       normalizing.  This was done to prevent overflow.
 */

/* LIBRARIES needed:
 *
 * 1. filterkit
 *       readFilter() - reads standard filter file.
 *       FilterUD()   - applies filter to sample for any Factor.
 *       GetDouble()  - prompt user for a double with help.
 *
 * 2. math
 */




#include <stdio.h>
#include <math.h>
#include <string.h>
#include "stdefs.h"
#include "filterkit.h"
#include "resample.h"

#define IBUFFSIZE 1024                         /* Input buffer size */
#define OBUFFSIZE (IBUFFSIZE*MAXFACTOR+2)      /* Calc'd out buffer size */


fail(s)
char *s;
{
   printf("warp: %s\n\n", s);     /* Display error message  */
   exit(1);                       /* Exit, indicating error */
}

fails(s,s2)
char *s, *s2;
{
   printf("warp: ");              /* Display error message  */
   printf(s, s2);
   printf("\n\n");
   exit(1);                       /* Exit, indicating error */
}



/* Global file pointers: */
FILE *fin, *fout;


int openData(argc, argv)
int argc;
char *argv[];
{
   if (argc != 3)
      fail("format is 'warp <file-in> <file-out>'");
   if (NULL == (fin = fopen(*++argv,"r")))
      fails("could not open <file-in> file '%s'",*argv);
   if (NULL == (fout = fopen(*++argv,"w")))
      fails("could not open <file-out> file '%s'",*argv);
}


closeData()
{
   (void) fclose(fin);
   (void) fclose(fout);
}



int readData(Data, DataArraySize, Xoff)  /* return: 0 - notDone */
HWORD Data[];                            /*        >0 - index of last sample */
int DataArraySize, Xoff;
{
   int Nsamps, Scan, val;
   HWORD *DataStart;

   DataStart = Data;
   Nsamps = DataArraySize - Xoff;   /* Calculate number of samples to get */
   Data += Xoff;                    /* Start at designated sample number */
   for (; Nsamps>0; Nsamps--)
      {
      Scan = fscanf(fin, "%d", &val);      /* Read in Nsamps samples */
      if (Scan==EOF || Scan==0)            /*   unless read error or EOF */
         break;                            /*   in which case we exit and */
      *Data++ = val;
      }
   if (Nsamps > 0)
      {
      val = Data - DataStart;              /* (Calc return value) */
      while (--Nsamps >= 0)                /*   fill unread spaces with 0's */
         *Data++ = 0;                      /*   and return FALSE */
      return(val);
      }
   return(0);
}



writeData(Nout, Data)
int Nout;
HWORD Data[];
{
   while (--Nout >= 0)                  /* Write Nout samples to ascii file */
      fprintf(fout, "%d\n", *Data++);
}




check()
{
   /* Check for illegal constants */
   if (Np >= 16)
      fail("Error: Np>=16");
   if (Nb+Nhg+NLpScl >= 32)
      fail("Error: Nb+Nhg+NLpScl>=32");
   if (Nh+Nb > 32)
      fail("Error: Nh+Nb>32");
}


/* Globals for warping frequency */
double a,b,c,d,e,f,Total;
int Acc;

initWarp()
{
   Total = GetDouble("\n# of input samples",12000.0,"");

   printf("Warping function:  Fout/Fin = w(n)\n");
   printf("  w(n) = off + [amp * sin(ph1+frq1*n/N) * sin(ph2+frq2*n/N)]\n");
   printf("  where: off,amp,ph1   - parameters\n");
   printf("         frq1,ph2,frq2 - more parameters\n");
   printf("         n             - current input sample number\n");
   printf("         N             - total number of input samples\n");

   a = GetDouble("off",1.5,"");
   b = GetDouble("amp",-0.5,"");
   c = D2R * GetDouble("ph1 (degrees)",-90.0,"");
   d = GetDouble("frq1",1.0,"");
   e = D2R * GetDouble("ph2 (degrees)",90.0,"");
   f = GetDouble("frq2",0.0,"");
}


double warpFunction(Time)
UWORD Time;
{
   double fTime,t;

   /* Calc absolute position in input file: */
   fTime = (double)Time / (double)(1<<Np) + (double)Acc;
   /* Calc fraction of file processed: */
   t = fTime/Total;
   /* Return warp factor: */
   return (1.0 / (a + b*sin(c+PI2*d*t)*sin(e+PI2*f*t)));
}


/* Sampling rate conversion subroutine */

SrcUD(X, Y, Time, Nx, Nwing, LpScl, Imp, ImpD, Interp)
HWORD X[], Y[], Imp[], ImpD[];
UHWORD Nx, Nwing, LpScl;
UWORD *Time;
BOOL Interp;
{
   HWORD *Xp, *Ystart;
   WORD v;

   UHWORD NewScl;              /* Unity gain scale factor */
   double dh;                  /* Step through filter impulse response */
   double dt;                  /* Step through input signal */
   UWORD endTime;              /* When Time reaches EndTime, return to user */
   UWORD dhb, dtb;             /* Fixed-point versions of Dh,Dt */
   double Factor;              /* Current resampling factor */

   Ystart = Y;
   endTime = *Time + (1<<Np)*(WORD)Nx;
   while (*Time < endTime)
      {
      Factor = warpFunction(*Time);     /* Get new conversion Factor */
      NewScl = LpScl * Factor;          /* Calc new normalizing scalar */
      dt = 1.0 / Factor;                /* New output sampling period */
      dtb = dt*(1<<Np) + 0.5;           /* Fixed-point representation */
      dh = MIN(Npc, Factor * Npc);      /* New filter sampling period */
      dhb = dh*(1<<Na) + 0.5;           /* Fixed-point representation */
      Xp = &X[*Time>>Np];               /* Ptr to current input sample */
      v = FilterUD(Imp, ImpD, Nwing, Interp, Xp, (HWORD)(*Time&Pmask),
          -1, dhb);                     /* Perform left-wing inner product */
      v += FilterUD(Imp, ImpD, Nwing, Interp, Xp+1, (HWORD)((-*Time)&Pmask),
           1, dhb);                     /* Perform right-wing inner product */
      v >>= Nhg;                        /* Make guard bits */
      v *= NewScl;                      /* Normalize for unity filter gain */
      *Y++ = v>>NLpScl;                 /* Deposit output */
      *Time += dtb;                     /* Move to next sample by time inc */
      }
   return (Y - Ystart);                 /* Return the # of output samples */
}



int resample(Imp, ImpD, LpScl, Nmult, Nwing, InterpFilt)
HWORD Imp[], ImpD[];
UHWORD LpScl, Nmult, Nwing;
BOOL InterpFilt;
{
   UWORD Time;                       /* Current time/pos in input sample */
   UHWORD Xp, Xread, Ncreep, Xoff;
   HWORD X[IBUFFSIZE], Y[OBUFFSIZE]; /* I/O buffers */
   UHWORD Nout, Nx;
   int i, Ycount, last;

   /* Calc reach of LP filter wing & give some creeping room */
   Xoff = ((Nmult+1)/2.0) * MAX(1.0,1.0*MAXFACTOR) + 10;
   if (IBUFFSIZE < 2*Xoff)      /* Check input buffer size */
      fail("IBUFFSIZE (or Factor) is too small");
   Nx = IBUFFSIZE - 2*Xoff;     /* # of samples to proccess each iteration */

   last = FALSE;                /* Have we read last input sample yet? */
   Ycount = 0;                  /* Output sample # and length of out file */
   Xp = Xoff;                   /* Current position in input buffer */
   Xread = Xoff;                /* Position in input array to read into */
   Time = (Xoff<<Np);           /* Current-time pointer for converter */
   Acc = -Xoff;                 /* Acc + int(Time) = index into input file */

   for (i=0; i<Xoff; X[i++]=0); /* Need Xoff zeros at begining of sample */

   do {
      if (!last)                /* If haven't read last sample yet */
         {
         last = readData(X, IBUFFSIZE, (int)Xread); /* Fill input buffer */
         if (last && (last+Xoff<Nx))  /* If last sample has been read... */
            Nx = last + Xoff;   /* ...calc last sample affected by filter */
	 }
      Ycount += Nout = SrcUD(X,Y,&Time,Nx,Nwing,LpScl,Imp,ImpD,InterpFilt);
      Time -= (Nx<<Np);           /* Move converter Nx samples back in time */
      Xp += Nx;                   /* Advance by number of samples processed */
      Acc += Nx;                  /* We moved Nx samples in the input file */
      Ncreep = (Time>>Np) - Xoff; /* Calc time accumulation in Time */
      if (Ncreep)
         {
         Time -= (Ncreep<<Np);    /* Remove time accumulation */
         Xp += Ncreep;            /* and add it to read pointer */
         Acc += Ncreep;
         }
      for (i=0; i<IBUFFSIZE-Xp+Xoff; i++)   /* Copy portion of input signal */
         X[i] = X[i+Xp-Xoff];               /*    that must be re-used */
      if (last)                             /* If near end of sample... */
         {
         last -= Xp;             /* ...keep track were it ends */
         if (!last)              /* Lengthen input by 1 sample */
            last++;              /*  if needed to keep flag TRUE */
	 }
      Xread = i;                 /* Pos in input buff to read new data into */
      Xp = Xoff;

      if (Nout > OBUFFSIZE)      /* Check to see if output buff overflowed */
         fail("Output array overflow");

      writeData((int)Nout, Y);   /* Write data in output buff to file */
      } while (last >= 0);       /* Continue until done processing samples */
   return(Ycount);               /* Return # of samples in output file */
}




main(argc,argv)
int argc;
char *argv[];
{
   BOOL InterpFilt;             /* TRUE means interpolate filter coeffs */
   UHWORD LpScl, Nmult, Nwing;
   HWORD Imp[MAXNWING];         /* Filter coefficients */
   HWORD ImpD[MAXNWING];        /* ImpD[n] = Imp[n+1]-Imp[n] */
   int outCount;

   Nmult = 13;
   InterpFilt = TRUE;
   printf("\nDynamic Rate Resampling Program (for interesting effects)\n");
   printf("Written by:  Christopher Lee Fraley (cf0v@spice.cs.cmu.edu)\n");
   printf("Based upon SAIL program by Julius O. Smith (CCRMA)\n");
   printf("Version %s\n", warpVERSION);
   check();                      /* Check constants for validity */
   openData(argc, argv);         /* Interp arguments and open i&o files */
   initWarp();                   /* Set up the warp function's parameters */
   if (readFilter(Imp, ImpD, &LpScl, Nmult, &Nwing))
      fail("could not open Filter file, or syntax error in filter file");
   printf("\nWarp Speed Full Ahead...\n");
   outCount = resample(Imp, ImpD, LpScl, Nmult, Nwing, InterpFilt);
   closeData();

   printf("...Returning To Ion Drive:  %d output samples.\n\n", outCount);
}

