/*********************************************************************

This software module was originally developed by

Eric D. Scheirer (MIT Media Laboratory)

in the course of development of the MPEG-2 NBC/MPEG-4 Audio standard
ISO/IEC 13818-7, 14496-1,2 and 3. This software module is an
implementation of a part of one or more MPEG-2 NBC/MPEG-4 Audio tools
as specified by the MPEG-2 NBC/MPEG-4 Audio standard.  ISO/IEC gives
users of the MPEG-2 NBC/MPEG-4 Audio standards free license to this
software module or modifications thereof for use in hardware or
software products claiming conformance to the MPEG-2 NBC/ MPEG-4 Audio
standards. Those intending to use this software module in hardware or
software products are advised that this use may infringe existing
patents. The original developer of this software module and his/her
company, the subsequent editors and their companies, and ISO/IEC have
no liability for use of this software module or modifications thereof
in an implementation.

This software module is hereby released into the public domain.

***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saol.h"
#include "y.tab.h"
#include "saol_interp.h"
#include "saol_sched.h"

double eval_block(context *cx, block *b, long rate) {
  /* this is where the action is! */
  int ind;
  long checkrate;
  double val;
  
  for (;b;b=b->next) {
    checkrate = b->st->rate == XSIG ? cx->xsigrate : b->st->rate;
		
    switch ((int)b->st->type) {
			
			
    case EQ:
      /* only need to do this at the rate of the exprression */
      if (checkrate >= rate) {
	if (b->st->lvalue->op == ARRAYREF)
	  ind = (int)(eval_expr(cx,b->st->lvalue->left,rate) +0.5);
	else
	  ind = 0;
	val = eval_expr(cx,b->st->expr,rate);
	if (checkrate == rate ||
	    (checkrate == SPECIALOP && rate == KSIG))
	  set_var_value(cx,b->st->lvalue,ind,val);
      }
				
      break;
			
    case IF:
      if (checkrate >= rate) {
				
	val = eval_expr(cx,b->st->expr,rate);
				
	if (val)
	  if (val = eval_block(cx,b->st->b,rate))
	    return(val);	/* nonzero value means block called 'return' */
      }
      /*if (checkrate == SPECIALOP && rate == ASIG)
	eval_expr(cx,b->st->expr,rate);*/
			
      break;
			
    case ELSE:
      if (checkrate >= rate) {
	val = eval_expr(cx,b->st->expr,rate);
	if (val) {
	  if (val = eval_block(cx,b->st->b,rate))
	    return(val);
	}
	else
	  if (val = eval_block(cx,b->st->elseb,rate))
	    return(val);
      }
      /* if (checkrate == SPECIALOP && rate == ASIG)
	 eval_expr(cx,b->st->expr,rate); */
      
      break;
			
    case WHILE:
      if (checkrate == rate) {
	while (eval_expr(cx,b->st->expr,rate))
	  if (val = eval_block(cx,b->st->b,rate))
	    return(val);
      }
      if (checkrate == SPECIALOP && rate == ASIG) {
	eval_expr(cx,b->st->expr,rate);
      }
			
      break;
			
    case OUTPUT:
      if (checkrate == ASIG && rate == ASIG) {
	actparam_list *pl = NULL;
	exprlist *el;
				
	/* allow whole-array output? */
				
	for (el = b->st->params;el;el=el->next)
	  pl = add_actparam_list(pl,eval_expr(cx,el->p,rate),NULL,
				 el->p->rate == XSIG ? cx->xsigrate : el->p->rate);
				
	/* multiple outputs? */
				
	instr_output(cx,pl);
				
	free_actparam_list(pl);
      }
      break;
			
    case OUTBUS:
      if (rate == ASIG) {
	actparam_list *pl = NULL;
	exprlist *el;
				
	/* allow whole-array output? */
				
	for (el = b->st->params;el;el=el->next)
	  pl = add_actparam_list(pl,eval_expr(cx,el->p,rate),NULL,
				 el->p->rate == XSIG ? cx->xsigrate : el->p->rate);
				
	bus_output(b->st->bus,cx,pl);
				
	free_actparam_list(pl);
      }
      break;
			
    case RETURN:
			
      /* rate check? */
      return(eval_expr(cx,b->st->expr,rate));
      break;
			
    case TURNOFF:
      if (rate == KSIG) {
	instr_turnoff(cx);
      }
      break;
			
    case EXTEND:
      if (rate == KSIG || (checkrate == SPECIALOP && rate == KSIG)) {
	instr_extend(cx,eval_expr(cx,b->st->expr,rate));
      }
      if (checkrate == SPECIALOP && rate == ASIG)
	eval_expr(cx,b->st->expr,rate);
      
      break;
			
    case INSTR:
      if (checkrate == rate || (checkrate == SPECIALOP && rate >= KSIG)) {
	actparam_list *pl = NULL;
	exprlist *el;
	instr_handle *h;
				
	for (el = b->st->params;el;el=el->next)
	  pl = add_actparam_list(pl,eval_expr(cx,el->p,rate),NULL,
				 IVAR);
				
	if (rate == KSIG)
	  h = new_instr_instance(cx->sa,b->st->iname,pl,0);
				
	free_actparam_list(pl);
      }
      break;
    }
  }
  return 0;
}


double eval_expr(context *cx,expr *p,long rate) {
  actparam_list *val=NULL;
  exprlist *el;
  double ret;
  
  switch (p->op) {
  case IDENT:
    return(get_var_value(cx,p,0));
    break;
  case NUMBER:
    return(p->d->cval);
    break;
  case STRCONST:
    interror("can't evaluate string constant!");
    break;
  case ARRAYREF:
    return(get_var_value(cx,p,(int)(eval_expr(cx,p->left,rate)+0.5)));
    break;
  case OPCALL:
    for (el=p->params;el && el->p;el=el->next) {
      if (el->p->op == IDENT || el->p->op == ARRAYREF) /* call by reference */
	val = add_actparam_list(val,0.,el->p,
				el->p->rate == XSIG ? cx->xsigrate : el->p->rate);
      else val = add_actparam_list(val,eval_expr(cx,el->p,rate),NULL,
				   el->p->rate == XSIG ? cx->xsigrate : el->p->rate);
    }
    ret = eval_opcode(cx,p,val,rate);
    free_actparam_list(val);
    return(ret);
		
    break;
  case NOT:
    return(!(eval_expr(cx,p->left,rate)));
    break;
  case UMINUS:
    return(-(eval_expr(cx,p->left,rate)));
    break;
  case LP:
    return(eval_expr(cx,p->left,rate));
    break;
  case Q:
    return(eval_expr(cx,p->left,rate) ? eval_expr(cx,p->right,rate) :
	   eval_expr(cx,p->extra,rate));
    break;
  case LEQ:
    return(eval_expr(cx,p->left,rate) <= eval_expr(cx,p->right,rate));
    break;
  case GEQ:
    return(eval_expr(cx,p->left,rate) >= eval_expr(cx,p->right,rate));
    break;
  case EQEQ:
    return(eval_expr(cx,p->left,rate) == eval_expr(cx,p->right,rate));
    break;
  case NEQ:
    return(eval_expr(cx,p->left,rate) != eval_expr(cx,p->right,rate));
    break;
  case GT:
    return(eval_expr(cx,p->left,rate) > eval_expr(cx,p->right,rate));
    break;
  case LT:
    return(eval_expr(cx,p->left,rate) < eval_expr(cx,p->right,rate));
    break;
  case AND:
    return(eval_expr(cx,p->left,rate) && eval_expr(cx,p->right,rate));
    break;
  case OR:
    return(eval_expr(cx,p->left,rate) || eval_expr(cx,p->right,rate));
    break;
  case PLUS:
    return(eval_expr(cx,p->left,rate) + eval_expr(cx,p->right,rate));
    break;
  case MINUS:
    return(eval_expr(cx,p->left,rate) - eval_expr(cx,p->right,rate));
    break;
  case STAR:
    return(eval_expr(cx,p->left,rate) * eval_expr(cx,p->right,rate));
    break;
  case SLASH:
    return(eval_expr(cx,p->left,rate) / eval_expr(cx,p->right,rate));
    break;
  }
  return 0;			/* not reached */
}

double eval_opcode(context *enc_cx, expr *opcall,
		   actparam_list *pl, long rate) {
  context *cx;
  
  double rval;
  int ct = 0;
  
  /* look up local context storage */
  
  if (!(cx = get_child_context(enc_cx,opcall))) {
    /* have to make new context */
    cx = new_child_context(enc_cx,opcall,pl);
    /* do its i-time setup if we're already faster */
    if (rate > IVAR) {
      push_context(cx,enc_cx,pl,IVAR);
      pop_context(cx,enc_cx,pl,IVAR);
    }
  }
  
  
  /* copy parameter and global values into context */
  cx->asample_ptr = enc_cx->asample_ptr;
  push_context(cx,enc_cx,pl,rate);
  
  
  /* evaluate block wrt new context */
  if (opcall->co_ptr)
    rval = eval_core_opcode(cx,opcall,rate);
  else {
    symbol *sym = get_opcode_decl(enc_cx->sa->all->g->sym,opcall->d->iname);
    rval = eval_block(cx,((opcode_decl *)(sym->defn))->code,rate);
  }
  
  pop_context(cx,enc_cx,pl,rate);
  return(rval);
}

void push_context(context *cx, context *enc_cx,
		  actparam_list *pl, long rate) {
  actparam_list *pval=NULL;
  symtable *t;
  int idx;
  frameval *fv;
  int i;
  
  /* first, any standard names */
  for (t=cx->localvars;
       t && t->s->binding == STANDARD_NAME;
       t=t->next) {
    if (t->s->type == rate)
      if (t->s->width != 0)	/* skip "input" */
	for (i=0;i!=t->s->width;i++)
	  cx->framevals[t->s->offset+i+cx->inchan]->val =
	    get_host_value(cx->instr,t->s,i);
      else
	if (!cx->framevals[0]->asig) /* haven't connected it yet */
	  for (i=0;i!=cx->inchan;i++)
	    cx->framevals[i]->asig = cx->instr->input[i];
  }
  
  /* next, any actual parameters */
  if (pl)			/* o/w, it's a k or arate loop through an instr instance */
    for (pval=pl;
	 pval && t && t->s->binding == FORMALPARAM;
	 t=t->next,pval=pval->next) {
      if (pval && pval->rate && pval->rate <= rate) { /* anything which might have changes */
	if (pval->ref) {	/* call by reference */
	  cx->frameref[t->s->offset + cx->inchan] = 1;
	  if (pval->ref->op == ARRAYREF) /* array value */
	    idx = (int)(eval_expr(enc_cx,pval->ref->left,rate) +.5);
	  else
	    idx = 0;
	  /* just a pointer assignment */
	  cx->framevals[t->s->offset+cx->inchan] = get_frameval(enc_cx,pval->ref,idx);
	}
	else 
	  cx->framevals[t->s->offset+cx->inchan]->val = pval->val;
      }
    }
  else
    for (t; t && t->s->binding == FORMALPARAM; t=t->next);
  /* skip pfields during instr k-time and a-time */
  
  if (pval)			/* still have actual parameters left; they're varargs */
    cx->varargs = make_varargs_list(pval,enc_cx);
  else
    cx->varargs = NULL;
  
  
  /* finally, any imported variables and tables */
  
  /* for each local variable */
  for (; t; t = t->next) {
				
    /* if it's an imported variable */
    if (t->s->imported &&
	/* and we're at the variable's rate */
	((t->s->type == rate) ||
	 /* or the variable is a table and we're in i-time */
	 (t->s->type == TABLE && rate == IVAR) ||
	 /* or the variable is an xsig and the xsigrate is what we're at */
	 (t->s->type == XSIG && rate == cx->xsigrate) ||
	 /* or the variable is an asig (or xsig->asig), and we're in
	    a k-cycle */
	 ((t->s->type == ASIG ||
	   (t->s->type == XSIG && cx->xsigrate == ASIG)) &&
	  rate == KSIG))) {
      /* then copy in the global variable or host variable */
      if (t->s->glink)		/* global variable */
	for (i=0;i!=t->s->width;i++) {
	  /* copy each channel */
	  fv = cx->sa->global_cx->framevals[t->s->glink->offset+i];
	  cx->framevals[t->s->offset+i+cx->inchan]->val = fv->val;
	  cx->framevals[t->s->offset+i+cx->inchan]->table = fv->table; /* VIOLATION */
	  cx->framevals[t->s->offset+i+cx->inchan]->asig = fv->asig; /* VIOLATION */		       
	}
      else			/* shared with host */
	for (i=0;i!=t->s->width;i++)
	  if (t->s->type == ASIG ||
	      t->s->type == XSIG && cx->xsigrate == ASIG)
	    cx->framevals[t->s->offset+i+cx->inchan]->asig =
	      get_host_asig(cx->instr,t->s,i);
	  else
	    cx->framevals[t->s->offset+i+cx->inchan]->val =
	      get_host_value(cx->instr,t->s,i);
    }
				
    /* make tables and asigs if needed */
    if (!t->s->imported && t->s->type == TABLE &&
	!cx->framevals[t->s->offset+cx->inchan]->table)
      cx->framevals[t->s->offset+cx->inchan]->table = make_table(cx,t->s->table);
				
    if (!t->s->imported && !cx->framevals[t->s->offset+cx->inchan]->asig &&
	((t->s->type == ASIG) ||
	 (t->s->type == XSIG && cx->xsigrate == ASIG)))
      for (i=0;i!=t->s->width;i++)
	cx->framevals[t->s->offset+i+cx->inchan]->asig = make_asig_frame(cx->sa->ksmps);

    /* push static asigs forward */
    if (!t->s->imported && 
	cx->framevals[t->s->offset+cx->inchan]->asig)
      if (cx->asample_ptr > 0)
	for (i=0;i!=t->s->width;i++)
	  cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->asample_ptr] =
	    cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->asample_ptr-1];
      else
	for (i=0;i!=t->s->width;i++)
	  cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->asample_ptr] =
	    cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->sa->ksmps-1];

  }			
  cx->init = 1;
}

actparam_list *make_varargs_list(actparam_list *first, context *cx) {
  /* no reference parameters here */
  actparam_list *p,*ptr = NULL;
  for (p = first;p;p=p->next) {
    ptr = add_actparam_list(ptr,p->val,NULL,p->rate);
    if (p->ref) {
      ptr->val = get_var_value(cx,p->ref,0);
    }
  }
  return(ptr);
}


asig_frame *make_asig_frame(int ksmps) {
  asig_frame *asig;
	
  if (!(asig = (asig_frame *)calloc(sizeof(asig_frame),ksmps)))
    runtime("Couldn't allocate space for asig frame.\n");
	
  return(asig);
}

void pop_context(context *cx, context *enc_cx,
		 actparam_list *pl, long rate) {
  int i;
  symtable *t;
  frameval *fv;
  
  for (t=cx->localvars;
       t && t->s->binding == STANDARD_NAME;
       t=t->next);
  for (;t && t->s->binding == FORMALPARAM; t=t->next);
  
  /* only need to export variables (and tables) */
  
  /* for each local variable */
  for (; t; t=t->next) {
    
    /* if it's an exported variable */
    if (t->s->exported && ((t->s->type == rate) ||
			   (t->s->type == XSIG && rate == cx->xsigrate) ||
			   ((t->s->type == ASIG ||
			     (t->s->type == XSIG && cx->xsigrate == ASIG))
			    && (rate == KSIG)))) {
      
      if (t->s->glink)		/* global variable */
	for (i=0;i!=t->s->width;i++) {
	  /* copy each channel */
	  fv = cx->sa->global_cx->framevals[t->s->glink->offset];
	  fv->val = cx->framevals[t->s->offset+i+cx->inchan]->val;
	  /* don't need to do tables or asigs
	     since we're cheating in this implem. */
	}
      else			/* shared with host */
	for (i=0;i!=t->s->width;i++) {
	  if (t->s->type == KSIG ||
	      (t->s->type == XSIG && cx->xsigrate == KSIG))
	    set_host_value(cx->instr,t->s->name,i,
			   cx->framevals[t->s->offset+i+cx->inchan]->val);
	  if (t->s->type == ASIG ||
	      (t->s->type == XSIG && cx->xsigrate == ASIG))
	    set_host_asig(cx->instr,t->s,i,
			  cx->framevals[t->s->offset+i+cx->inchan]->asig);
	}

    }
    
  }
}


context *new_context(sa_decoder *sa, symtable *t,int inchan) {
  int ct = 0; 
  symtable *ptr;
  context *cx;
  int i;
	
  for (ptr = t; ptr; ptr=ptr->next)
    ct += (ptr->s->width == 0 ? inchan : ptr->s->width);
	
  PROT_MAL(cx,context,new_context);

  if (ct) { /* otherwise, no vars */
    if (!(cx->framevals = (frameval **)calloc(sizeof(frameval *), ct)))
      interror("calloc() failure in new_context()\n");
    if (!(cx->frameref = (int *)calloc(sizeof(int *),ct)))
      interror("calloc() failure in new_context()\n");
  }
	
  cx->instr = NULL;
  cx->init = 0;
  cx->ccx = NULL;
  cx->localvars = t;
  cx->varargs = NULL;
  cx->xsigrate = 0;
  cx->asample_ptr = 0;
  cx->cop_storage = NULL;
  cx->cop_dyn = NULL;
  cx->inchan = inchan;
  cx->sa = sa;
	
  for (ptr=t,ct=0;ptr;ct+=ptr->s->width ? ptr->s->width : inchan,ptr=ptr->next) {
    for (i=0;i!=(ptr->s->width ? ptr->s->width : inchan);i++) {
      PROT_MAL(cx->framevals[ct+i],frameval,new_context);
      cx->framevals[ct+i]->val = 0;
      cx->framevals[ct+i]->table = NULL;
      cx->framevals[ct+i]->asig = 0;
      cx->frameref[ct+i] = 0;
    }
  }
	
  /* register_context(cx) ? */
  return(cx);
}

context *get_child_context(context *cx, expr *opcall) {
  /* opcall is a call to an opcode; see if there's a child context for
     this call and if so, return it */
  child_cx_list *ccx;
	
  for (ccx = cx->ccx; ccx; ccx=ccx->next)
    if (ccx->ref == opcall)
      return(ccx->cx);
		
  return NULL;
}

context *new_child_context(context *cx, expr *opcall, actparam_list *pval) {
  /* make a new child context of the context 'cx' for the opcode
     call 'opcall' and return it */
  context *newcx;
  symbol *sym;
  opcode_decl *op;
  long fastest = 0;
  child_cx_list *ccx,*ptr,*last;
	
  sym = get_opcode_decl(cx->sa->all->g->sym,opcall->d->iname);
  op = (opcode_decl *)(sym->defn);
  
  newcx = new_context(cx->sa,op->sym,0);
  if (op->rate == XSIG) {
    for (;pval;pval=pval->next) /* figure out xsig rate */
      if (pval->rate > fastest)
	fastest = pval->rate;
    newcx->xsigrate = fastest;
  }
  else
    newcx->xsigrate = op->rate;
  newcx->instr = cx->instr;	/* its in the same instrument */
		
  /* make a new ccx list entry */
  PROT_MAL(ccx,child_cx_list,new_child_context);
  ccx->ref = opcall;
  ccx->cx = newcx;
  ccx->next = NULL;
	
  /* put it in the child context list of the enclosing context */
  if (!cx->ccx) 
    cx->ccx = ccx;
  else {
    last = cx->ccx;
    for (ptr=cx->ccx; ptr; last=ptr, ptr=ptr->next) ;
    last->next = ccx;
  }
  return(newcx);
}

frameval *new_frameval(double val, table_storage *table, asig_frame *asig) {
  frameval *fv;
	
  PROT_MAL(fv, frameval, new_frameval); printf("(3)");
  fv->val = 0;
  fv->table = NULL;
  fv->asig = NULL;
  if (val) fv->val = val;
  if (table) fv->table = table;
  if (asig) fv->asig = asig;
  return(fv);
}


table_storage *make_table(context *cx, tabledef *td) {
  actparam_list *tv = NULL;
  exprlist *el;
  double retn;
  table_storage *t;
	
  for (el = td->params; el; el=el->next) {
    if ( (el->p->op == IDENT &&
	  get_table_decl(cx->localvars,el->p->d->iname)) ||
	(el->p->op == STRCONST) )
      tv = add_actparam_list(tv,0,el->p,IVAR);
    else {
      retn = eval_expr(cx,el->p,IVAR);
      tv = add_actparam_list(tv,retn,NULL,IVAR);
    }
  }
	
  t = gen_table(td->gen,tv);
  t->name = td->name;
  return(t);
}

frameval *get_frameval(context *cx, expr *p, int index) {
  char s[1000];
  symbol *sym;
  int ct=0;
	
  sym = p->d->sym;
  if (!sym) {
    sprintf(s,"Wavetable '%s' undeclared.\n",p->d->iname);
    runtime(s);
  }
	
  if ((sym->width && sym->width <= index) || 
      (sym->width == 0 && cx->inchan <= index)) {
    sprintf(s,"Array '%s' out of bounds (max %d, index %d)",
	    p->d->iname, sym->width-1, index);
    runtime(s);
  }
  if (sym->offset < 0) /* input */
    return(cx->framevals[index]);
  else return(cx->framevals[sym->offset+index+cx->inchan]);
}

void set_frameval(context *cx, expr *p, int index, frameval *val) {
  symbol *sym;
  int ct=0;
  char s[1000];
	
  sym = p->d->sym;
  
  if (sym->width <= index) {
    sprintf(s,"Array '%s' out of bounds (max %d, index %d)",
	    p->d->iname, sym->width-1, index);
    runtime(s);
  }
  else {
    cx->framevals[sym->offset+index]->val = val->val;
    cx->framevals[sym->offset+index]->table = val->table;
  }
}

void set_var_value(context *cx, expr *p, int index, double val) {
  frameval *fv = get_frameval(cx,p,index);
	
  if (fv->asig)
    fv->asig[cx->asample_ptr] = val;
  else
    fv->val = val;
	
}

void set_var_value_byname(context *cx, char *vname, int index, double val) {
  /* only used for host setting variables */
  symtable *t;
  char s[800];
	
  for (t = cx->localvars;strcmp(t->s->name,vname);t=t->next);
  if (!t) {
    sprintf(s,"No such variable '%s' in instrument '%s'.",
	    vname,cx->instr->id->name);
    runtime(s);
  }
	
  else cx->framevals[t->s->offset + index +cx->inchan]->val = val;
}

double get_var_value(context *cx, expr *p, int index) {
  /* we know p->d is a terminal */
  frameval *fv = get_frameval(cx,p,index);
	
  if (fv->asig)
    return(fv->asig[cx->asample_ptr]);
  return(fv->val);
}

actparam_list *add_actparam_list(actparam_list *pl, double val,
				 expr *ref, long rate) {
  actparam_list *newp,*last,*ptr;
  
  PROT_MAL(newp,actparam_list,add_actparam_list);
  newp->val = val;
  newp->ref = ref;
  newp->next = NULL;
  newp->rate = rate;
  
  if (!pl) return newp;
  
  for (ptr=pl;ptr; last=ptr,ptr=ptr->next) ;
  last->next = newp;
  return(pl);
}

void free_actparam_list(actparam_list *pl) {
  actparam_list *last=NULL;
	
  for (;pl;last=pl,pl=pl->next)
    if (last) free(last);
  if (last) free(last);
}

instr_handle *new_instr_instance(sa_decoder *sa, char *iname, actparam_list *pf,int inchan) {
  /* allocate new context for instr; run i-time; register with scheduler */
  /* nb first value in pf parameter list is the duration */
  symbol *sym;
  instr_decl *id;
  context *cx;
  instr_handle *h;
	
  sym = get_instr_decl(sa->all->g->sym,iname); /* look up instr in symbol table */
  if (!sym) {
    printf("Unknown instrument '%s', continuing...\n",iname);
  } else {
		
    id = (instr_decl *)(sym->defn); /* get its declaration */
    cx = new_context(sa,id->sym,inchan); /* make the context */
    cx->localvars = id->sym;
    cx->xsigrate = ASIG;
		
    /* NB first argument is duration */
    h = register_inst(sa,id,cx,pf->val); /* tell the scheduler about it */
    h->origin = ORIGIN_INSTR;
    h->inchan = inchan;
    cx->instr = h;		/* save its handle */
		
    add_default_host_vars(h);
    add_imported_host_vars(h);
    set_host_value(h,"time",0,sa->sched->time);
    set_host_value(h,"dur",0,pf->val);
    run_itime(id,cx,pf->next);	/* and run its i-time stuff */
    return(h);
  }
}


void add_default_host_vars(instr_handle *h) {
  int i;
	
  for (i=0;i!=NUM_STD_NAMES;i++)
    new_host_var(h,std_names[i].name,std_names[i].width,std_names[i].type);
  set_host_value(h,"s_rate",0,h->cx->sa->all->g->srate);
  set_host_value(h,"k_rate",0,h->cx->sa->all->g->krate);
  set_host_value(h,"inchan",0,h->inchan);
  set_host_value(h,"outchan",0,h->cx->sa->all->g->outchan);
  set_host_value(h,"released",0,0);
  set_host_value(h,"stolen",0,0);
  set_host_value(h,"cpuload",0,0);
}

void add_imported_host_vars(instr_handle *h) {
  /* add host variables declared as 'imported' in this instrument */
  symtable *t;

  for (t = h->cx->localvars; t && t->s; t=t->next) {
    if (t->s->imported && !t->s->glink && !t->s->table) {
      new_host_var(h,t->s->name,t->s->width, t->s->type);
      set_host_value(h,t->s->name,0,0);
    }
  }
}


void run_itime(instr_decl *id, context *cx, actparam_list *pf) {
  /* run the i-time things in instrument ID */
  int ct=0;
	
	
  push_context(cx,NULL,pf,IVAR);
	
  /* eval the i-time code wrt the new context */
  eval_block(cx,id->code,IVAR);
	
  /* copy out global variables */
	
  pop_context(cx,NULL,pf,IVAR);
}

void run_katime(instr_decl *id, context *cx,long rate) {
  int i;
	
  push_context(cx,NULL,NULL,rate);
	
  if (rate == KSIG) {		/* run once */
    set_asample_ptr(cx,0);
    eval_block(cx,id->code,KSIG);
  }
  else {			/* run many times; the sample pointer tells
				   get_ and set_var_value where to read/put
				   audio samples */
    zero_instr_output(cx->instr);
    for (i=0;i!=cx->sa->ksmps;i++) {
      set_asample_ptr(cx,i);
      eval_block(cx,id->code,ASIG);
    }
  }
  
  pop_context(cx,NULL,NULL,rate);
}


void set_asample_ptr(context *cx, int which) {
  symtable *t;
  int i;
  
  cx->asample_ptr = which;
  
  for (t=cx->localvars; t && t->s; t=t->next) {
    if (t->s->binding == LOCAL && 
	!t->s->imported && 
	cx->framevals[t->s->offset+cx->inchan]->asig)
      if (cx->asample_ptr > 0)
	for (i=0;i!=t->s->width;i++)
	  cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->asample_ptr] =
	    cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->asample_ptr-1];
      else
	for (i=0;i!=t->s->width;i++)
	  cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->asample_ptr] =
	    cx->framevals[t->s->offset+i+cx->inchan]->asig[cx->sa->ksmps-1];
  }
}
