#include "snd.h"
#include <X11/cursorfont.h>

/* support for mixing operations
 *
 * the widgets used to handle mix-marks are independent of the mix-data structs;
 * off-screen mix data has no associated widget; on-screen mixes have a
 * widget while they remain on screen, and remain unsaved (active);  the 
 * mix-mark widget pool is global across channels/sounds.  The intent here
 * is to minimize widgets as far as possible (default is just the name button,
 * opens to the console requested in Mix Preferences, minimum in chans possible).
 */


/* ---------------- MIX CURSOR ---------------- */

static Cursor mix_cursor;

static void make_mix_cursor(snd_state *ss)
{
  mix_cursor = XCreateFontCursor(XtDisplay(main_SHELL(ss)),XC_left_ptr);
}

static void mix_mouse_enter(Widget w, XtPointer clientData, XEvent *event, Boolean *flag)
{
  XDefineCursor(XtDisplay(w),XtWindow(w),mix_cursor);
}

static void mix_mouse_leave(Widget w, XtPointer clientData, XEvent *event, Boolean *flag)
{
  XUndefineCursor(XtDisplay(w),XtWindow(w));
}



/* ---------------- MIX CONSOLE PREFERENCES ---------------- */
/* 
 * called from main menu Options:Mix Preferences option.
 *
 * set tiny font, main font, waveform on/off, controls on/off, console size, cursor shape, 
 *     group levels, stat display, whether to have consoles at all (see snd_state.consoling)
 */


/* TODO: some form of error for user when font is not available */
/* TODO: how to specify font "6x12" or whatever as XLFD? */

/* font preference handlers */

typedef struct {
  Widget row,name,size,name_text,size_text;
} prefont;

static char fontbuf[128];

static void set_font_row(prefont *pf, char *name, int size)
{
  XmTextSetString(pf->name_text,name);
  sprintf(fontbuf,"%d",size);
  XmTextSetString(pf->size_text,fontbuf);
}

static char *get_font_row(prefont *pf)
{
  sprintf(fontbuf,"-*-%s-medium-*-*-*-%s-*-*-*-*-*-iso8859-1",
	  XmTextGetString(pf->name_text),
	  XmTextGetString(pf->size_text));
  return(fontbuf);
}

static prefont *make_font_row(Widget w, snd_state *ss, char *label, Arg *in_args, int in_n)
{
  int n;
  Arg args[20];
  prefont *pf;
  pf = (prefont *)calloc(1,sizeof(prefont));
  
  pf->row = XtCreateManagedWidget("pfrow",xmRowColumnWidgetClass,w,in_args,in_n);
  
  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  pf->name = XtCreateManagedWidget(label,xmLabelWidgetClass,pf->row,args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,pf->name); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNcolumns,12); n++;
  pf->name_text = sndCreateTextFieldWidget(ss,"name_text",pf->row,args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  pf->size = XtCreateManagedWidget(snd_string_size_p,xmLabelWidgetClass,pf->row,args,n);

  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(args[n],XmNleftWidget,pf->size); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNcolumns,3); n++;
  pf->size_text = sndCreateTextFieldWidget(ss,"size_text",pf->row,args,n);

  return(pf);
}

static int mix_waving = 1;
static int mix_stating = 1;
static int mix_controling = 1;
static int mix_width = 100;
static int mix_height = 100;
static int tiny_size = 7;
static int normal_size = 14;
static char *tiny_name = NULL;
static char *normal_name = NULL;


/* tie to snd-clm */
int get_mix_int_var (int which)
{
  switch (which)
    {
    case 0: return(mix_waving); break;
    case 1: return(mix_stating); break;
    case 2: return(mix_controling); break;
    case 3: return(mix_width); break;
    case 4: return(mix_height); break;
    case 5: return(tiny_size); break;
    case 6: return(normal_size); break;
    }
  return(0);
}

char *get_mix_char_var (int which)
{
  switch (which)
    {
    case 0: return(tiny_name); break;
    case 1: return(normal_name); break;
    }
  return(NULL);
}

void set_mix_int_var (int which, int val)
{
  switch (which)
    {
    case 0: mix_waving = val; break;
    case 1: mix_stating = val; break;
    case 2: mix_controling = val; break;
    case 3: mix_width = val; break;
    case 4: mix_height = val; break;
    case 5: tiny_size = val; break;
    case 6: normal_size = val; break;
    }
}

void set_mix_char_var (int which, char *val)
{
  switch (which)
    {
    case 0: if (tiny_name) free(tiny_name); tiny_name = copy_string(val); break;
    case 1: if (normal_name) free(normal_name); normal_name = copy_string(val); break;
    }
}


static Widget mix_preferences = NULL;

static prefont *tiny_pf,*normal_pf;
static Widget wm_form,wm_wave,wm_stat,wm_control,wm_togrow;
static Widget wm_width,wm_height,wm_widthl,wm_heightl,wm_widrow;

static void make_tiny_label(Widget w, char *str)
{
  XmString s1;
  s1=XmStringCreate(str,"tiny_font");
  XtVaSetValues(w,XmNlabelString,s1,NULL);
  XmStringFree(s1);
}

static void make_normal_label(Widget w, char *str)
{
  XmString s1;
  s1=XmStringCreate(str,"normal_font");
  XtVaSetValues(w,XmNlabelString,s1,NULL);
  XmStringFree(s1);
}

static int tag_font(snd_state *ss, char *fontname, char *tagname)
{
  XFontStruct *fs;
  Display *dpy;
  dpy = main_DISPLAY(ss);
  fs = XLoadQueryFont(dpy,fontname);
  if (fs)
    XmFontListEntryCreate(tagname,XmFONT_IS_FONT,fs);
  else return(-1);
  return(0);
}

static void set_mix_preferences (snd_state *ss)
{
  char *tf;
  int err;
  err = tag_font(ss,get_font_row(tiny_pf),"tiny_font");
  if (!err) 
    {
      tiny_name = XmTextGetString(tiny_pf->name_text);
      tf = XmTextGetString(tiny_pf->size_text);
      sscanf(tf,"%d",&tiny_size);
    }
  err = tag_font(ss,get_font_row(normal_pf),"normal_font");
  if (!err) 
    {
      normal_name = XmTextGetString(normal_pf->name_text);
      tf = XmTextGetString(normal_pf->size_text);
      sscanf(tf,"%d",&normal_size);
    }
  mix_waving = XmToggleButtonGetState(wm_wave);
  mix_stating = XmToggleButtonGetState(wm_stat);
  mix_controling = XmToggleButtonGetState(wm_control);
  tf = XmTextGetString(wm_width);
  sscanf(tf,"%d",&err);
  if (err > 0) mix_width = err;
  tf = XmTextGetString(wm_height);
  sscanf(tf,"%d",&err);
  if (err > 0) mix_height = err;
}

static void reflect_mix_preferences(snd_state *ss)
{
  sprintf(fontbuf,"%d",tiny_size);
  XmTextSetString(tiny_pf->size_text,fontbuf);
  sprintf(fontbuf,"%d",normal_size);
  XmTextSetString(normal_pf->size_text,fontbuf);
  sprintf(fontbuf,"%d",mix_width);
  XmTextSetString(wm_width,fontbuf);
  sprintf(fontbuf,"%d",mix_height);
  XmTextSetString(wm_height,fontbuf);
  XmToggleButtonSetState(wm_wave,mix_waving,FALSE);
  XmToggleButtonSetState(wm_stat,mix_stating,FALSE);
  XmToggleButtonSetState(wm_control,mix_controling,FALSE);
  if (!tiny_name) tiny_name = copy_string("courier");
  XmTextSetString(tiny_pf->name_text,tiny_name);
  if (!normal_name) normal_name = copy_string("courier");
  XmTextSetString(normal_pf->name_text,normal_name);
}

static void Preferences_Ok_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  set_mix_preferences((snd_state *)clientData);
} 

static void Preferences_Cancel_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  reflect_mix_preferences((snd_state *)clientData);
} 

static void Preferences_Help_Callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_help((snd_state *)clientData,
	   "Mix Preferences",
"This dialog determines various aspects of\n\
the mix mini-consoles.  The three buttons\n\
control the display in the console of the\n\
associated sound's overall amplitude envelope,\n\
various statistics that describe it, and\n\
the set of controls that accompany each mix.\n\
The 'tiny' font is used when a console is\n\
otherwise closed (to save screen space and\n\
reduce clutter), and the 'normal' font is\n\
used when the console is open.  The size of\n\
each console is set (in pixels) by the 'console\n\
width' and 'height' fields.  The 'cancel'\n\
button returns to the state of the mix preferences\n\
before the dialog was activated.\n\
");
} 

void get_mix_preferences(snd_state *ss)
{
  XmString helpstr,cancelstr,okstr;
  int n;
  Arg args[20];
  if (!mix_preferences)
    {
      n=0;
      okstr=XmStringCreateLocalized(snd_string_Ok);
      XtSetArg(args[n],XmNokLabelString,okstr); n++;
      helpstr = XmStringCreateLocalized(snd_string_Help);
      XtSetArg(args[n],XmNhelpLabelString,helpstr); n++;
      cancelstr = XmStringCreateLocalized(snd_string_Cancel);
      XtSetArg(args[n],XmNcancelLabelString,cancelstr); n++;
#ifdef LINUX
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      mix_preferences = XmCreateMessageDialog(main_PANE(ss),snd_string_Mix_Preferences,args,n);
      XmStringFree(okstr);
      XmStringFree(helpstr);
      XmStringFree(cancelstr);

      XtUnmanageChild(XmMessageBoxGetChild(mix_preferences,XmDIALOG_SYMBOL_LABEL));
      XtAddCallback(mix_preferences,XmNhelpCallback,Preferences_Help_Callback,ss);
      XtAddCallback(mix_preferences,XmNcancelCallback,Preferences_Cancel_Callback,ss);
      XtAddCallback(mix_preferences,XmNokCallback,Preferences_Ok_Callback,ss);

      n=0;
      wm_form = XtCreateManagedWidget("wmform",xmFormWidgetClass,mix_preferences,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmVERTICAL); n++;
      wm_togrow = XtCreateManagedWidget("wmtogrow",xmRowColumnWidgetClass,wm_form,args,n);

      n=0;
      wm_wave = XtCreateManagedWidget(snd_string_show_overall_waveform,xmToggleButtonWidgetClass,wm_togrow,args,n);
      wm_stat = XtCreateManagedWidget(snd_string_show_various_statistics,xmToggleButtonWidgetClass,wm_togrow,args,n);
      wm_control = XtCreateManagedWidget(snd_string_show_mix_controls,xmToggleButtonWidgetClass,wm_togrow,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,wm_togrow); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      tiny_pf = make_font_row(wm_form,ss,snd_string_tiny_font_name_p,args,n);
      
      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,tiny_pf->row); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      normal_pf = make_font_row(wm_form,ss,snd_string_normal_font_name_p,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNtopWidget,normal_pf->row); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
      wm_widrow = XtCreateManagedWidget("wmwidrow",xmRowColumnWidgetClass,wm_form,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      wm_widthl = XtCreateManagedWidget(snd_string_console_width_p,xmLabelWidgetClass,wm_widrow,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,wm_widthl); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNcolumns,5); n++;
      wm_width = sndCreateTextFieldWidget(ss,"wm_width",wm_widrow,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      wm_heightl = XtCreateManagedWidget(snd_string_height_p,xmLabelWidgetClass,wm_widrow,args,n);

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_WIDGET); n++;
      XtSetArg(args[n],XmNleftWidget,wm_heightl); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_NONE); n++;
      XtSetArg(args[n],XmNcolumns,5); n++;
      wm_height = sndCreateTextFieldWidget(ss,"wm_height",wm_widrow,args,n);
      if (!(ss->using_schemes)) map_over_children(mix_preferences,set_main_color_of_widget,(void *)ss);
      XtVaSetValues(wm_wave,XmNselectColor,(ss->sgx)->text,NULL);
      XtVaSetValues(wm_stat,XmNselectColor,(ss->sgx)->text,NULL);
      XtVaSetValues(wm_control,XmNselectColor,(ss->sgx)->text,NULL);
    }
  reflect_mix_preferences(ss);
  if (!XtIsManaged(mix_preferences)) XtManageChild(mix_preferences);
}



/* ---------------- WIDGET POOL ---------------- */

#define WITH_CONTROLS 1
#define WITH_WAVEFORM 2
#define WITH_CONSOLE 4
#define WITH_GROUPS 8
#define WITH_STATS 16

enum {mm_main,           /* form holds console, catches mouse acts (click, drag) and cursor motion */
      mm_title,          /* top row of widgets */
      mm_play,           /* togglebutton, click to play "solo" or to stop */
      mm_name,           /* label; if console closed, tiny font+small box and click=open; if open click = highlight?, drag=reposition */
      mm_close,          /* pushbutton; if clicked, close to tiny */
      mm_title_sep,      /* if open, horizontal sep below title */
      mm_button_sep,     /* if open, horizontal sep after controls */
                         /* buttons at bottom (mimic dialog box) */
      mm_help,           /* help monolog */
      mm_mute,           /* (temporarily) remove current from output (for play) */
      mm_ok,             /* finalize mix -- close mix console and remove mix data info */
      mm_speed,          /* if open, srate control */
      mm_remix,          /* force update of original to ripple through new */
      mm_chans};         /* if open, start of controls (per input channel) */
enum {mm_scl,mm_env};    /* if open, each channel in has these (scaler, envelope as text field) */
#define MIX_CHAN_SIZE 2

typedef struct {
  int type,inuse,state,chans_allocated;
  Widget *w;
} mixmark;

#define MPOOL_INCREMENT 16
static mixmark *mpool;
static int mpool_size = 0;

static mixmark get_mixmark(void)
{
  int i,next;
  if (mpool_size == 0) 
    {
      mpool_size = MPOOL_INCREMENT;
      mpool = (mixmark *)calloc(mpool_size,sizeof(mixmark));
    }
  for (i=0;i<mpool_size;i++)
    {
      if (!(mpool[i].inuse)) return(mpool[i]);
    }
  /* no free mixmark slots found */
  next = mpool_size;
  mpool_size += MPOOL_INCREMENT;
  mpool = (mixmark *)realloc(mpool,sizeof(mixmark)*mpool_size);
  return(mpool[next]);
}

static void unuse_mixmark(mixmark m)
{
  m.inuse = 0;
  if ((m.w) && ((m.w[mm_main]) && (XtIsManaged(m.w[mm_main])))) 
    XtUnmanageChild(m.w[mm_main]);
}


/* ---------------- LOCAL MIX CONSOLE ---------------- */

static void add_mixer_chans(mixmark m, int chans_needed)
{
}

static void mix_console_play_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* a toggle button -- if playing, click=>stop else start */
}

static void mix_console_close_callback(Widget w,XtPointer clientData,XtPointer callData) 
{
  /* a push button -- remove console associated with mix (i.e. make mix unchangeable) */
}

static void mix_title_button_press(Widget w, XtPointer clientData, XEvent *event, Boolean *flag) {}
static void mix_title_button_release(Widget w, XtPointer clientData, XEvent *event, Boolean *flag) {}
static void mix_title_button_motion(Widget w, XtPointer clientData, XEvent *event, Boolean *flag) {}
static void mix_title_key_press(Widget w, XtPointer clientData, XEvent *event, Boolean *flag) {}

/* drag label => reposition mix, click label => open console if it's closed */

static void create_mixer(snd_state *ss, void *userData, mixmark m, char *name, int state, int chans)
{
  /* make a new mixer console */
  int n;
  Arg args[32];
  /* for now, just the titlebar -- main, row, name pushbutton, play pushbutton, close pushbutton */
  /* assume m.w already allocated */

  n=0;
  m.w[mm_main] = XtCreateManagedWidget("mm_main",xmFormWidgetClass,main_PANE(ss),args,n);
  
  n=0;
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_NONE); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNorientation,XmHORIZONTAL); n++;
  m.w[mm_title] = XtCreateManagedWidget("mm_title",xmRowColumnWidgetClass,m.w[mm_main],args,n);

  n=0;
  m.w[mm_name] = XtCreateManagedWidget(name,xmLabelWidgetClass,m.w[mm_title],args,n);
  m.w[mm_play] = XtCreateManagedWidget("play",xmToggleButtonWidgetClass,m.w[mm_title],args,n);
  m.w[mm_close] = XtCreateManagedWidget("close",xmPushButtonWidgetClass,m.w[mm_title],args,n);
  XtAddCallback(m.w[mm_play],XmNactivateCallback,mix_console_play_callback,userData);
  XtAddCallback(m.w[mm_close],XmNactivateCallback,mix_console_close_callback,userData);

  XtAddEventHandler(m.w[mm_main],EnterWindowMask,FALSE,mix_mouse_enter,(XtPointer)NULL);
  XtAddEventHandler(m.w[mm_main],LeaveWindowMask,FALSE,mix_mouse_leave,(XtPointer)NULL);
  XtAddEventHandler(m.w[mm_title],ButtonPressMask,FALSE,mix_title_button_press,(XtPointer)userData);
  XtAddEventHandler(m.w[mm_title],ButtonMotionMask,FALSE,mix_title_button_motion,(XtPointer)userData);
  XtAddEventHandler(m.w[mm_title],ButtonReleaseMask,FALSE,mix_title_button_release,(XtPointer)userData);
  XtAddEventHandler(m.w[mm_title],KeyPressMask,FALSE,mix_title_key_press,(XtPointer)userData);
}

static void use_mixmark(snd_state *ss, mixmark m, char *name, int chans_needed, int initial_state)
{
  m.inuse = 1;
  if (!(m.w))
    {
      /* never previously used */
      m.w = (Widget *)calloc(mm_chans+MIX_CHAN_SIZE*chans_needed,sizeof(Widget));
      m.chans_allocated = chans_needed;
      create_mixer(ss, NULL, m, name, initial_state, chans_needed);
    }
  else
    {
      /* check for chans in need of creation */
      if (chans_needed > m.chans_allocated)
	{
	  m.w = (Widget *)realloc(m.w,(mm_chans+MIX_CHAN_SIZE*chans_needed)*sizeof(Widget));
	  add_mixer_chans(m,chans_needed);
	  m.chans_allocated = chans_needed;
	}
    }
  /* manage main, etc (dependent on initial_state), make sure trailers are unmanaged */
  m.state = initial_state;
}


/* ---------------- MIX POOL ---------------- */
typedef struct {
  int chans;    /* size of arrays in this struct */
  int edit_ctr; 
  int beg,end;  /* in output */
  env **envs;   /* per channel input */
  char **envstrs;
  float *scalers;
  float *speeds;
} console_state;

typedef struct {
  int type;
  int size;
  int initial_edit_ctr;
  snd_state *ss;
  chan_info *cp;
  Widget graph;
  char *name;
  mixmark mixer;
  char *in_filename;
  char *out_filename;
  int *data;
  file_info *hdr;
  int console_state_size;
  console_state *states;
} mixdata;

#define MXD_INCREMENT 16

void free_mixes(chan_info *cp)
{
  int i;
  mixdata **mxd;
  if (cp->mixes)
    {
      mxd = (mixdata **)(cp->mixes);
      for (i=0;i<cp->mix_size;i++)
	{
	  if (mxd[i])
	    {
	      /* free malloc'd portions */
	      free(mxd[i]);
	      mxd[i] = NULL;
	    }
	}
      free(cp->mixes);
      cp->mixes = NULL;
    }
}

static int get_free_mixdata_index(chan_info *cp)
{
  int i,next;
  mixdata **mxd;
  if (cp->mix_size == 0)
    {
      cp->mix_size = MXD_INCREMENT;
      cp->mixes = (mixdata **)calloc(cp->mix_size,sizeof(mixdata *));
      make_mix_cursor(cp->state);
      return(0);
    }
  mxd = (mixdata **)(cp->mixes);
  for (i=1;i<cp->mix_size;i++)
    {
      if (!(mxd[i])) return(i);
    }
  /* now make more space */
  next = cp->mix_size;
  cp->mix_size += MXD_INCREMENT;
  cp->mixes = (mixdata **)realloc(cp->mixes,cp->mix_size * sizeof(mixdata *));
  return(next);
}



/* ---------------- MIX ACTIONS ---------------- */
static mixdata *make_mixdata(chan_info *cp)
{
  mixdata *md;
  md = (mixdata *)calloc(1,sizeof(mixdata));
  md->cp = cp;
  md->ss = cp->state;
  md->graph = chan_widget(cp,W_chn_graph);
  return(md);
}

void display_mixmark(chan_info *cp, axis_info *ap, int show) /* show=0 means erase */
{
  mixdata *md;

}

void update_all_consoles(snd_state *ss)
{
  /* called to turn console display on/off in snd-xmenu (show_consoles) */
}
