/*
  File: ContourFitter.c
  Authors: David Meyers (who no longer wants to admit it)
           K.R. Sloan
  Last Modified: 15 August 1990
  Purpose: "Industrial strength" version of the surface from contours project.
           The current version uses the "OneColumn" search method with biasing
	   of the subdivision.  All options relating to the choice of search
	   method and instrumentation of the performance of the algorithms
	   have been removed.
           The input format has been revised, and the output
	   writes a mesh compatible with the surface fitting and rendering
	   pipeline. The current implementation is a filter. Input comes from
	   stdin, output goes to stdout. For details of the input file format,
	   see InputStuff.h

           We are experimenting with normalization schemes to avoid
           problems due to translation, scaling, and rotation of sequential
           cross-sections.  Occasionally, we want to mix and match, or
           perhaps simply turn off some of the normalization.

           So, a set of strange switches:

               -T : translate so that Contour1 and Contour2 are on
                    a common axis, which is normal to one of them,
                    usually Contour1.  This eliminates most of
                    the "double-cone" artifacts. [DEFAULT]

               -t : don't do that - usually to demonstrate the artifact

               -S : scale Contour2 so that "maximum distance of a point
                    from the centroid" matches Contour1. [DEFAULT]

               -s : don't do that

               -R : rotate (2D?  3D) Contour2 to match (how?) Contour1
                    [DEFAULT]...but we don't really do it

               -r: don't do it, even if we figure out how...

	       -d: Debug mode

 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <TypeDefinitions.h>
#include <BitArray.h>
#include <Contours.h>
#include <TorGraph.h>
#include <Vector.h>
#include <InputStuff.h>
#include <OutputStuff.h>
#include <Aggregates.h>
#include <Scaling.h>

extern void free();
extern void exit();
extern BitArray OptimalArea();

static int TRANSLATE = 1;
static int SCALE = 1;
static int ROTATE = 1;
static double CloseEnough = 0.0001; /*need flag to allow the user to set this*/

static char *RoutineName;
static void usage()
 {
  (void)
   fprintf(stderr,
	   "Usage is:%s\n\t[-T][-t][-R][-r][-S][-s][-D Tolerance][-d]\n",
	   RoutineName);
 }

int
main(argc,argv)
 int argc;
 char *argv[];
{
 int ArgsParsed = 0;
 BitArray directions;
 int start1, start2;
 Section *Section1, *Section2, *temp;
 ContourDescriptor *CDesc1, *CDesc2;
 PointsType *Contour1, *Contour2, *C1Copy, *C2Copy;
 int TriangleNumber = 0;
 int Contour1Index, Contour2Index;
 int Debug = FALSE;

 RoutineName = argv[ArgsParsed++];
 while (ArgsParsed < argc)
  {
   if ('-' == argv[ArgsParsed][0])
    {
     switch (argv[ArgsParsed++][1])
      {
       case 'T': TRANSLATE = 1; break;
       case 't': TRANSLATE = 0; break;
       case 'S': SCALE = 1; break;
       case 's': SCALE = 0; break;
       case 'R': ROTATE = 1; break;
       case 'r': ROTATE = 0; break;
       case 'D': CloseEnough = atof(argv[ArgsParsed++]); break;
       case 'd': Debug = TRUE; break;
       default:
       case 'h': usage(); exit(-1);
      }
    }
   else {usage(); exit(-1);}
  }  

 Section1 = (Section *) malloc(sizeof(Section));
 Section2 = (Section *) malloc(sizeof(Section));

 ReadSection(stdin, Section1);
 BuildNameIndex(Section1, CloseEnough);
 WritePoints(Section1, stdout);
 ReadSection(stdin, Section2);
 BuildNameIndex(Section2, CloseEnough);

 while(NULL != Section2->TheContours)
  {
   WritePoints(Section2, stdout);
   for(Contour1Index=0; Contour1Index < Section1->NContours; Contour1Index++)
    {
     /*
       The Visited field of a contour is used to prevent duplication of
       output. If Contours are merged, they are marked as visited. When the
       merged contour is later checked by the next conditional, it will be
       skipped. Since we look at a section twice, once as Section1 and
       once as Section2, we need to be a bit tricky. We look for contours
       in Section2 which correspond to contours (or merged contours) in
       Section1. We don't mark contours visited in Section2, only in Section1.
       The net effect is that we only use each contour once.
      */
     if (FALSE == Section1->TheContours[Contour1Index].Visited)
      {
       CDesc1 = MergeContours(Section1,
			      Section1->TheContours[Contour1Index].Name,
			      FALSE, TRUE,
			      &TriangleNumber);
       if(FindContour(Section2, Section1->TheContours[Contour1Index].Name,
		      &Contour2Index, TRUE))
	{
	 CDesc2 = MergeContours(Section2,
				Section2->TheContours[Contour2Index].Name,
				TRUE, FALSE,
				&TriangleNumber);
	 Contour1 = MakeAggregateContour(Section1, CDesc1);
	 if(Debug) DumpAggregateContour(Section1, Contour1, CDesc1);
	 Contour2 = MakeAggregateContour(Section2, CDesc2);
	 if(Debug) DumpAggregateContour(Section2, Contour2, CDesc2);

	 C1Copy = CopyPoints(Contour1); C2Copy = CopyPoints(Contour2);
	 directions = FindTransform(C1Copy, C2Copy,
				    &start1, &start2,
				    TRANSLATE, SCALE, ROTATE);
	 FreePoints(C1Copy); FreePoints(C2Copy);

	 WriteSurface(Contour1, Contour2, CDesc1, CDesc2,
		      start1, start2, &TriangleNumber,
		      Section1, Section2, directions);
	 DisposeBitArray(directions);
	 FreePoints(Contour1);  FreePoints(Contour2);
	 FreeDescriptor(CDesc1);  FreeDescriptor(CDesc2);
	}
      }
    }
   temp = Section1;
   Section1 = Section2;
   Section2 = temp;
   ClearSection(Section2);
   ReadSection(stdin, Section2);
   BuildNameIndex(Section2, CloseEnough);
  }
 WriteMesh(TriangleNumber);
}
