/*
  File: ContourEditor.c
  Authors: K.R. Sloan, 
           Dave Meyers,
           Cheryl Buck
  Last Modified: 13 March 1990
  Purpose:  display  4 views of an object - top, front, side and perspective &
            allow editing of object sections.
 */

#include <stdio.h>
#include <math.h>
#include <errno.h>
#include "3D.h"
#include "Render.h"
#include <TypeDefinitions.h>
#include "ContourEditor.h"
#include "Contours.h"
#include "SectionInput.h"
#include "SectionOutput.h"

extern char *malloc();

int VERBOSE = 0;
int DEBUG = 0;
int UPTODATE = 1;
int CENTERSET = 0;
int WIDTHSET = 0;
int VIEWSET = 0;
int DeletePointBox = 0;
int EditBox = 0;
int NextBox = 0;
int PrevBox = 0;
int UpperInsetBox = 0;
int LowerInsetBox = 0;

static double Foreground = 0.0;
static double Background = 1.0;
static int NBoxes = 0;

#define MAXPOINTS 1024
static sgpPointType Points[MAXPOINTS];
static int NPoints = 0;

static char *DataFile = "ContourData";
static char *TempFile = "TempData";
static char *OutputFile = "NewContourData";
static int MaxSections = 0;
static int CurrentSection = 0;
static int KeepSection = 1;
static double OriginalFocalLength = 0.040;
static double FocalLength = 0.040;
static double ViewingDistance;
static double OriginalDistance;
static double Width;
static double DiscSize;
static PointType3D Center;
static PointType3D EyePoint;
static VectorType3D Up;
static VectorType3D OpticalAxis;


FILE *TempFilePtr;
FILE *NewFilePtr;
#define NUMBERofSECTIONS (1000)
static SectionSize SectionData[NUMBERofSECTIONS];
static Section *TheSection = NULL;
static char *RoutineName;
#define MaxBoxes 20
static BoxType Boxes[MaxBoxes];
static RulerType RulerData, FLengthData;
static void usage()
 {
  (void) fprintf(stderr,
  "Usage is\n\t%s [-c x y z] [-w d] [-d d] [-v] [-i inputfilename] \n         [-o outputfilename]\n", RoutineName);
 }

/*
  This program is based on a main loop which waits for a mouse-button
  click, determines which (if any) of several active viewports ("buttons")
  has been selected, and calls a routine attached to the button.

  All of this requires:
    0) a place to store the information about the buttons
    1) a set of action routines
    2) a set-up procedure
    3) the main loop

  The action routines below all take 2 parameters:
    0) an index into the set of buttons
    1) the NDC Screen coordinates of the mouse when the button was 
       clicked.  Most routines ignore both of these.  Routines which
       actually use the Screen coordinates will typically transform
       them back to World coordinates.
 */


static void BuildSectionRecords(FilePointer)
     FILE *FilePointer;
{
  int Count, status, i;
  double MinX, MaxX, MinY, MaxY, MinZ, MaxZ;
  long Position;
  Section *ASection;
  
  Count = -1; MinX = 0.0; MaxX = 0.0; MinY = 0.0; MaxY = 0.0; 
              MinZ = 0.0; MaxZ = 0.0;
 
  Position = ftell(FilePointer);
  if (-1 == Position) { perror("bad Position"); exit(-1); }

  ASection = (Section *) malloc(sizeof(Section));
  ReadSection(FilePointer, ASection);

  while(NULL != ASection->TheContours)
   {
     Count = Count + 1;     
     SectionData[Count].StartingLocation = Position;
     Position = ftell(FilePointer);
     if (-1 == Position) { perror("bad Position"); exit(-1); }
     SectionData[Count].SizeOfSection = 
          Position - SectionData[Count].StartingLocation;
     ReadSection(FilePointer, ASection);
     if ((!CENTERSET) || (!WIDTHSET))
       {
        for(i=0; i<ASection->NContours; i++)
         { int j, Npts;
 
           Npts = ASection->TheContours[i].ContourPoints->n ;
           for(j=0; j<Npts; j++)
            {
              if (ASection->TheContours[i].ContourPoints->P[j].x > MaxX)
                MaxX = ASection->TheContours[i].ContourPoints->P[j].x;
              else
                if (ASection->TheContours[i].ContourPoints->P[j].x < MinX)
                  MinX = ASection->TheContours[i].ContourPoints->P[j].x;

              if (ASection->TheContours[i].ContourPoints->P[j].y > MaxY)
                MaxY = ASection->TheContours[i].ContourPoints->P[j].y;
              else
                if (ASection->TheContours[i].ContourPoints->P[j].y < MinY)
                  MinY = ASection->TheContours[i].ContourPoints->P[j].y;

              if (ASection->TheContours[i].ContourPoints->P[j].z > MaxZ)
                MaxZ = ASection->TheContours[i].ContourPoints->P[j].z;
              else
                if (ASection->TheContours[i].ContourPoints->P[j].z < MinZ)
                  MinZ = ASection->TheContours[i].ContourPoints->P[j].z;
            }
         }
      }
    }

  MaxSections =  Count;
  
  if (!(WIDTHSET))
    {
      Width = MaxX - MinX;
      if ((MaxY - MinY) > Width) 
        Width = MaxY - MinY;
      Width = (Width * 1.50) / 2.0; /* width for window larger than object */
      DiscSize = Width / 80.0;
      WIDTHSET = 1;
    }

  if (!(CENTERSET))
    {
      Center.x = ((MaxX - MinX) / 2.0) + MinX; 
      Center.y = ((MaxY - MinY) / 2.0) + MinY;
      Center.z = ((MaxZ - MinZ) / 2.0) + MinZ;
      CENTERSET = 1;
    }

  if (!VIEWSET)
    {
      ViewingDistance = Width/1.5*4.0;  /* set vdist based on object width */
      OriginalDistance = ViewingDistance;
      VIEWSET = 1;
    }
 
  if (VERBOSE)
   {
    (void) fprintf(stderr,"%s: Viewing Distance = { %f}\n", RoutineName,
                        ViewingDistance);
    (void) fprintf(stderr,"%s: Center = { %f, %f, %f }\n", RoutineName,
                        Center.x, Center.y, Center.z);
    (void) fprintf(stderr,"%s: Width = { %f}\n", RoutineName, (Width/1.10)); 
    (void) fprintf(stderr,"%s: InputFile = %s, OutputFile = %s\n", 
                  RoutineName, DataFile, OutputFile);
   }


 }

static void UpdateFile()
{
  int status;
  char SystemCommand[85];
  char TempString[40];

  WriteNewFile(NewFilePtr, TempFilePtr, CurrentSection, 
            TheSection, MaxSections, SectionData, KeepSection);
  status = fclose(TempFilePtr);
  status = fclose(NewFilePtr);
  strcpy(SystemCommand, "mv ");
  strcpy(TempString, TempFile);
  strcat(SystemCommand, TempString);
  strcpy(TempString, OutputFile);
  strcat(SystemCommand, " ");
  strcat(SystemCommand, TempString);
  system(SystemCommand);   /*"mv TempData NewContourData") */

  NewFilePtr = fopen(OutputFile, "r");
  if ((FILE *)0 == NewFilePtr)
   {
    fprintf(stderr,"Cannot open %s\n",OutputFile);
    exit(-1);
   }
  TempFilePtr = fopen(TempFile, "w");
  if ((FILE *)0 == TempFilePtr)
   {
    fprintf(stderr,"Cannot open %s\n",TempFile);
    exit(-1);
   }
  BuildSectionRecords(NewFilePtr);
  UPTODATE = 1;
  KeepSection = 1;
}


static void DrawAdjacencies(ASection, TheAdjacency, ContourIndex)
   Section *ASection;
   Adjacency TheAdjacency;
   int ContourIndex;
{
  Adjacency FirstAdjacency;
  int EqualStrings, DONE;

  DONE = 0;

  FirstAdjacency = TheAdjacency;
  while (! DONE)
    {
     int k, AdjacentContIndex;
     sgpPointType P1, P2;

     AdjacentContIndex = -1;
     for (k=0; k<ASection->NContours; k++)
      {
        EqualStrings = strcmp(ASection->TheContours[k].Name,
                              TheAdjacency.Name);
        if (0 == EqualStrings)
            AdjacentContIndex = k;
       } 
     if (AdjacentContIndex == -1)
       fprintf(stderr, "Error in Adjacency naming.\n");           
     else
      {
        P1.x = 
          ASection->TheContours[ContourIndex].ContourPoints->P[TheAdjacency.LocalIndex].x;
        P1.y = 
          ASection->TheContours[ContourIndex].ContourPoints->P[TheAdjacency.LocalIndex].y;
        P2.x = 
          ASection->TheContours[AdjacentContIndex].ContourPoints->P[TheAdjacency.AdjacentIndex].x;
        P2.y = 
          ASection->TheContours[AdjacentContIndex].ContourPoints->P[TheAdjacency.AdjacentIndex].y;

        P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;
        P2.x = P2.x - Center.x; P2.y = P2.y - Center.y;  
        sgpLine(P1, P2);
     
     sgpGrayLevel(Background);
     sgpDisc(P1, (DiscSize * 0.65));     /* DiscSize * .65 */
     sgpGrayLevel(Foreground);

      }
      TheAdjacency = *(TheAdjacency.next);
      EqualStrings = strcmp(TheAdjacency.Name, FirstAdjacency.Name);
      if ((EqualStrings == 0) &&
       (TheAdjacency.LocalIndex == FirstAdjacency.LocalIndex) &&
       (TheAdjacency.AdjacentIndex == FirstAdjacency.AdjacentIndex))
      DONE = 1;
    }
 }  



static void DrawEditor(ASection, BoxNum)
   Section *ASection;
   int BoxNum;
{
  int i, DONE;
  sgpRectangleType Window;

  sgpGrayLevel(Background);
  sgpSetViewport(Boxes[BoxNum].Viewport);
  sgpSetWindow(Boxes[BoxNum].Window);
  sgpRectangle(Boxes[BoxNum].Window);
  sgpGrayLevel(Foreground);
  {
   sgpPointType P[5];
   P[0].x = -Width; P[0].y = -Width;
   P[1].x = Width; P[1].y = -Width;
   P[2].x = Width; P[2].y = Width;
   P[3].x = -Width; P[3].y = Width;
   P[4].x = P[0].x; P[4].y = P[0].y;
   sgpPolyLine(5, P);
  }

  for (i=0; i<ASection->NContours; i++)
    {
     int j, n;
     sgpPointType P0, P1;
     Adjacency FirstAdjacency;

     n = ASection->TheContours[i].ContourPoints->n;

     P0.x = ASection->TheContours[i].ContourPoints->P[0].x;
     P0.y = ASection->TheContours[i].ContourPoints->P[0].y;
     P0.x = P0.x - Center.x; P0.y = P0.y - Center.y;
     
     for(j=1; j<=n; j++)
       {

         if(VERBOSE)
         {
          (void) fprintf(stderr,
                "P0.x=%f, P0.y=%f\n", P0.x, P0.y);
          (void) fprintf(stderr,
                "Calling sgpDisc(P0, %f)\n", DiscSize);
         }
         sgpDisc(P0, DiscSize);

      P1.x = ASection->TheContours[i].ContourPoints->P[j%n].x;
      P1.y = ASection->TheContours[i].ContourPoints->P[j%n].y;
 
         P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;
      sgpLine(P0, P1);

         P0.x = P1.x; 
      P0.y = P1.y;
       }

     sgpDisc(P1, DiscSize);

/* handle the previous adjacency list */
      if (! (ASection->TheContours[i].AdjContPrev == NULL))
       {
         FirstAdjacency = *(ASection->TheContours[i].AdjContPrev);
         DrawAdjacencies(ASection, FirstAdjacency, i);
       }

/* handle the next adjacency list */
      if (! (ASection->TheContours[i].AdjContNext == NULL))
       {
         FirstAdjacency = *(ASection->TheContours[i].AdjContNext);
         DrawAdjacencies(ASection, FirstAdjacency, i);

       } 
    }     
   
}


static void DrawViews(FilePointer)
   FILE *FilePointer; 
{
  int ViewNumber, SeekStatus;
  long Offset;  
  Section *ASection = NULL;

  CurrentSection = 0;
  Offset = SectionData[CurrentSection].StartingLocation;
  for (ViewNumber = 0; ViewNumber <= 3; ViewNumber++)
  {
    SeekStatus = fseek(FilePointer, Offset, 0);
    if (-1 == SeekStatus) { perror("bad Seek"); exit(-1); }
    sgpGrayLevel(Background);  

    switch (ViewNumber)
    {
      case  0 :   EyePoint.x = 0.0; 
                  EyePoint.y = 0.0;
                  EyePoint.z = ViewingDistance;
                  Up.dx = 0.0; Up.dy = 1.0; Up.dz = 0.0;
                  OpticalAxis.dx = 0.0; OpticalAxis.dy = 0.0;
                  OpticalAxis.dz = -1.0;
                  break;
      case  1 :   EyePoint.x = 0.0; 
                  EyePoint.y = -ViewingDistance;
                  EyePoint.z = 0.0;
                  Up.dx = 0.0; Up.dy = 0.0; Up.dz = 1.0;
                  OpticalAxis.dx = 0.0; OpticalAxis.dy = 1.0;
                  OpticalAxis.dz = Center.x;
                  break;
      case  2 :   EyePoint.x = ViewingDistance;  
                  EyePoint.y = 0.0;
                  EyePoint.z = 0.0;
                  OpticalAxis.dx = -1.0; OpticalAxis.dy = 0.0;
                  break;
      case  3 :   EyePoint.x = ViewingDistance * 0.80;  
                  EyePoint.y = -ViewingDistance * 0.80; 
                  EyePoint.z = ViewingDistance * 0.80; 
                  OpticalAxis.dz = -1.0; OpticalAxis.dy = 1.0;
                  OpticalAxis.dx = -1.0; 
                  Up.dx = 0.0; Up.dy = 0.0; Up.dz = 1.0;
                  break;
     }

  sgpSetViewport(Boxes[ViewNumber].Viewport);
  sgpSetWindow(Boxes[ViewNumber].Window);
  sgpRectangle(Boxes[ViewNumber].Window); 
  Camera3D(EyePoint, OpticalAxis, Up);
  sgpGrayLevel(Foreground);

  if (ASection != NULL)
    ClearSection(ASection);

  ASection = (Section *) malloc(sizeof(Section));
  ReadSection(FilePointer, ASection);

  while(NULL != ASection->TheContours)
   {
    int i;

    for (i=0; i<ASection->NContours; i++)
     {
      int j, n;
      PointType3D P1, P2;

      n = ASection->TheContours[i].ContourPoints->n;
      P1.x = ASection->TheContours[i].ContourPoints->P[0].x;
      P1.y = ASection->TheContours[i].ContourPoints->P[0].y;
      P1.z = ASection->TheContours[i].ContourPoints->P[0].z;

      P1.x = P1.x - Center.x; P1.y = P1.y - Center.y; P1.z = P1.z - Center.z;

      for(j=1; j<=n; j++)
       {
         P2.x = ASection->TheContours[i].ContourPoints->P[j%n].x;
         P2.y = ASection->TheContours[i].ContourPoints->P[j%n].y;
         P2.z = ASection->TheContours[i].ContourPoints->P[j%n].z;

         P2.x = P2.x - Center.x; P2.y = P2.y - Center.y; 
         P2.z = P2.z - Center.z;

         if(VERBOSE)
         {
          (void) fprintf(stderr,
                "P1.x=%f, P1.y=%f, P1.z=%f\n", P1.x, P1.y, P1.z);
          (void) fprintf(stderr,
                "P2.x=%f, P2.y=%f, P2.z=%f\n", P2.x, P2.y, P2.z);
          (void) fprintf(stderr,
                "Calling Line3D(P1, P2)\n");
         }

        Line3D(P1, P2);
        P1.x = P2.x;
        P1.y = P2.y;
        P1.z = P2.z;
       }
     }

    if (ASection != NULL)
      ClearSection(ASection);

    ASection = (Section *) malloc(sizeof(Section));
    ReadSection(FilePointer, ASection);
   } /* end of while */
  }  /* end of for loop */
}


static void DrawCurrentSection(FilePointer, CurrentSection)
     FILE *FilePointer; 
     int CurrentSection;
{
  int SeekStatus;
  long Offset;

  Offset = SectionData[CurrentSection].StartingLocation;
  SeekStatus = fseek(FilePointer, Offset, 0);
  if (-1 == SeekStatus) { perror("bad Seek"); exit(-1); }
  if (TheSection != NULL)
    ClearSection(TheSection);
  TheSection = (Section *) malloc(sizeof(Section));
  ReadSection(FilePointer, TheSection);
  DrawEditor(TheSection, EditBox);  
}


static void DrawLine(WhatPoint, NumLines)
     PointLocator WhatPoint;
     int NumLines;
{
  sgpPointType P0, P1, P2;
  int PointNum, ContourNum, NumPoints;

  PointNum = WhatPoint.ThePoint;
  ContourNum = WhatPoint.TheContour;
  NumPoints = TheSection->TheContours[ContourNum].ContourPoints->n;

  P0.x = TheSection->TheContours[ContourNum].ContourPoints->P[PointNum].x;
  P0.y = TheSection->TheContours[ContourNum].ContourPoints->P[PointNum].y;

  P0.x = P0.x - Center.x; P0.y = P0.y - Center.y; 

/* get adjacent points */
  if (PointNum == 0) 
   {
    P1.x = TheSection->TheContours[ContourNum].ContourPoints->P[NumPoints-1].x;
    P1.y = TheSection->TheContours[ContourNum].ContourPoints->P[NumPoints-1].y;
   }
  else
   {
    P1.x = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum - 1].x;
    P1.y = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum - 1].y;
   }
   P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;

  if (PointNum == NumPoints - 1) 
   {
    P2.x = TheSection->TheContours[ContourNum].ContourPoints->P[0].x;
    P2.y = TheSection->TheContours[ContourNum].ContourPoints->P[0].y;
   }
  else
   {
    P2.x = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum + 1].x;
    P2.y = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum + 1].y;
   }

  P2.x = P2.x - Center.x; P2.y = P2.y - Center.y;

  sgpGrayLevel(Foreground);
  if (NumLines == 1)
    sgpLine(P1, P2);
  else
    {
      sgpLine(P0, P1);
      sgpLine(P0, P2);
    }

 /*  redraw adjacent discs */
  sgpDisc(P1, DiscSize);
  sgpDisc(P2, DiscSize);

}


static void EraseLine(WhatPoint, NumLines)
     PointLocator WhatPoint;
     int NumLines;
{
  sgpPointType P0, P1, P2;
  int PointNum, ContourNum, NumPoints;

  PointNum = WhatPoint.ThePoint;
  ContourNum = WhatPoint.TheContour;
  NumPoints = TheSection->TheContours[ContourNum].ContourPoints->n;

  P0.x = TheSection->TheContours[ContourNum].ContourPoints->P[PointNum].x;
  P0.y = TheSection->TheContours[ContourNum].ContourPoints->P[PointNum].y;

  P0.x = P0.x - Center.x; P0.y = P0.y - Center.y; 

/* get adjacent points */
  if (PointNum == 0) 
   {
    P1.x = TheSection->TheContours[ContourNum].ContourPoints->P[NumPoints-1].x;
    P1.y = TheSection->TheContours[ContourNum].ContourPoints->P[NumPoints-1].y;
   }
  else
   {
    P1.x = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum - 1].x;
    P1.y = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum - 1].y;
   }
   P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;

  if (PointNum == NumPoints - 1) 
   {
    P2.x = TheSection->TheContours[ContourNum].ContourPoints->P[0].x;
    P2.y = TheSection->TheContours[ContourNum].ContourPoints->P[0].y;
   }
  else
   {
    P2.x = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum + 1].x;
    P2.y = 
        TheSection->TheContours[ContourNum].ContourPoints->P[PointNum + 1].y;
   }

  P2.x = P2.x - Center.x; P2.y = P2.y - Center.y;

  sgpGrayLevel(Background);
  sgpLine(P0, P1);
  if (NumLines == 2)  
    sgpLine(P0, P2);

 /*  redraw adjacent discs */
  sgpGrayLevel(Foreground);
  if (NumPoints > 1)
   {
     sgpDisc(P1, DiscSize);
     sgpDisc(P2, DiscSize);
   }
}

static void EraseAPoint(WhatPoint)
  PointLocator WhatPoint;

{
  sgpPointType P0;
  int PointNum, ContourNum, NumPoints;

  PointNum = WhatPoint.ThePoint;
  ContourNum = WhatPoint.TheContour;

  P0.x = TheSection->TheContours[ContourNum].ContourPoints->P[PointNum].x;
  P0.y = TheSection->TheContours[ContourNum].ContourPoints->P[PointNum].y;

  P0.x = P0.x - Center.x; P0.y = P0.y - Center.y; 

/* erase the point */
  sgpGrayLevel(Background);
  if(VERBOSE)
    {
     (void) fprintf(stderr,
         "P0.x=%f, P0.y=%f\n", P0.x, P0.y);
     (void) fprintf(stderr,
         "Calling sgpDisc(P0, %f)\n", DiscSize);
    }

  sgpDisc(P0,DiscSize); 
  sgpGrayLevel(Foreground);
 }

static void EraseTheAdjacency(LocalPoint, AdjacentPoint)
     PointLocator LocalPoint, AdjacentPoint;
{ 
  sgpPointType P0, P1;
  
  P0.x = 
    TheSection->TheContours[LocalPoint.TheContour].ContourPoints->P[LocalPoint.ThePoint].x;
  P0.y = 
    TheSection->TheContours[LocalPoint.TheContour].ContourPoints->P[LocalPoint.ThePoint].y;
  P1.x = 
    TheSection->TheContours[AdjacentPoint.TheContour].ContourPoints->P[AdjacentPoint.ThePoint].x;
  P1.y = 
    TheSection->TheContours[AdjacentPoint.TheContour].ContourPoints->P[AdjacentPoint.ThePoint].y;

  P0.x = P0.x - Center.x; P0.y = P0.y - Center.y;
  P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;

  sgpGrayLevel(Background);
  sgpLine(P0, P1);
  
  sgpGrayLevel(Foreground);
  sgpDisc(P0, DiscSize);
  sgpDisc(P1, DiscSize);
}


static void GetOldPoint(APoint)
  PointLocator *APoint;

{ int PointSelected, ValidPoint, button;
  sgpPointType PickScreen, PickWorld;

  PointSelected = 0;
  ValidPoint = 0;
  while (ValidPoint ==0)
    {
      while (PointSelected == 0)
        {
          sgpPick(&Points[NPoints],&button); if (!button) continue;
      
          for(;button;) sgpPick(&Points[NPoints],&button);  /* debounce */
        
          sgpWorldToScreen(Points[NPoints], &PickScreen);
          if ( (Boxes[EditBox].Viewport.Left < PickScreen.x) 
            && (PickScreen.x < Boxes[EditBox].Viewport.Right)
            && (Boxes[EditBox].Viewport.Bottom < PickScreen.y)
            && (PickScreen.y < Boxes[EditBox].Viewport.Top) )
          {
            PointSelected = 1; 
          }
         }
      sgpScreenToWorld(PickScreen, &PickWorld);
      PickWorld.x = PickWorld.x + Center.x; 
      PickWorld.y = PickWorld.y + Center.y;
  
      ValidPoint=FindClosestPoint(PickWorld, APoint, TheSection); 
      if (0 == ValidPoint)
     {
       fprintf(stderr, "Point selected too far from valid point\n");
       PointSelected = 0;
     }
    }
}


static void ChooseNextSection()
{
  int SeekStatus, Selection, PointSelected, button;
  sgpPointType PickScreen;
  long Offset;

  if (0 == UPTODATE)
    UpdateFile();

  while (PointSelected == 0)
    {
     sgpPick(&Points[NPoints],&button); if (!button) continue;

     for(;button;) sgpPick(&Points[NPoints],&button);  /* debounce */

     sgpWorldToScreen(Points[NPoints], &PickScreen);
     if ( (Boxes[NextBox].Viewport.Left < PickScreen.x) 
         && (PickScreen.x < Boxes[NextBox].Viewport.Right)
         && (Boxes[NextBox].Viewport.Bottom < PickScreen.y)
         && (PickScreen.y < Boxes[NextBox].Viewport.Top) ) 
       {
      PointSelected = 1;
         Selection = 1;
       }

     if ( (Boxes[PrevBox].Viewport.Left < PickScreen.x) 
         && (PickScreen.x < Boxes[PrevBox].Viewport.Right)
         && (Boxes[PrevBox].Viewport.Bottom < PickScreen.y)
         && (PickScreen.y < Boxes[PrevBox].Viewport.Top) )
       {
         PointSelected = 1; 
      Selection = 0;
       }
     }

   if (Selection == 0)
     if (CurrentSection > 0)
       CurrentSection = CurrentSection - 1 ;
     else
       fprintf(stderr, 
      "There were no previous sections, NEXT section is in the edit screen\n");

   if (Selection == 1)
     if (CurrentSection > MaxSections)
      {
       fprintf(stderr, 
      "There were no next sections, PREVIOUS section is in the edit screen\n");
       CurrentSection = CurrentSection - 1;
      }     
    Offset = SectionData[CurrentSection].StartingLocation;
    SeekStatus = fseek(NewFilePtr, Offset, 0);
    if (-1 == SeekStatus) { perror("bad Seek"); exit(-1); }
    if (TheSection != NULL)
      ClearSection(TheSection);
    TheSection = (Section *) malloc(sizeof(Section));
    ReadSection(NewFilePtr, TheSection);
    DrawEditor(TheSection, EditBox);  
}


/*
   Functions attached to the buttons
 */

/*
  DistanceChange is a slider, selecting the viewing distance.  The distance 
  changes depending on where on the ruler the last click is.  When a new
  distance is selected we re-draw the icon, to indicate the current ruler.
 */
void DistanceChange(box,Screen)
 int box;
 sgpPointType Screen;
{
  sgpPointType P[2], WorldPoint;
  double Slide;
  double MaxPower = 2.0;
  double Base = 10.0;

  sgpSetViewport(Boxes[box].Viewport);
  sgpSetWindow(Boxes[box].Window);
  if (Screen.x < RulerData.MinX)
    Screen.x = RulerData.MinX;
  else
    if (Screen.x > RulerData.MaxX)
      Screen.x = RulerData.MaxX;

  if (Screen.x < RulerData.OriginalX)
    ViewingDistance = 
      ((Screen.x - RulerData.MinX) / (RulerData.OriginalX - RulerData.MinX)) * 
     OriginalDistance;
  else
    {
     Slide = 
        pow(Base, ((Screen.x - RulerData.OriginalX) / 
             (RulerData.MaxX - RulerData.OriginalX) * MaxPower));
     ViewingDistance = Slide * OriginalDistance;
    }

  fprintf(stderr, "Viewing Distance changed to: %f\n", ViewingDistance);

  sgpGrayLevel(Background);
  P[0] = RulerData.CurrentArrow;
  P[1].x = P[0].x; P[1].y = 0.2;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x - 0.05; P[1].y = 0.325;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x + 0.05;
  sgpLine(P[0], P[1]);

  sgpGrayLevel(Foreground);
  sgpScreenToWorld(Screen, &WorldPoint);
  P[0].x = WorldPoint.x;
  RulerData.CurrentArrow = P[0];
  P[1].x = WorldPoint.x; P[1].y = 0.2;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x - 0.05; P[1].y = 0.325;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x + 0.05;
  sgpLine(P[0], P[1]);
}


void FocalLengthChange(box,Screen)
 int box;
 sgpPointType Screen;
{
  sgpPointType P[2], WorldPoint;
  double Slide;
  double MaxPower = 3.0;
  double Base = 10.0;

  sgpSetViewport(Boxes[box].Viewport);
  sgpSetWindow(Boxes[box].Window);
  if (Screen.x < FLengthData.MinX)
    Screen.x = FLengthData.MinX;
  else
    if (Screen.x > FLengthData.MaxX)
      Screen.x = FLengthData.MaxX;

  if (Screen.x < FLengthData.OriginalX)
    {
      if (Screen.x < ((FLengthData.OriginalX - FLengthData.MinX)/2.0 + 
                                 FLengthData.MinX))
     FocalLength = 0;
      else
        FocalLength = 
         ((Screen.x - FLengthData.MinX) / (FLengthData.OriginalX - 
                   FLengthData.MinX)) * OriginalFocalLength;
    }
  else
    {
     Slide = 
        pow(Base, ((Screen.x - FLengthData.OriginalX) / 
             (FLengthData.MaxX - FLengthData.OriginalX) * MaxPower));
     FocalLength = Slide * OriginalFocalLength;
    }

  fprintf(stderr, "Focal Length changed to: %f\n", FocalLength);

  sgpGrayLevel(Background);
  P[0] = FLengthData.CurrentArrow;
  P[1].x = P[0].x; P[1].y = 0.1;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x - 0.05; P[1].y = 0.225;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x + 0.05;
  sgpLine(P[0], P[1]);

  sgpGrayLevel(Foreground);
  sgpScreenToWorld(Screen, &WorldPoint);
  P[0].x = WorldPoint.x;
  FLengthData.CurrentArrow = P[0];
  P[1].x = WorldPoint.x; P[1].y = 0.1;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x - 0.05; P[1].y = 0.225;
  sgpLine(P[0], P[1]);
  P[1].x = P[0].x + 0.05;
  sgpLine(P[0], P[1]);

  Lens3D(FocalLength); SetHither(FocalLength);

}


void Redraw(box, Screen)
  int box;
  sgpPointType Screen;
{
  int SaveSection;

  if (0 == UPTODATE)
    UpdateFile();
  
  SaveSection = CurrentSection;
  DrawViews(NewFilePtr);
  CurrentSection = SaveSection;
  DrawCurrentSection(NewFilePtr, CurrentSection);
}

void NoAction(box, Screen)
  int box;
  sgpPointType Screen;
{

}


void PreviousSection(box, Screen)
  int box;
  sgpPointType Screen;
{
  int SeekStatus;
  long Offset;

  if (0 == UPTODATE)
    UpdateFile();

  if (CurrentSection > 0)
    {
     CurrentSection = CurrentSection - 1;
     Offset = SectionData[CurrentSection].StartingLocation;
     SeekStatus = fseek(NewFilePtr, Offset, 0);
     if (-1 == SeekStatus) { perror("bad Seek"); exit(-1); }
     if (TheSection != NULL)
       ClearSection(TheSection);
     TheSection = (Section *) malloc(sizeof(Section));
     ReadSection(NewFilePtr, TheSection);
     DrawEditor(TheSection, EditBox);  
    }
  else
     fprintf(stderr, "There are no previous sections\n");
}


static void NextSection(box, Screen)
  int box;
  sgpPointType Screen;
{
  int SeekStatus;
  long Offset;

  if (0 == UPTODATE)
    UpdateFile();

  if (CurrentSection < MaxSections)
    {
     CurrentSection = CurrentSection + 1;
     Offset = SectionData[CurrentSection].StartingLocation;
     SeekStatus = fseek(NewFilePtr, Offset, 0);
     if (-1 == SeekStatus) { perror("bad Seek"); exit(-1); }
     if (TheSection != NULL)
       ClearSection(TheSection);
     TheSection = (Section *) malloc(sizeof(Section));
     ReadSection(NewFilePtr, TheSection);
     DrawEditor(TheSection, EditBox);  
    }
  else
     fprintf(stderr, "There are no more sections.\n");
}


void DeletePointOrAdjacency(box, Screen)
  int box;
  sgpPointType Screen;

{
  PointLocator WhatPoint, AdjPoint, LocalPoint;
  int InAdjacency;
 
  UPTODATE = 0;
  WhatPoint.ThePoint = 0; WhatPoint.TheContour = 0;
  GetOldPoint(&WhatPoint);
  InAdjacency = HandleLocalAdjacencies(WhatPoint, &AdjPoint, TheSection);

  if (InAdjacency)
    EraseTheAdjacency(WhatPoint, AdjPoint);

  InAdjacency = HandleAdjacentAdjacencies(WhatPoint, &LocalPoint, TheSection);

  if (InAdjacency)
    EraseTheAdjacency(LocalPoint, WhatPoint);

  if (box == DeletePointBox)
    {
     RenumberAdjacencies(WhatPoint, 0, TheSection);
     EraseAPoint(WhatPoint);
     if (TheSection->TheContours[WhatPoint.TheContour].ContourPoints->n > 1)
       {
        EraseLine(WhatPoint, 2);
        DrawLine(WhatPoint, 1);
       }
     DeletePointFromSection(WhatPoint, TheSection, &KeepSection);

     if (! KeepSection)   /* all pts in the section are deleted */
       {
      fprintf(stderr, "You must make a Next or Previous section selection now.\n");
      ChooseNextSection();
         KeepSection = 1;
       }
    }
  if (! UPTODATE)
    UpdateFile();
  DrawCurrentSection(NewFilePtr, CurrentSection);  
}


void MovePoint(box, Screen)
  int box;
  sgpPointType Screen;
{
  PointLocator WhatPoint;
  sgpPointType PickWorld, PickScreen;
  int PointSelected, button;

  UPTODATE = 0;
  WhatPoint.ThePoint = 0; WhatPoint.TheContour = 0;
  GetOldPoint(&WhatPoint);

  PointSelected = 0; 
  while (PointSelected == 0)
    {
      sgpPick(&Points[NPoints],&button); if (!button) continue;

      for(;button;) sgpPick(&Points[NPoints],&button);  /* debounce */

      sgpWorldToScreen(Points[NPoints], &PickScreen);
      if ( (Boxes[4].Viewport.Left < PickScreen.x) 
          && (PickScreen.x < Boxes[4].Viewport.Right)
          && (Boxes[4].Viewport.Bottom < PickScreen.y)
          && (PickScreen.y < Boxes[4].Viewport.Top) )
       {
         PointSelected = 1; 
       }
    }
/* erase old point */
    EraseAPoint(WhatPoint);
    EraseLine(WhatPoint, 2);

/* draw NewPoint */
    sgpScreenToWorld(PickScreen, &PickWorld);
    if(VERBOSE)
       {
        (void) fprintf(stderr,
              "Pick.x=%f, Pick.y=%f\n", PickWorld.x, PickWorld.y);
        (void) fprintf(stderr,
              "Calling sgpDisc(Pick, %f)\n", DiscSize);
       }
       sgpDisc(PickWorld, DiscSize); 
       PickWorld.x = PickWorld.x + Center.x; 
       PickWorld.y = PickWorld.y + Center.y;


/* manipulate the section data */
    MovePointInSection(WhatPoint, TheSection, PickWorld);
    UpdateFile();
    DrawCurrentSection(NewFilePtr, CurrentSection);

}
 

void AddPoint(box, Screen)
  int box;
  sgpPointType Screen;
{
  int PointSelected, button, ValidPoint, OneLine;
  PointLocator WhatPoint;
  sgpPointType PickScreen, PickWorld;

  UPTODATE = 0;
  OneLine = 1;
  WhatPoint.ThePoint = 0; WhatPoint.TheContour = 0;
  GetOldPoint(&WhatPoint);

  PointSelected = 0;
  while (PointSelected == 0)
    {
      sgpPick(&Points[NPoints],&button); if (!button) continue;

      for(;button;) sgpPick(&Points[NPoints],&button);  /* debounce */

      sgpWorldToScreen(Points[NPoints], &PickScreen);
      if ( (Boxes[4].Viewport.Left < PickScreen.x) 
        && (PickScreen.x < Boxes[4].Viewport.Right)
        && (Boxes[4].Viewport.Bottom < PickScreen.y)
        && (PickScreen.y < Boxes[4].Viewport.Top) )
      {
        PointSelected = 1; 
      }
    }

/* draw NewPoint */
    sgpScreenToWorld(PickScreen, &PickWorld);
    if(VERBOSE)
       {
        (void) fprintf(stderr,
              "Pick.x=%f, Pick.y=%f\n", PickWorld.x, PickWorld.y);
        (void) fprintf(stderr,
              "Calling sgpDisc(Pick, %f)\n", DiscSize);
       }
       sgpDisc(PickWorld, DiscSize); 
    PickWorld.x = PickWorld.x + Center.x; 
    PickWorld.y = PickWorld.y + Center.y;

    EraseLine(WhatPoint, OneLine);
    RenumberAdjacencies(WhatPoint, 1, TheSection);

/* manipulate the section data */

    AddPointToSection(WhatPoint, TheSection, PickWorld);
    UpdateFile();
    DrawCurrentSection(NewFilePtr, CurrentSection);

} 


void AddNextAdjacency(box, Screen)
    int box;
    sgpPointType Screen;
{
  PointLocator LocalPoint, AdjacentPoint;
  Adjacency *AdjacentPtr;
  sgpPointType P1, P2;
  int ContourNum;

  UPTODATE = 0;
  LocalPoint.ThePoint = 0; LocalPoint.TheContour = 0;
  AdjacentPoint.ThePoint = 0; AdjacentPoint.TheContour = 0;
  GetOldPoint(&LocalPoint);
  GetOldPoint(&AdjacentPoint);

  if (ContourNum > -1)
   {
     P1.x = 
      TheSection->TheContours[LocalPoint.TheContour].ContourPoints->P[LocalPoint.ThePoint].x;
     P1.y = 
      TheSection->TheContours[LocalPoint.TheContour].ContourPoints->P[LocalPoint.ThePoint].y;
     P2.x = 
      TheSection->TheContours[AdjacentPoint.TheContour].ContourPoints->P[AdjacentPoint.ThePoint].x;
     P2.y = 
      TheSection->TheContours[AdjacentPoint.TheContour].ContourPoints->P[AdjacentPoint.ThePoint].y;

     P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;
     P2.x = P2.x - Center.x; P2.y = P2.y - Center.y;  
     sgpLine(P1, P2);
     sgpGrayLevel(Background);
     sgpDisc(P1, (DiscSize * 0.65));
     sgpGrayLevel(Foreground);

     AdjacentPtr = (Adjacency *) malloc(sizeof(Adjacency));
     AdjacentPtr->next = AdjacentPtr;
     AdjacentPtr->prev = AdjacentPtr;
     strcpy( AdjacentPtr->Name, TheSection->TheContours[AdjacentPoint.TheContour].Name);  
     AdjacentPtr->LocalIndex = LocalPoint.ThePoint;
     AdjacentPtr->AdjacentIndex = AdjacentPoint.ThePoint;

     if (TheSection->TheContours[LocalPoint.TheContour].AdjContNext == NULL)
       TheSection->TheContours[LocalPoint.TheContour].AdjContNext = AdjacentPtr;
     else
     {
      Adjacency *FirstPtr, *InsertPointer;
   
      FirstPtr = (Adjacency *) malloc(sizeof(Adjacency));
      FirstPtr = TheSection->TheContours[LocalPoint.TheContour].AdjContNext;

      InsertPointer = FirstPtr;

      if ( AdjacentPtr->LocalIndex < FirstPtr->LocalIndex )
       /* insert at head of list */
       {
        AdjacentPtr->next = FirstPtr;
        AdjacentPtr->prev = FirstPtr->prev;
        FirstPtr->prev->next = AdjacentPtr;
        FirstPtr->prev = AdjacentPtr;
       }
      else /* insert not at head of list */
       {
        while ( (AdjacentPtr->LocalIndex > InsertPointer->next->LocalIndex) &&
           (InsertPointer->next != FirstPtr))
         {
          InsertPointer = InsertPointer->next;
         }
        /*
         AdjacentPtr->LocalIndex is >InsertPointer->LocalIndex here and we are
         either at wraparound point of the circular list or AdjacentPtr->LocalIndex is
          less than InsertPointer->next->LocalIndex
        */
        AdjacentPtr->next = InsertPointer->next;
        AdjacentPtr->prev = InsertPointer;
        AdjacentPtr->prev->next = AdjacentPtr;
        AdjacentPtr->next->prev = AdjacentPtr;
        InsertPointer->next = AdjacentPtr;
       }
     }
   }
  else
   {
    fprintf(stderr, "There is no next section - no action taken.\n");
    UPTODATE = 1;
   }
}

void AddPrevAdjacency(box, Screen)
    int box;
    sgpPointType Screen;
{
  PointLocator LocalPoint, AdjacentPoint;
  Adjacency *AdjacentPtr;
  sgpPointType P1, P2;
  int ContourNum;

  UPTODATE = 0;
  LocalPoint.ThePoint = 0; LocalPoint.TheContour = 0;
  AdjacentPoint.ThePoint = 0; AdjacentPoint.TheContour = 0;
  GetOldPoint(&LocalPoint);
  GetOldPoint(&AdjacentPoint);

  if (ContourNum > -1)
   {
     P1.x = 
      TheSection->TheContours[LocalPoint.TheContour].ContourPoints->P[LocalPoint.ThePoint].x;
     P1.y = 
      TheSection->TheContours[LocalPoint.TheContour].ContourPoints->P[LocalPoint.ThePoint].y;
     P2.x = 
      TheSection->TheContours[AdjacentPoint.TheContour].ContourPoints->P[AdjacentPoint.ThePoint].x;
     P2.y = 
      TheSection->TheContours[AdjacentPoint.TheContour].ContourPoints->P[AdjacentPoint.ThePoint].y;

     P1.x = P1.x - Center.x; P1.y = P1.y - Center.y;
     P2.x = P2.x - Center.x; P2.y = P2.y - Center.y;  
     sgpLine(P1, P2);
     sgpGrayLevel(Background);
     sgpDisc(P1, (DiscSize * 0.65));
     sgpGrayLevel(Foreground);

     AdjacentPtr = (Adjacency *) malloc(sizeof(Adjacency));
     AdjacentPtr->next = AdjacentPtr;
     AdjacentPtr->prev = AdjacentPtr;
     strcpy( AdjacentPtr->Name, TheSection->TheContours[AdjacentPoint.TheContour].Name);  
     AdjacentPtr->LocalIndex = LocalPoint.ThePoint;
     AdjacentPtr->AdjacentIndex = AdjacentPoint.ThePoint;

     if (TheSection->TheContours[LocalPoint.TheContour].AdjContPrev == NULL)
       TheSection->TheContours[LocalPoint.TheContour].AdjContPrev = AdjacentPtr;
     else
     {
      Adjacency *FirstPtr, *InsertPointer;
   
      FirstPtr = (Adjacency *) malloc(sizeof(Adjacency));
      FirstPtr = TheSection->TheContours[LocalPoint.TheContour].AdjContPrev;

      InsertPointer = FirstPtr;

      if ( AdjacentPtr->LocalIndex < FirstPtr->LocalIndex )
       /* insert at head of list */
       {
        AdjacentPtr->next = FirstPtr;
        AdjacentPtr->prev = FirstPtr->prev;
        FirstPtr->prev->next = AdjacentPtr;
        FirstPtr->prev = AdjacentPtr;
       }
      else /* insert not at head of list */
       {
        while ( (AdjacentPtr->LocalIndex > InsertPointer->next->LocalIndex) &&
           (InsertPointer->next != FirstPtr))
         {
          InsertPointer = InsertPointer->next;
         }
        /*
         AdjacentPtr->LocalIndex is >InsertPointer->LocalIndex here and we are
         either at wraparound point of the circular list or AdjacentPtr->LocalIndex is
          less than InsertPointer->next->LocalIndex
        */
        AdjacentPtr->next = InsertPointer->next;
        AdjacentPtr->prev = InsertPointer;
        AdjacentPtr->prev->next = AdjacentPtr;
        AdjacentPtr->next->prev = AdjacentPtr;
        InsertPointer->next = AdjacentPtr;
       }
     }
   }
  else
   {
    fprintf(stderr, "There is no previous section - no action taken\n");
    UPTODATE = 1;
   }

}

void Quit(box, Screen)
    int box;
    sgpPointType Screen;
{
  int status;
  char SystemCommand[85];
  char TempString[40];

  if (0 == UPTODATE)
    UpdateFile();

  status = fclose(TempFilePtr);
  status = fclose(NewFilePtr);
  strcpy(SystemCommand, "rm ");
  strcpy(TempString, TempFile);
  strcat(SystemCommand, TempString);
  system(SystemCommand);
  exit(0);
}


/* END of BUTTON FUNCTIONS */


/*
   Do everything required to set up the viewports, buttons, etc.
*/
static void SetUp()
 {
  sgpRectangleType Window;

  Window.Left = -Width; Window.Right = Width; Window.Bottom = -Width;
  Window.Top = Width;
  NBoxes = 0;

  if (DEBUG)
    sgpInit(DebugDisplay) ;
  else
    sgpInit(MonochromeDisplay);
  sgpGrayLevel(Background);  sgpClearScreen();
  Lens3D(FocalLength);  SetHither(FocalLength);

  /* box for ABOVE VIEW */
  Boxes[NBoxes].Viewport.Left   = 0.0; Boxes[NBoxes].Viewport.Right = 0.33;
  Boxes[NBoxes].Viewport.Bottom = 0.33; Boxes[NBoxes].Viewport.Top   = 0.66;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  NBoxes++;

  /* box for FRONT VIEW */
  Boxes[NBoxes].Viewport.Left   = 0.0; Boxes[NBoxes].Viewport.Right = 0.33;
  Boxes[NBoxes].Viewport.Bottom = 0.0; Boxes[NBoxes].Viewport.Top   = 0.33;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  NBoxes++;

  /* box for SIDE VIEW */ 
  Boxes[NBoxes].Viewport.Left   = 0.33; Boxes[NBoxes].Viewport.Right = 0.66;
  Boxes[NBoxes].Viewport.Bottom = 0.0; Boxes[NBoxes].Viewport.Top   = 0.33;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  NBoxes++;

  /* box for PERSPECTIVE VIEW */
  Boxes[NBoxes].Viewport.Left   = 0.66; 
  Boxes[NBoxes].Viewport.Right = 1.0;
  Boxes[NBoxes].Viewport.Bottom = 0.0; 
  Boxes[NBoxes].Viewport.Top   = 0.33;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  NBoxes++;

  /* box for EDITTING */
  Boxes[NBoxes].Viewport.Left   = 0.33; Boxes[NBoxes].Viewport.Right = 1.0;
  Boxes[NBoxes].Viewport.Bottom = 0.33; Boxes[NBoxes].Viewport.Top   = 1.0;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  EditBox = NBoxes;
  NBoxes++;

  /* box for Upper Inset EDITTING */
  Boxes[NBoxes].Viewport.Left   = 0.75; Boxes[NBoxes].Viewport.Right = 1.0;
  Boxes[NBoxes].Viewport.Bottom = 0.75; Boxes[NBoxes].Viewport.Top   = 1.0;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  UpperInsetBox = NBoxes;
  NBoxes++;

  /* box for Lower Inset EDITTING */
  Boxes[NBoxes].Viewport.Left   = 0.75; Boxes[NBoxes].Viewport.Right = 1.0;
  Boxes[NBoxes].Viewport.Bottom = 0.33; Boxes[NBoxes].Viewport.Top   = 0.58;
  Boxes[NBoxes].Window = Window;
  Boxes[NBoxes].ButtonClick = NoAction;
  LowerInsetBox = NBoxes;
  NBoxes++;

  /* Button Boxes - with icons */

  /* RULER for Distance indication */
  Boxes[NBoxes].Viewport.Left   =  0.01; Boxes[NBoxes].Viewport.Right = 0.07;
  Boxes[NBoxes].Viewport.Bottom =  0.92; Boxes[NBoxes].Viewport.Top   = 0.98;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 1.0;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 1.0;
  sgpSetViewport(Boxes[NBoxes].Viewport);
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground);
  {
    sgpPointType P[4], ScreenPoint;
    P[0].x = 0.1;  P[0].y = 0.4;
    P[1].x = 0.9;  P[1].y = 0.4;
    P[2].x = 0.9;  P[2].y = 0.6;
    P[3].x = 0.1;  P[3].y = 0.6;
    sgpLine(P[0], P[1]);
    sgpLine(P[1], P[2]);
    sgpLine(P[2], P[3]);
    sgpLine(P[3], P[0]);
    sgpWorldToScreen(P[1], &ScreenPoint);
    RulerData.MaxX = ScreenPoint.x;
    sgpWorldToScreen(P[0], &ScreenPoint);
    RulerData.MinX = ScreenPoint.x;

    P[0].x = 0.5; P[0].y = 0.4; P[1].x = 0.5; P[1].y = 0.5;
    sgpLine(P[0], P[1]);
    P[0].x = 0.3; P[1].x = 0.3; P[1].y = 0.45;
    sgpLine(P[0], P[1]);
    P[0].x = 0.7; P[1].x = 0.7; 
    sgpLine(P[0], P[1]);
    P[0].x = 0.2; P[1].x = 0.2; P[1].y = 0.425;
    sgpLine(P[0], P[1]);
    P[0].x = 0.4; P[1].x = 0.4;
    sgpLine(P[0], P[1]);
    P[0].x = 0.6; P[1].x = 0.6;
    sgpLine(P[0], P[1]);
    P[0].x = 0.8; P[1].x = 0.8;
    sgpLine(P[0], P[1]);

    P[0].x = 0.35; P[0].y = 0.39;
    P[1].x = 0.35; P[1].y = 0.2;
    sgpLine(P[0], P[1]);
    sgpWorldToScreen(P[0], &ScreenPoint);
    RulerData.OriginalX = ScreenPoint.x;
    RulerData.CurrentArrow = P[0];
    P[1].x = 0.3; P[1].y = 0.325;
    sgpLine(P[0], P[1]);
    P[1].x = 0.4;
    sgpLine(P[0], P[1]);
  } 
  Boxes[NBoxes].ButtonClick = DistanceChange;
  NBoxes++;

  /* FocalLength indicator */
  Boxes[NBoxes].Viewport.Left   =  0.09; Boxes[NBoxes].Viewport.Right = 0.15;
  Boxes[NBoxes].Viewport.Bottom =  0.92; Boxes[NBoxes].Viewport.Top   = 0.98;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 1.0;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 1.0;
  sgpSetViewport(Boxes[NBoxes].Viewport);
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground);
  {
    sgpPointType P[4], ScreenPoint;
    P[0].x = 0.1;  P[0].y = 0.4;
    P[1].x = 0.1;  P[1].y = 0.9;
    P[2].x = 0.9;  P[2].y = 0.4;
    P[3].x = 0.9;  P[3].y = 0.9;
    sgpLine(P[0], P[1]);
    sgpLine(P[1], P[2]);
    sgpLine(P[0], P[3]);
    sgpWorldToScreen(P[2], &ScreenPoint);
    FLengthData.MaxX = ScreenPoint.x;
    sgpWorldToScreen(P[0], &ScreenPoint);
    FLengthData.MinX = ScreenPoint.x;

    P[0].x = 0.5; P[0].y = 0.3;
    P[1].x = 0.5; P[1].y = 0.1;
    sgpLine(P[0], P[1]);
    sgpWorldToScreen(P[0], &ScreenPoint);
    FLengthData.OriginalX = ScreenPoint.x;
    FLengthData.CurrentArrow = P[0];
    P[1].x = 0.55; P[1].y = 0.225;
    sgpLine(P[0], P[1]);
    P[1].x = 0.45;
    sgpLine(P[0], P[1]);
  } 
  Boxes[NBoxes].ButtonClick = FocalLengthChange;
  NBoxes++;



  /* Pencil for redraw */
  Boxes[NBoxes].Viewport.Left   =  0.17; Boxes[NBoxes].Viewport.Right = 0.23;
  Boxes[NBoxes].Viewport.Bottom =  0.92; Boxes[NBoxes].Viewport.Top   = 0.98;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 1.0;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 1.0;
  sgpSetViewport(Boxes[NBoxes].Viewport);
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P[4];
    P[0].x = 0.3;  P[0].y = 0.2;
    P[1].x = 0.9;  P[1].y = 0.8;
    P[2].x = 0.8;  P[2].y = 0.9;
    P[3].x = 0.2;  P[3].y = 0.3;
    sgpLine(P[0], P[1]);
    sgpLine(P[1], P[2]);
    sgpLine(P[2], P[3]);
    sgpLine(P[3], P[0]);
    P[1].x = 0.1; P[1].y = 0.1;
    sgpLine(P[0], P[1]);
    sgpLine(P[3], P[1]);
    P[0].x = 0.8; P[0].y = 0.7;
    P[1].x = 0.7; P[1].y = 0.8;     
    sgpLine(P[0], P[1]);
  } 
  Boxes[NBoxes].ButtonClick = Redraw;
  NBoxes++;

  /* Quit Option */
  Boxes[NBoxes].Viewport.Left   =  0.25; Boxes[NBoxes].Viewport.Right = 0.31;
  Boxes[NBoxes].Viewport.Bottom =  0.92; Boxes[NBoxes].Viewport.Top = 0.98;  
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport);
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0, P1;
    P0.x = 0.2; P0.y = 0.2;
    sgpCircle(P0, 0.1);
    P1.x = 0.3; P1.y = 0.1;
    sgpLine(P0, P1);
  }
  Boxes[NBoxes].ButtonClick = Quit;
  NBoxes++;

  /* Down Arrow for next section */
  Boxes[NBoxes].Viewport.Left   =  0.01; Boxes[NBoxes].Viewport.Right = 0.07;
  Boxes[NBoxes].Viewport.Bottom =  0.84; Boxes[NBoxes].Viewport.Top = 0.90;  
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport);
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P[4];
    P[0].x = 0.2;  P[0].y = 0.05;
    P[1].x = 0.2;  P[1].y = 0.35;
    P[2].x = 0.15; P[2].y = 0.15;
    P[3].x = 0.25; P[3].y = 0.15;
    sgpLine(P[0], P[1]);
    sgpLine(P[0], P[2]);
    sgpLine(P[0], P[3]);
  }
  NextBox = NBoxes;
  Boxes[NBoxes].ButtonClick = NextSection;
  NBoxes++;

  /* Up Arrow for previous section */
  Boxes[NBoxes].Viewport.Left   =  0.09; Boxes[NBoxes].Viewport.Right = 0.15;
  Boxes[NBoxes].Viewport.Bottom =  0.84; Boxes[NBoxes].Viewport.Top   = 0.90;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P[4];
    P[0].x = 0.2;  P[0].y = 0.05;
    P[1].x = 0.2;  P[1].y = 0.35;
    P[2].x = 0.15; P[2].y = 0.25;
    P[3].x = 0.25; P[3].y = 0.25;
    sgpLine(P[0], P[1]);
    sgpLine(P[1], P[2]);
    sgpLine(P[1], P[3]);
  }
  PrevBox = NBoxes;
  Boxes[NBoxes].ButtonClick = PreviousSection;
  NBoxes++;

 /* window for Delete a Point */
  Boxes[NBoxes].Viewport.Left   =  0.01; Boxes[NBoxes].Viewport.Right = 0.07;
  Boxes[NBoxes].Viewport.Bottom =  0.76; Boxes[NBoxes].Viewport.Top   = 0.82;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0;
    P0.x = 0.1; P0.y = 0.2;
    sgpDisc(P0, 0.025);
    P0.x = 0.2; P0.y = 0.3;
    sgpDisc(P0, 0.025);
    P0.x = 0.3; P0.y = 0.2;
    sgpCircle(P0, 0.040);
  }
  Boxes[NBoxes].ButtonClick = DeletePointOrAdjacency;
  DeletePointBox = NBoxes;
  NBoxes++;

 /* window for ADD a Point */
  Boxes[NBoxes].Viewport.Left   =  0.09; Boxes[NBoxes].Viewport.Right = 0.15;
  Boxes[NBoxes].Viewport.Bottom =  0.76; Boxes[NBoxes].Viewport.Top   = 0.82;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0;
    P0.x = 0.1; P0.y = 0.2;
    sgpDisc(P0, 0.025);
    P0.x = 0.2; P0.y = 0.3;
    sgpDisc(P0, 0.025);
    P0.x = 0.3; P0.y = 0.2;
    sgpDisc(P0, 0.025);
    sgpCircle(P0, 0.040);
  }
  Boxes[NBoxes].ButtonClick = AddPoint;
  NBoxes++;

 /* window for MOVE a Point */
  Boxes[NBoxes].Viewport.Left   =  0.17; Boxes[NBoxes].Viewport.Right = 0.23;
  Boxes[NBoxes].Viewport.Bottom =  0.76; Boxes[NBoxes].Viewport.Top   = 0.82;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0, P1;
    P0.x = 0.1; P0.y = 0.3;
    sgpDisc(P0, 0.025);
    P0.x = 0.3; P0.y = 0.1;
    sgpDisc(P0, 0.025);
    P0.x = 0.15; P0.y = 0.25;
    P1.x = 0.25; P1.y = 0.15;
    sgpLine(P0, P1);
    P0.x = 0.2; P0.y = 0.15;
    sgpLine(P0, P1);
    P0.x = 0.25; P1.y = 0.20;
    sgpLine(P0, P1);
  }
  Boxes[NBoxes].ButtonClick = MovePoint;
  NBoxes++;

 /* window for Delete an Adjacency */
  Boxes[NBoxes].Viewport.Left   =  0.01; Boxes[NBoxes].Viewport.Right = 0.07;
  Boxes[NBoxes].Viewport.Bottom =  0.68; Boxes[NBoxes].Viewport.Top   = 0.74;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0, P1;
    P0.x = 0.1; P0.y = 0.3;
    sgpDisc(P0, 0.025);
    P1.x = 0.3; P1.y = 0.3;
    sgpDisc(P1, 0.025);
    sgpLine(P0, P1);

    P0.y = 0.1;
    sgpDisc(P0, 0.025);
    P1.y = 0.1;
    sgpDisc(P1, 0.025);

    P0.x = 0.2; P0.y = 0.275;
    P1.x = 0.2; P1.y = 0.125;
    sgpLine(P0, P1);
    P0.x = 0.175; P0.y = 0.175;
    sgpLine(P0, P1);
    P0.x = 0.225;
    sgpLine(P0, P1);
  }
  Boxes[NBoxes].ButtonClick = DeletePointOrAdjacency;
  NBoxes++;

 /* window for ADD a Previous Adjacency */
  Boxes[NBoxes].Viewport.Left   =  0.09; Boxes[NBoxes].Viewport.Right = 0.15;
  Boxes[NBoxes].Viewport.Bottom =  0.68; Boxes[NBoxes].Viewport.Top   = 0.74;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0, P1;
    P0.x = 0.05; P0.y = 0.3;
    sgpDisc(P0, 0.025);
    P1.x = 0.25; P1.y = 0.3;
    sgpDisc(P1, 0.025);

    P0.y = 0.1;
    sgpDisc(P0, 0.025);
    P1.y = 0.1;
    sgpDisc(P1, 0.025);
    sgpLine(P0, P1);

    P0.x = 0.15; P0.y = 0.275;
    P1.x = 0.15; P1.y = 0.125;
    sgpLine(P0, P1);
    P0.x = 0.125; P0.y = 0.175;
    sgpLine(P0, P1);
    P0.x = 0.175;
    sgpLine(P0, P1);

    P0.x = 0.325; P0.y = 0.35;
    P1.x = 0.325; P1.y = 0.20;
    sgpLine(P0, P1);
    P1.x = 0.3; P1.y = 0.3;
    sgpLine(P0, P1);
    P1.x = 0.35;  
    sgpLine(P0, P1);
  }
  Boxes[NBoxes].ButtonClick = AddPrevAdjacency;
  NBoxes++;

 /* window for ADD a Next Adjacency */
  Boxes[NBoxes].Viewport.Left   =  0.17; Boxes[NBoxes].Viewport.Right = 0.23;
  Boxes[NBoxes].Viewport.Bottom =  0.68; Boxes[NBoxes].Viewport.Top   = 0.74;
  Boxes[NBoxes].Window.Left     =  0.0; Boxes[NBoxes].Window.Right   = 0.40;
  Boxes[NBoxes].Window.Bottom   =  0.0; Boxes[NBoxes].Window.Top     = 0.40;
  sgpSetViewport(Boxes[NBoxes].Viewport); 
  sgpSetWindow(Boxes[NBoxes].Window);
  sgpGrayLevel(Foreground); 
  {
    sgpPointType P0, P1;
    P0.x = 0.05; P0.y = 0.3;
    sgpDisc(P0, 0.025);
    P1.x = 0.25; P1.y = 0.3;
    sgpDisc(P1, 0.025);

    P0.y = 0.1;
    sgpDisc(P0, 0.025);
    P1.y = 0.1;
    sgpDisc(P1, 0.025);
    sgpLine(P0, P1);

    P0.x = 0.15; P0.y = 0.275;
    P1.x = 0.15; P1.y = 0.125;
    sgpLine(P0, P1);
    P0.x = 0.125; P0.y = 0.175;
    sgpLine(P0, P1);
    P0.x = 0.175;
    sgpLine(P0, P1);

    P0.x = 0.325; P0.y = 0.05;
    P1.x = 0.325; P1.y = 0.20;
    sgpLine(P0, P1);
    P1.x = 0.3; P1.y = 0.1;
    sgpLine(P0, P1);
    P1.x = 0.35;  
    sgpLine(P0, P1);

  }
  Boxes[NBoxes].ButtonClick = AddNextAdjacency;



 }  /* end of SETUP */



int main(argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int button;
  char SystemCommand[85];
  char TempString[40];
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'c':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Center.x = atof(argv[ArgsParsed++]);
         Center.y = atof(argv[ArgsParsed++]);
         Center.z = atof(argv[ArgsParsed++]);
         CENTERSET = 1;
         break;
        case 'd':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         ViewingDistance = atof(argv[ArgsParsed++]);
         OriginalDistance = ViewingDistance;
         VIEWSET = 1;
      break;
        case 'w':
        if ((argc-ArgsParsed)<1) {usage(); exit (-1); }
         Width = atof(argv[ArgsParsed++]);
         Width = Width / 2.0;
         DiscSize = Width / 80.0;
         WIDTHSET = 1;
         if (!VIEWSET)
          {
           ViewingDistance = Width * 4.0;
           OriginalDistance = ViewingDistance;
           VIEWSET = 1;
          }
         break; 
        case 'i':
         if ((argc-ArgsParsed)<1) {usage(); exit (-1); }
         DataFile = argv[ArgsParsed++];
         break;
        case 'o':
         if ((argc-ArgsParsed)<1) {usage(); exit (-1); }
         OutputFile = argv[ArgsParsed++];
      break;
        case 'D':
      DEBUG = 1;
      break;
        case 'v':
         VERBOSE = 1;
         break;
        case 'h': { usage(); exit (-1); }
        default : ;
       }
     }
    else { usage(); exit (-1); }
   } 

  Lens3D(FocalLength); SetHither(FocalLength);

  strcat(SystemCommand, "cp ");
  strcpy(TempString, DataFile);
  strcat(SystemCommand, TempString);
  strcpy(TempString, OutputFile);
  strcat(SystemCommand, " ");
  strcat(SystemCommand, TempString);

  system(SystemCommand);   /*"cp ContourData NewContourData" */

  NewFilePtr = fopen(OutputFile, "r");
  if ((FILE *)0 == NewFilePtr)
   {
    fprintf(stderr,"Cannot open %s\n",OutputFile);
    exit(-1);
   }
  TempFilePtr = fopen(TempFile, "w");
  if ((FILE *)0 == TempFilePtr)
   {
    fprintf(stderr,"Cannot open %s\n",TempFile);
    exit(-1);
   }

  BuildSectionRecords(NewFilePtr); /* compute default center and width */
  SetUp();  /* sets the Window, Viewport, draws button icons, etc. */
  DrawViews(NewFilePtr);
  CurrentSection = 0;
  DrawCurrentSection(NewFilePtr, CurrentSection);

  /*
     wait for a button click, deal with it, do it again, and again, and...
   */
  for(;;)
   {
    int  box;
    sgpPointType Screen;

    sgpPick(&Points[NPoints],&button); if (!button) continue;

    for(;button;) sgpPick(&Points[NPoints],&button);  /* debounce */
   
    /*
      note that the action occurs when the button is RELEASED
     */
    sgpWorldToScreen(Points[NPoints],&Screen);
    for(box=0;box<=NBoxes;box++)
     {
      if (  (Boxes[box].Viewport.Left  < Screen.x)
                                  && (Screen.x < Boxes[box].Viewport.Right)
         &&(Boxes[box].Viewport.Bottom<Screen.y)
                                  && (Screen.y < Boxes[box].Viewport.Top  ))
       {
        Boxes[box].ButtonClick(box,Screen);
        break;
       }
     } 
   }
  /* warning - statement not reached... */
  fprintf(stderr,"-30-\n");
  sgpQuit();
  exit(0);
 }
