/*
  File: Torus.c
  Authors: Jinling Lee (Bill),
           K.R. Sloan   
  Last Modified: 18 April 1990
  Purpose: Draw GRID^2 supertori, with different e1,e2
           Note - all measurements are in meters.  
 */

#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"
#include "Triangles.h"
#include "Render.h"
#include "Scanline.h"

double atof();
double pow();

#define EPSILON (0.0000000000001)
#define GRID (4)
#define MAXTRIANGLES (16376)
#define PRECISION (30)

int VERBOSE = 0;    /* exported for the real renderers to see */
static int DEBUG = 0;
static Preview = 0;
static Hidden = 1;
static Quality = 0;
static sgpColorType InteriorColor = { 1.0, 1.0, 1.0 };
static sgpColorType OutlineColor  = { 0.0, 0.0, 0.0 };

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-v][-p][-P][-D][-Q][-e x y z][-o x y z][-u x y z][-f f][-x x]\n",
                 RoutineName);
 }

static int t=0;
static TriangleType3D T[MAXTRIANGLES];  /* that should be enough... */  

double SpaceSample(u,e)
 double u;
 {
  if (u < -M_PI) return(u);
  if (u < -(M_PI/2.0))
   return(pow(2.0*(u+    M_PI  )/M_PI,e) -     M_PI  );
  if (u < 0.0)
   return(pow(2.0*(u+(M_PI/2.0))/M_PI,e) - (M_PI/2.0));
  if (u < (M_PI/2.0))
   return(pow(2.0*(u           )/M_PI,e)             );
  if (u < M_PI)
   return(pow(2.0*(u-(M_PI/2.0))/M_PI,e) + (M_PI/2.0));
  else return(u);
 }

double Raise(x,y)
 double x,y;
 {
 if (fabs(x)<EPSILON) return(0.0);
 if (x < 0.0) return( - pow(-x,y));
 else         return(   pow( x,y));
 }

void DrawTorus(Center, a1,a2,a3,a4, e1,e2, Precision,
                Diffuse, Specular, Specularity,
                OpticalAxis)
 PointType3D Center;
 double a1,a2,a3,a4,e1,e2;
 int Precision;
 sgpColorType Diffuse, Specular;
 double Specularity;
 VectorType3D OpticalAxis;
 {
  int i, j;
  PointType3D Pa, Pb, Pc, Pd;
  VectorType3D Na, Nb, Nc, Nd;
  double u0,u1,v0,v1,e,newe;
  double Cu0, Cu1, Cv0, Cv1;
  double Su0, Su1, Sv0, Sv1;
  double Cu0n, Cu1n, Cv0n, Cv1n;
  double Su0n, Su1n, Sv0n, Sv1n;

  fprintf(stderr,"DrawTorus: %e,%e,%e,%e\n\t\t %e,%e,\n", a1,a2,a3,a4,e1,e2);

  /* 
    Divide the Torus into triangles.

   */
  e = M_PI/(double)Precision;
  for(i=0; i<2*Precision; i++)
   {
    /* u values for this quadrilateral */
    u0 = SpaceSample((i  -Precision)*e,e1);
    u1 = SpaceSample((i+1-Precision)*e,e1);

    /* cosines and sines */
    Cu0 = cos(u0);     Su0 = sin(u0);
    Cu1 = cos(u1);     Su1 = sin(u1);

    /* raised to the powers used in the normal calculation */
    Cu0n = Raise(Cu0,2.0-e1);    Su0n = Raise(Su0,2.0-e1);
    Cu1n = Raise(Cu1,2.0-e1);    Su1n = Raise(Su1,2.0-e1);

    /* raised to the powers used in the position calculation */
    Cu0 = Raise(Cu0,e1);    Su0 = Raise(Su0,e1);
    Cu1 = Raise(Cu1,e1);    Su1 = Raise(Su1,e1);

    for(j=0; j<2*Precision; j++)
     {
      /* v values for this quadrilateral */
      v0 = SpaceSample((j  -Precision)*e,e2);
      v1 = SpaceSample((j+1-Precision)*e,e2);

      /* cosines and sines */
      Cv0 = cos(v0);     Sv0 = sin(v0);
      Cv1 = cos(v1);     Sv1 = sin(v1);

      /* raised to the powers used in the normal calculation */
      Cv0n = Raise(Cv0,2.0-e2);    Sv0n = Raise(Sv0,2.0-e2);
      Cv1n = Raise(Cv1,2.0-e2);    Sv1n = Raise(Sv1,2.0-e2);

      /* raised to the powers used in the position calculation */
      Cv0 = Raise(Cv0,e2);    Sv0 = Raise(Sv0,e2);
      Cv1 = Raise(Cv1,e2);    Sv1 = Raise(Sv1,e2);
     
      /* normals */

      Na.dx = (1.0/a1)*Cu0n*Cv0n; 
      Na.dy = (1.0/a2)*Cu0n*Sv0n;
      Na.dz = (1.0/a3)*Su0n;
      Normalize3D(&Na);

      Nb.dx = (1.0/a1)*Cu1n*Cv0n;
      Nb.dy = (1.0/a2)*Cu1n*Sv0n;
      Nb.dz = (1.0/a3)*Su1n;
      Normalize3D(&Nb);

      Nc.dx = (1.0/a1)*Cu1n*Cv1n;
      Nc.dy = (1.0/a2)*Cu1n*Sv1n;
      Nc.dz = (1.0/a3)*Su1n;
      Normalize3D(&Nc);

      Nd.dx = (1.0/a1)*Cu0n*Cv1n;
      Nd.dy = (1.0/a2)*Cu0n*Sv1n;
      Nd.dz = (1.0/a3)*Su0n;
      Normalize3D(&Nd);

      /* points */

      Pa.x = Center.x + a1*(a4+Cu0)*Cv0;
      Pa.y = Center.y + a2*(a4+Cu0)*Sv0;
      Pa.z = Center.z + a3*Su0;

      Pb.x = Center.x + a1*(a4+Cu1)*Cv0;
      Pb.y = Center.y + a2*(a4+Cu1)*Sv0;
      Pb.z = Center.z + a3*Su1;

      Pc.x = Center.x + a1*(a4+Cu1)*Cv1;
      Pc.y = Center.y + a2*(a4+Cu1)*Sv1;
      Pc.z = Center.z + a3*Su1;

      Pd.x = Center.x + a1*(a4+Cu0)*Cv1;
      Pd.y = Center.y + a2*(a4+Cu0)*Sv1;
      Pd.z = Center.z + a3*Su0;

      /* draw the triangles */

      T[t].Vertex[0] = Pa; T[t].Vertex[1] = Pb; T[t].Vertex[2] = Pc;
      T[t].Normal[0] = Na; T[t].Normal[1] = Nb; T[t].Normal[2] = Nc;
      T[t].Diffuse[0] = Diffuse;
      T[t].Diffuse[1] = Diffuse;
      T[t].Diffuse[2] = Diffuse;
      T[t].Specular[0] = Specular;
      T[t].Specular[1] = Specular;
      T[t].Specular[2] = Specular;
      T[t].Specularity[0] = Specularity;
      T[t].Specularity[1] = Specularity;
      T[t].Specularity[2] = Specularity;
      if (Hidden) t++;
      else
       {
        sgpColor(OutlineColor);
        Line3D(T[t].Vertex[0], T[t].Vertex[1]);
        Line3D(T[t].Vertex[1], T[t].Vertex[2]);
        Line3D(T[t].Vertex[2], T[t].Vertex[0]);
       }
      T[t].Vertex[0] = Pa; T[t].Vertex[1] = Pc; T[t].Vertex[2] = Pd;
      T[t].Normal[0] = Na; T[t].Normal[1] = Nc; T[t].Normal[2] = Nd;
      T[t].Diffuse[0] = Diffuse;
      T[t].Diffuse[1] = Diffuse;
      T[t].Diffuse[2] = Diffuse;
      T[t].Specular[0] = Specular;
      T[t].Specular[1] = Specular;
      T[t].Specular[2] = Specular;
      T[t].Specularity[0] = Specularity;
      T[t].Specularity[1] = Specularity;
      T[t].Specularity[2] = Specularity;
      if (Hidden) t++;
      else
       {
        sgpColor(OutlineColor);
        Line3D(T[t].Vertex[0], T[t].Vertex[1]);
        Line3D(T[t].Vertex[1], T[t].Vertex[2]);
        Line3D(T[t].Vertex[2], T[t].Vertex[0]);
       }
     }
   }
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  static int ArgsParsed = 0;
  static PointType3D  EyePoint    = {  2.0, -3.0,  4.0 };
  static VectorType3D OpticalAxis = { -2.0,  3.0, -4.0 };
  static VectorType3D Up          = {  0.0,  1.0,  0.0 };
  static double FocalLength = 0.0040;
  static sgpRectangleType Window   = {-0.0015, -0.0012, 0.0015, 0.0012};
  static sgpRectangleType Viewport = {-0.125,   0.00,   1.125,  1.00};  
  static double L = 100.0;      /* half width of light cube */
  double Exposure = 0.5;
  VectorType3D Right, TrueUp;
  VectorType3D KeyDirection;
  PointType3D LightPosition;
  sgpColorType Color;
  sgpColorType Ambient, Diffuse, Specular;
  double Specularity;
  PointType3D Center;
  double a1,a2,a3,a4,e1,e2;
  int Precision;
  int i,j;
  sgpRectangleType subViewport;

  RoutineName = argv[ArgsParsed++];


  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'e':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         EyePoint.x = atof(argv[ArgsParsed++]);
         EyePoint.y = atof(argv[ArgsParsed++]);
         EyePoint.z = atof(argv[ArgsParsed++]);
         break;
        case 'o':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         OpticalAxis.dx = atof(argv[ArgsParsed++]);
         OpticalAxis.dy = atof(argv[ArgsParsed++]);
         OpticalAxis.dz = atof(argv[ArgsParsed++]);
         break;
        case 'u':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Up.dx = atof(argv[ArgsParsed++]);
         Up.dy = atof(argv[ArgsParsed++]);
         Up.dz = atof(argv[ArgsParsed++]);
         break;
        case 'f':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         FocalLength = atof(argv[ArgsParsed++]);
         break;
        case 'x':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         Exposure = atof(argv[ArgsParsed++]);
         break;
        case 'p':
         Preview = 1;
         break;
        case 'P':
         Hidden = 0;
         Preview = 1;
         break;
        case 'Q':
         Quality = 1;
         Hidden = 1;
         Preview = 0;
         break;
        case 'D':
         DEBUG = 1;
         break;
        case 'v':
         VERBOSE = 1;
         break;
        case 'h': 
        default:
         usage(); exit(-1);
       }
     }
    else { usage(); exit (-1); }
   }   

  while (ArgsParsed < argc) { usage(); exit (-1); }

  sgpInit(ColorDisplay);
  sgpSetWindow(Window);
  sgpSetViewport(Viewport);

  Camera3D(EyePoint, OpticalAxis, Up);
  Lens3D(FocalLength);  SetHither(FocalLength);


  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  if (VERBOSE)
   {
    fprintf(stderr,"%s: EyePoint = { %f, %f, %f }\n", RoutineName,
                        EyePoint.x,
                        EyePoint.y,
                        EyePoint.z);
    fprintf(stderr,"%s: OpticalAxis = { %f, %f, %f }\n", RoutineName,
                        OpticalAxis.dx,
                        OpticalAxis.dy,
                        OpticalAxis.dz);
    fprintf(stderr,"%s: Up = { %f, %f, %f }\n", RoutineName,
                        Up.dx,
                        Up.dy,
                        Up.dz);
    fprintf(stderr,"%s: FocalLength = %f\n", RoutineName, FocalLength);
   }
  if (Preview)
   {
    if (VERBOSE)
     {
      fprintf(stderr,"%s: Previewing\n", RoutineName);
      fprintf(stderr,"%s: InteriorColor = { %f, %f, %f }\n", RoutineName,
                          InteriorColor.r, InteriorColor.g, InteriorColor.b);
      fprintf(stderr,"%s: OutlineColor = { %f, %f, %f }\n", RoutineName,
                          OutlineColor.r, OutlineColor.g, OutlineColor.b);
     }
    RenderMode(Preview, InteriorColor, OutlineColor);
    sgpColor(InteriorColor); sgpClearScreen();
   }
  else
   {
    Ambient.r = 0.2; Ambient.g = 0.2; Ambient.b = 0.5;
    SetAmbient(Ambient);

    LightPosition.x=  L; LightPosition.y= -L; LightPosition.z= -L;
    Color.r = 1.0; Color.g = 0.0; Color.b = 0.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x= -L; LightPosition.y=  L; LightPosition.z= -L;
    Color.r = 0.0; Color.g = 1.0; Color.b = 0.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x= -L; LightPosition.y= -L; LightPosition.z=  L;
    Color.r = 0.0; Color.g = 0.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x=  L; LightPosition.y=  L; LightPosition.z= -L;
    Color.r = 1.0; Color.g = 1.0; Color.b = 0.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x=  L; LightPosition.y= -L; LightPosition.z=  L;
    Color.r = 1.0; Color.g = 0.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);
    LightPosition.x= -L; LightPosition.y=  L; LightPosition.z=  L;
    Color.r = 0.0; Color.g = 1.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x = L; LightPosition.y = L; LightPosition.z=  L;
    Color.r = 1.0; Color.g = 1.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    SetExposure(Exposure);
    if (VERBOSE)
     fprintf(stderr,"%s: Exposure = %f\n", RoutineName, Exposure);

   }

  Diffuse.r  = 0.5;  Diffuse.g  = 0.5;  Diffuse.b  = 0.5;
  Specular.r = 0.5;  Specular.g = 0.5;  Specular.b = 0.5;
  Specularity = 6.0;

  Precision = PRECISION;
  while ((Precision*Precision*8)>MAXTRIANGLES) Precision--;

  for(j=0;j<GRID;j++)
   {
    subViewport.Bottom  = Viewport.Bottom
                         +(double)   j   *(Viewport.Top-Viewport.Bottom)
                                         /(double)GRID;
    subViewport.Top     = Viewport.Bottom
                         +(double)(j+1.0)*(Viewport.Top-Viewport.Bottom)
                                         /(double)GRID;
    e1 = 0.3 + (3.0-0.3)*(double)j/ (double)(GRID-1.0);
    for(i=0;i<GRID;i++)
     {
      subViewport.Left  = Viewport.Left
                         +(double)   i   *(Viewport.Right-Viewport.Left)
                                         /(double)GRID;
      subViewport.Right = Viewport.Left
                         +(double)(i+1.0)*(Viewport.Right-Viewport.Left)
                                         /(double)GRID;
      sgpSetViewport(subViewport);      
      e2 = 0.3 + (3.0-0.3)*(double)i/ (double)(GRID-1.0);
      Center.x = 0.0; Center.y = 0.0; Center.z = 0.0;
      a1 = 0.3; a2 = 0.3; a3 = 0.3; a4 = 10.0/3.0;

      t=0;
      DrawTorus(Center, a1,a2,a3,a4,e1,e2, Precision, 
                 Diffuse, Specular, Specularity,
                 OpticalAxis);

      if (Hidden)
       if (Quality)
        {
         if (VERBOSE)
          fprintf(stderr,"%s: calling ScanTriangles(%d, T, %d)\n",
                         RoutineName,t,MAXTRIANGLES);
         ScanTriangles(t,T,MAXTRIANGLES);
         if (VERBOSE)
          fprintf(stderr,"%s: ScanTriangles returns\n",RoutineName);
        }
       else
        {
         if (VERBOSE)
          fprintf(stderr,"%s: calling RenderTriangles(%d, T, %d)\n",
                        RoutineName,t,MAXTRIANGLES);
         RenderTriangles(t,T,MAXTRIANGLES);

         if (VERBOSE) fprintf(stderr,
                             "%s: RenderTriangles returns\n",RoutineName);
        }
     }
   }
  sgpQuit();
  if (VERBOSE) fprintf(stderr,"-30-\n");
  exit(0);
 }
