/*
   File: framebuffer.c
   Authors: Tony DeRose,
            K.R. Sloan
   Last Modified: 26 April 1991
   Purpose: simple framebuffer abstraction, using X11
            this is a color display, with the origin at the lower left 

            rectangles are specified as: Left, Top, Right, Bottom

            It is assumed (BUT NOT CHECKED) that:
                     Left   <= Right
                     Bottom <= Top

            This version adds mouse tracking and button pushes.

            This version attempts to handle binary displays as well
            as 8-bit color mapped displays.  

 */

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Form.h>
#include <X11/StringDefs.h>
#include <X11/Box.h>
#include <X11/Command.h>
#include <X11/cursorfont.h>

#include "framebuffer.h"

int DitherPattern[8][8] = {{ 0, 48, 12, 60,  3, 51, 15, 63},
                           {32, 16, 44, 28, 35, 19, 47, 31},
                           { 8, 56,  4, 52, 11, 59,  7, 55},
                           {40, 24, 36, 20, 43, 27, 39, 23},
                           { 2, 40, 14, 62,  1, 49, 13, 61},
                           {34, 18, 46, 30, 33, 17, 45, 29},
                           {10, 58,  6, 54,  9, 57,  5, 53},
                           {42, 26, 38, 22, 41, 25, 37, 21} };

XStandardColormap FB_cmap;
int FB_depth;
int FB_ncolors;
unsigned long FB_pixels[256];
int FB_BINARY = 0;
unsigned long FB_BLACK;
unsigned long FB_WHITE;
int FB_grayscale = 0;
int FB_mycolormap = 0;
XColor FB_colors[256];

XWMHints xwmh =
 {
  (InputHint|StateHint),    /* flags */
  False,                    /* input */
  NormalState,              /* initial state */
  0,                        /* icon pixmap */
  0,                        /* icon window */
  0, 0,                     /* icon location */
  0,                        /* icon mask  */
  0                         /* Window group */      
 };

/* X-Framebuffer definitions */
#define fb_NAME "FrameBuffer"

/* Variables local to the library */
int WindowOpened = FALSE;         /* TRUE if the window is opened */
static Display *display;          /* The display to post the window to */
static int screen;
static Visual *visual;
static Window window, parent;     /* Window descriptors */
static GC gc;                     /* GC to draw with */        
static XEvent event;              /* Event received */
static XSizeHints xsh;            /* Size hints for window manager */
static char *geomspec;            /* Window geometry string */
static XSetWindowAttributes xswa; /* Temporary Set Window Attribute */
static XWindowAttributes pxwa;    /* parent's attributes */
static XWindowAttributes xwa;     /* the returned truth */ 
static Cursor NoButtons, Buttons; /* cursors, eh? */
static unsigned long fg, bg, bd;  /* pixel values */

/* Exported variables */
/*
  fb_error_message points at an explanation of your error
 */
char *fb_error_message = (char *)0;
  
/*
  Create the frame buffer window, display it on the screen,
  and return the range of its device coordinates.

  This version sets up a colormap (or uses the standard one...)
*/

int InitializeColormap(FB_grayscale)
 int *FB_grayscale;
 {
  int i,r,g,b;

   /* Get the appropriate standard colormap */

  FB_mycolormap = 0;
  FB_cmap.colormap = 0;

  XGetStandardColormap(display, DefaultRootWindow(display),
                       &FB_cmap, 
                       *FB_grayscale ? XA_RGB_GRAY_MAP : XA_RGB_DEFAULT_MAP);

  /*
     If the standard map IS defined, make a guess about the
     number of allocated cells (DOES ANYONE KNOW HOW TO DO THIS RIGHT?)
     and fetch the colors, so that devDecodeColor can figure it out.
   */
  if ((Colormap)0 != FB_cmap.colormap) 
   {
    if      (FB_depth < 4) { *FB_grayscale = TRUE; FB_ncolors = 4;}
    else if (FB_depth < 8) { *FB_grayscale = TRUE; FB_ncolors = 16;}
    else             FB_ncolors = (*FB_grayscale ? FB_cmap.red_max+1 : 216);

    for(i=0;i<FB_ncolors;i++)
     {
      FB_colors[i].pixel =  FB_cmap.base_pixel + i; 
     }
    XQueryColors(display,FB_cmap.colormap,FB_colors,FB_ncolors);
    XSync(display,False);
   }

  /* 
    If the standard map is not defined, define it.
    See StandardMaps.c - where this code came from
   */
  if ((Colormap)0 == FB_cmap.colormap) 
   {
    Colormap cm;
    Status cmstatus;
    int oFB_ncolors;

    if (FB_depth < 4)
     {
      *FB_grayscale = TRUE;
      FB_ncolors = 2;
     }
    else if (FB_depth < 8)
     {
      *FB_grayscale = TRUE;
      FB_ncolors = 16;
     }
    else
     {
      FB_ncolors =  (*FB_grayscale) ? 256 : 216;
     }

    cm = DefaultColormap(display, screen);
    oFB_ncolors = FB_ncolors;

    do 
     {
      cmstatus = XAllocColorCells(display, cm,
                                  TRUE,                   /* contiguous     */
                                  (unsigned long *)0, 0,  /* no plane masks */
                                  FB_pixels, FB_ncolors); /* pixel values   */
      if ((Status)0 == cmstatus) 
       {
        if (*FB_grayscale) FB_ncolors--; /* try for fewer */
        else break; /* insist on all 216 for color */
       } 
     } while ((FB_ncolors > 1) && ((Status)0 == cmstatus));

    if ((Status)0 == cmstatus) 
     {
      FB_ncolors = oFB_ncolors;
      fprintf(stderr, "framebuffer: Unable to allocate colormap cells.\n" );
      fprintf(stderr, "framebuffer: Hope you like this colormap.\n" );
      FB_pixels[0] = 3;
     }

    FB_cmap.colormap = cm;
    FB_cmap.base_pixel = FB_pixels[0];

    /* Load the color map */
    if (*FB_grayscale) 
     {
      for(i=0; i<FB_ncolors; i++) 
       {
        FB_colors[i].pixel = FB_pixels[i];
        FB_colors[i].red = FB_colors[i].green = FB_colors[i].blue
                      = (i * 65535) / FB_ncolors;
        FB_colors[i].flags = DoRed | DoGreen | DoBlue;
       }
      FB_cmap.red_max =FB_ncolors-1; FB_cmap.green_max =0; FB_cmap.blue_max =0;
      FB_cmap.red_mult=1;            FB_cmap.green_mult=0; FB_cmap.blue_mult=0;
     }
    else 
     {
      i = 0;
      for(b=0; b<6; b++)
       for(g=0; g<6; g++)
        for(r=0; r<6; r++) 
         {
          FB_colors[i].pixel = FB_pixels[i+1];
          FB_colors[i].red   = r * 13107; /* 65535 / 5; */
          FB_colors[i].green = g * 13107;
          FB_colors[i].blue  = b * 13107;
          FB_colors[i++].flags = DoRed | DoGreen | DoBlue;
         }
      FB_cmap.red_max =5; FB_cmap.green_max =5; FB_cmap.blue_max = 5;
      FB_cmap.red_mult=1; FB_cmap.green_mult=6; FB_cmap.blue_mult=36;
     } 

     /*
       Make this the standard map, for as long as we're around...
     */
    XStoreColors(display, cm, FB_colors, FB_ncolors);
    XInstallColormap(display, cm);

    FB_mycolormap = 1;
    XChangeProperty(display, DefaultRootWindow(display), 
                    *FB_grayscale ? XA_RGB_GRAY_MAP : XA_RGB_DEFAULT_MAP,
                    XA_RGB_COLOR_MAP, 32, PropModeReplace,
                    (char *) &FB_cmap, 
                    sizeof(FB_cmap)/sizeof(long));
   }

  /* Set the colormap property for the window */
  XSetWindowColormap(display, window, FB_cmap.colormap);
 }

int fb_open(Left, Bottom, Right, Top, color)
 int *Left, *Bottom, *Right, *Top;
 int color;
 {
  FB_grayscale = (color == 0);

  if (WindowOpened) 
   {
    fb_error_message
        = "fb_open: attempt to open the already opened framebuffer\n";
    return(FB_ERROR);
   }
  display = XOpenDisplay((char *)0);  /* use DISPLAY environment variable */
  if ((Display *)0 == display)
   {
    fb_error_message = "fb_open: couldn't open the display";
    return(FB_ERROR);
   }

  screen = DefaultScreen(display);
  visual = DefaultVisual(display, screen);
  gc     = DefaultGC(display, screen);
  parent = RootWindow(display, screen);

  XSync(display, False);

  XGetWindowAttributes(display, parent, &pxwa);
  FB_depth = pxwa.depth;

/* for creating libX1 only - for testing */
/*         FB_depth = 1;                 */
/* take it away - it's ugly...           */

  /*
    provide hints for initial position and size
   */

  xsh.flags = (PPosition | PSize | PMinSize | PMaxSize | PResizeInc);
  xsh.height = 480;
  xsh.width  = 640;
  xsh.x = (pxwa.width  - xsh.width ) / 2;
  xsh.y = (pxwa.height - xsh.height) / 2;
  xsh.min_width = 128;  xsh.min_height = 128;
  xsh.max_width = 1280;  xsh.max_height = 920;
  xsh.width_inc = 16;    xsh.height_inc = 16;
  
  /*
    create two cursors
   */
  NoButtons = XCreateFontCursor(display,XC_crosshair);
  Buttons = XCreateFontCursor(display,XC_circle);

  xswa.cursor = NoButtons;

  /*
    Create the window
   */
  window = XCreateWindow(display, parent,
            xsh.x, xsh.y, xsh.width, xsh.height, pxwa.border_width,
            pxwa.depth, InputOutput, visual,
            CWCursor,
	    &xswa);

  /*
    Set standard properties for the window managers
   */
  XSetStandardProperties(display, window, fb_NAME, fb_NAME, None, "", 0, &xsh);
  XSetWMHints(display, window, &xwmh);




  /*
     this is experimental - we are trying to support:
         8-bit pseudocolor displays
         1-bit binary displays

     anything else is "not understood", and causes an immediate error.
   
     Feel free to fix this - I don't have the time.       
   */
  switch (FB_depth)
   {
    case 1:  FB_BINARY = -1; 
             FB_grayscale = -1;
             FB_ncolors = 64;
             FB_mycolormap = 0;
             FB_BLACK = BlackPixel(display, screen);
             FB_WHITE = WhitePixel(display, screen);
             break;
    case 8:  FB_BINARY = 0; 
             InitializeColormap(&FB_grayscale); 
             break;

    default: fprintf(stderr,
                    "Sorry - we do not handle displays of depth %d\n",
                    FB_depth);
             exit(-1);
   }




  /*
    allow Exposure events    
   */

  XSelectInput(display, window, ExposureMask);
 
  /*
    map window to make it visible
   */

  XMapWindow(display, window);

  /*
    wait for an exposure event
   */
  while(1)
   {
    XNextEvent(display, &event);
    if (  (Expose == event.type) && (0 == event.xexpose.count) )
     {
      while (XCheckTypedEvent(display, Expose, &event)); /* get them all */
      break;
     }
   }
 
  /*
    refuse all events
   */
  XSelectInput(display,window,NoEventMask);
  
  /*
    Find out what size the window ended up being set to 
   */
  if (0 == XGetWindowAttributes(display,window,&xwa))
   {
    fb_error_message = "Couldn't query the frame buffer window.";
    return FB_ERROR;
   }

  /*
    Set the output parameters - notice that we are reporting the
    coordinate system we are exporting - NOT the coordinate system
    provided by X11.  
   */
  *Left = 0;
  *Bottom = 0;
  *Right = xwa.width  - 1;
  *Top   = xwa.height - 1;

  /*
    Everything went well...
   */
  WindowOpened = TRUE;
  return FB_SUCCESS;
 }

/*
  Destroy the frame buffer window.
 */
int fb_close()
 {
  if (!WindowOpened) 
   {
    fb_error_message = 
        "fb_close: attempt to close already closed framebuffer.";
    return FB_ERROR;
   }

  fprintf(stderr,"fb_close: going away soon...\n");
  sleep(120); /* let the user look, then die */
  XDestroyWindow(display,window);
  WindowOpened = FALSE;
  if (FB_mycolormap)
   {
    XGrabServer(display);
    XDeleteProperty(display, DefaultRootWindow(display),
                    FB_grayscale ? XA_RGB_GRAY_MAP : XA_RGB_DEFAULT_MAP);  
    XUngrabServer(display);
   }
  FB_mycolormap = 0;
  FB_cmap.colormap = 0;
  XSync(display,True);
  return FB_SUCCESS;
 }

/*
   Write a pixel in the given color.
 */
int fb_writePixel(x, y, color)
 int x, y;
 FB_COLOR color;
 {
  unsigned long pixel;
  if (!WindowOpened) 
   {
    fb_error_message = "fb_writePixel: window not opened.";
    return FB_ERROR;
   }


  /*
     for binary displays, threshold against a dither pattern

     for color mapped displays, use the color map
   */

  if (FB_BINARY)
   {
    int t;

    t = DitherPattern[x % 8][y % 8];

    pixel = FB_WHITE;
    if ((t > color) || (0 == color)) pixel = FB_BLACK;
    
   }
  else
   {
    pixel = FB_cmap.base_pixel + color;
   }

  XSetForeground(display, gc, pixel);
  XDrawPoint(display, window, gc,
              x, xwa.height-1-y); /* reverse y coord */
  return FB_SUCCESS;
 }

/*
   paint a rectangle in the given color
 */
int fb_writeRectangle(L, B, R, T, color)
 int L, T, R, B;
 FB_COLOR color;
 {
  unsigned long pixel;
  if (!WindowOpened) 
   {
    fb_error_message = "fb_writeRectangle: window not opened.";
    return FB_ERROR;
   }

  /*
    for binary displays, paint the rectangle ourselves, and rely
    on writePixel to do the dithering.

    If you know a better way to do this, please tell me - it's just
    a hack to support poor souls with 1-bit displays.

    It's slow.  You want speed?  buy an 8-bit display.
   */

  if (FB_BINARY)
   {
    int x,y;
    for(y=B;y<=T;y++)
     for(x=L; x<=R; x++)
      fb_writePixel(x,y,color);
   }
  else
   {
    pixel = FB_cmap.base_pixel + color;
    XSetForeground(display, gc, pixel);
    XFillRectangle(display, window, gc, 
                  L, xwa.height-1-T,  /* reverse y-coordinate */
                  R-L+1, T-B+1);
   }
  fb_flush();
  return FB_SUCCESS;
 }

/*
  Draw a line in the given color.
 */
int fb_drawLine(x1, y1, x2, y2, color)
 int x1, y1, x2, y2;
 FB_COLOR color;
 {
  unsigned long pixel;
  if (!WindowOpened) 
   {
    fb_error_message = "fb_drawLine: window not opened.";
    return FB_ERROR;
   }

  /*
    we could fake grayscale lines by using our own line drawer, but
    that would make things slower, and require work on my part.

    Instead, we make lines either white or black.  Feel free to fix this. 
    For line drawing routines which use fb_drawRectangle, see one of
    the other /dev? directories
   */

  if (FB_BINARY)
   {
    pixel = FB_WHITE;
    if (color < (FB_ncolors/2)) pixel = FB_BLACK;
   }
  else
   {
    pixel = FB_cmap.base_pixel + color;
   }

  XSetForeground(display, gc, pixel);
  XDrawLine(display, window, gc,
              x1, xwa.height-1-y1,   /* reverse y-coordinate */
              x2, xwa.height-1-y2);  /* reverse y-coordinate */
  fb_flush();
 }

/*
  Draw some text at the given position.
 */
int fb_text( x, y, str, color)
 int x, y;
 char *str;
 FB_COLOR color;
 {
  unsigned long pixel;
  if (!WindowOpened) 
   {
    fb_error_message = "fb_text: window not opened.";
    return FB_ERROR;
   }

  /*
     text, like lines, is either black or white.  Even if I had
     the time, I'm not sure I know how to do any better than this.

     Feel free...
   */

  if (FB_BINARY)
   {
    pixel = FB_WHITE;
    if (color < (FB_ncolors/2)) pixel = FB_BLACK;
   }
  else
   {
    pixel = FB_cmap.base_pixel + color;
   }

  XSetForeground(display, gc, pixel);
  XDrawString(display, window, gc, 
                x, xwa.height-1-y, /* reverse y-coordinate */
               str, strlen(str));
  fb_flush();
  return FB_SUCCESS;
 }

/*
 fb_flush() flushes all buffers
 */
int fb_flush()
 {
  XFlush(display);    
  return(FB_SUCCESS);
 }

/*
 fb_pick( x, y, buttons ) returns the location of the mouse, and
 the status of the buttons
 */
int fb_pick( x, y, buttons )
 int *x, *y;
 int *buttons;
 {
  int MouseX, MouseY, MouseButtons;
  int GotEvent;
  
  /*
    allow Motion and Button
   */

  XSelectInput(display, window,
               PointerMotionMask | ButtonMotionMask 
             | ButtonPressMask | ButtonReleaseMask);
 
  /*
    wait for an event
   */
  do
   {
    XNextEvent(display, &event);
    switch (event.type)
     {
      case ButtonPress:
        MouseButtons = 1;
        MouseX = event.xbutton.x;
        MouseY = event.xbutton.y;
        break;
      case ButtonRelease:
        MouseButtons = 0;
        MouseX = event.xbutton.x;
        MouseY = event.xbutton.y;
        break;
      case MotionNotify:
        MouseButtons = event.xmotion.state & (  Button1Mask
                                              | Button2Mask
                                              | Button3Mask
                                              | Button4Mask
                                              | Button5Mask);
        MouseX = event.xmotion.x;
        MouseY = event.xmotion.y;
        break;
      default: break;
     }
   } while (XPending(display));
  /*
    refuse all events
   */
  XSelectInput(display,window,NoEventMask);

  *x = MouseX; 
  *y = xwa.height-1-MouseY;
  *buttons = (MouseButtons != 0);

  if (*buttons) XDefineCursor(display,window,Buttons);
  else          XDefineCursor(display,window,NoButtons);
  XSync(display,True);

  return(0); /* success */
 }

