/*
 File: Tiebreaks.c
 Author: K.R. Sloan
 Last Modified: 11 December 1992
 Purpose: report various tiebreaks

 Input:
        PLAYERS - number of players
        ROUNDS  - number of rounds
        PairingCards - see "PairingCards.h"


        Tiebreaks() computes all tiebreaks and prints a pretty report.
        call RateTournament first, to fill in performance ratings, etc.

        Individual tiebreak routines may be called separately.  Note that
        in these cases it is up to the calling program to allocate the
        memory for the results.  Some tiebreaks require you to call
        RateTournament first, others don't.  Use the source, Luke...
       
        Many of these routines are very similar.  Resist the temptation to
        combine them.  They are written so as to be as "stand alone"
        as is reasonable, with the intent that the few that you really
        want can be lifted without modification.

 */

#include <stdio.h>
#include <malloc.h>
#include "Header.h"
#include "PairingCards.h"

extern int VERBOSE;

/* for sorting Scores - for Harkness, Skoff */
static doubleCompare(i,j)
 double *i, *j;
 {
  if (*i < *j) return(-1);
  if (*i > *j) return( 1);
               return( 0); 
 }


/* Individual Tiebreak routines */

int byScore(PLAYERS,ROUNDS,PairingCards,Score)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double Score[];
 {
  int p,r;

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    score = 0.0; 
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;

      G = &(PC->Games[r]);
      switch(G->Result)
       {
        case 'B':
        case 'W': score += 1.0; break;
        case 'H':
        case 'D': score += 0.5; break;
        default:                break;
       }
     }
    Score[p] = score;
   }
  return(0);
 }

int byCumulative(PLAYERS,ROUNDS,PairingCards,Cumulative)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double Cumulative[];
 {
  int p,r;

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score,cumulative;

    score = 0.0; cumulative = 0.0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;

      G = &(PC->Games[r]);
      switch(G->Result)
       {
        case 'B': cumulative -= 1.0;        /* -1 for unplayed wins */
        case 'W': score      += 1.0; break;
        case 'H':
        case 'D': score      += 0.5; break;
        default:                     break;
       }
      cumulative += score;
     }
    Cumulative[p] = cumulative;
   }
  return(0);
 }

int byHarkness(PLAYERS,ROUNDS,PairingCards,Harkness)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double Harkness[];
 {
  int p,r,start,stop;
  double *Scores;
  double *OppScores;

  if (!(Scores = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"byHarkness: no memory for Scores");
    exit(-1);
   }
  if (!(OppScores = (double *)malloc(ROUNDS*sizeof(double))))
   {
    fprintf(stderr,"byHarkness: no memory for OppScores");
    exit(-1);
   }

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    score = 0.0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;

      G = &(PC->Games[r]);
      switch(G->Result)
       {
        case 'W': score += 1.0; break;
        case 'L':               break;
        default:  score += 0.5; break;
       }
     }
    Scores[p] = score;  /* final score, adjusted for unplayed games */
   }

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W':
        case 'L':
        case 'D': OppScores[r] = Scores[o]; break; /*   played games */
        default:  OppScores[r] = 0.0;       break; /* unplayed games */
       }
     }
    qsort((char *)OppScores,ROUNDS,sizeof(double),doubleCompare);

    if      (ROUNDS <  9) {start = 1; stop = ROUNDS - 1;}
    else if (ROUNDS < 13) {start = 2; stop = ROUNDS - 2;}
    else                  {start = 3; stop = ROUNDS - 3;}

    score = 0.0;
    for(r=start;r<stop;r++) score += OppScores[r];
    Harkness[p] = score;
   }
  free((char *)OppScores);
  free((char *)Scores);
  return(0);
 }

int bySolkoff(PLAYERS,ROUNDS,PairingCards,Solkoff)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double Solkoff[];
 {
  int p,r;
  double *Scores;
  double *OppScores;

  if (!(Scores = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"bySolkoff: no memory for Scores");
    exit(-1);
   }
  if (!(OppScores = (double *)malloc(ROUNDS*sizeof(double))))
   {
    fprintf(stderr,"bySolkoff: no memory for OppScores");
    exit(-1);
   }

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    score = 0.0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;

      G = &(PC->Games[r]);
      switch(G->Result)
       {
        case 'W': score += 1.0; break;
        case 'L':               break;
        default:  score += 0.5; break;
       }
     }
    Scores[p] = score;  /* final score, adjusted for unplayed games */
   }

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W':
        case 'L':
        case 'D': OppScores[r] = Scores[o]; break; /*   played games */
        default:  OppScores[r] = 0.0;       break; /* unplayed games */
       }
     }

    score = 0.0;
    for(r=0;r<ROUNDS;r++) score += OppScores[r];
    Solkoff[p] = score;
   }
  free((char *)OppScores);
  free((char *)Scores);
  return(0);
 }

int bySkoff(PLAYERS,ROUNDS,PairingCards,Skoff)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double Skoff[];
 {
  int p,r,start,stop;
  double *Scores;
  double *OppScores;

  if (!(Scores = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"bySkoff: no memory for Scores");
    exit(-1);
   }
  if (!(OppScores = (double *)malloc(ROUNDS*sizeof(double))))
   {
    fprintf(stderr,"bySkoff: no memory for OppScores");
    exit(-1);
   }

  byScore(PLAYERS,ROUNDS,PairingCards,Scores);

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W':
        case 'L':
        case 'D': OppScores[r] = Scores[o]; break; /*   played games */
        default:  OppScores[r] = 0.0;       break; /* unplayed games */
       }
     }
    qsort((char *)OppScores,ROUNDS,sizeof(double),doubleCompare);

    score = 0.0;
    start = 1; stop = ROUNDS-1;
    for(r=start;r<stop;r++) score += OppScores[r];
    Skoff[p] = score;
   }
  free((char *)OppScores);
  free((char *)Scores);
  return(0);
 }


int byAverage(PLAYERS,ROUNDS,PairingCards,Average)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 int Average[];
 {
  int p,r;

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    unsigned long int Rc;
    int n;

    Rc = 0; n = 0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W':
        case 'L':
        case 'D':  /*   played games */
                 {
                  register int Ropp;

                  Ropp = PairingCards[o].Ro;
                  if (0 == Ropp) Ropp = PairingCards[o].Rn; /* unrated */
                  Rc += Ropp; n++;
                 }
                 break;
        default: break; /* unplayed games */
       }
     }
    if (n) Average[p] = Rc/n; else Average[p] = 0;
   }
  return(0);
 }

int byOppPerformance(PLAYERS,ROUNDS,PairingCards,OppPerformance)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 int OppPerformance[];
 {
  int p,r;

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    unsigned long int Op;
    int n;

    Op = 0; n = 0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W':
        case 'L':
        case 'D': Op += PairingCards[o].Rp; n++; break; /*   played games */
        default:                                 break; /* unplayed games */
       }
     }
    if (n) OppPerformance[p] = Op/n; else OppPerformance[p] = 0;
   }
  return(0);
 }

int byOppCumulative(PLAYERS,ROUNDS,PairingCards,OppCumulative)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double OppCumulative[];
 {
  int p,r;
  double *Cumulative;

  if (!(Cumulative = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"byOppCumulative: no memory for Cumulative");
    exit(-1);
   }
  byCumulative(PLAYERS,ROUNDS,PairingCards,Cumulative);

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double Oc;

    Oc = 0.0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W':
        case 'L':
        case 'D': Oc += Cumulative[o]; break; /*   played games */
        default:                       break; /* unplayed games */
       }
     }
    OppCumulative[p] = Oc;
   }
  free((char *)Cumulative);
  return(0);
 }

int bySonneborn(PLAYERS,ROUNDS,PairingCards,Sonneborn)
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 double Sonneborn[];
 {
  int p,r;
  double *Score;

  if (!(Score = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"bySonneborn: no memory for Score");
    exit(-1);
   }
  byScore(PLAYERS,ROUNDS,PairingCards,Score);

  for(p=0;p<PLAYERS;p++)
   {
    PlayerType *PC;
    double score;

    score = 0.0;
    PC = &(PairingCards[p]);
    for(r=0;r<ROUNDS;r++)
     {
      GameType *G;
      int o;

      G = &(PC->Games[r]);
      o = G->Opponent;
      switch(G->Result)
       {
        case 'W': score +=     Score[o]; break;
        case 'D': score += 0.5*Score[o]; break;
        default:                         break;
       }
     }
    Sonneborn[p] = score;
   }
  free((char *)Score);
  return(0);
 }


/*
   Tiebreaks computes all tiebreaks, and prints a report.
 */

/* for sorting output by tiebreak */
 static double *Score;
 static double *Harkness;
 static double *Solkoff;
 static double *Cumulative;
 static double *Skoff;
 static int    *Average;
 static int    *OppPerformance;
 static double *OppCumulative;
 static double *Sonneborn;

static int TBCompare(i,j)
 int *i, *j;
 {
  if (Score[*i] < Score[*j]) return(-1);
  if (Score[*i] > Score[*j]) return( 1);
  if (Harkness[*i] < Harkness[*j]) return(-1);
  if (Harkness[*i] > Harkness[*j]) return( 1);
  if (Solkoff[*i] < Solkoff[*j]) return(-1);
  if (Solkoff[*i] > Solkoff[*j]) return( 1);
  if (Cumulative[*i] < Cumulative[*j]) return(-1);
  if (Cumulative[*i] > Cumulative[*j]) return( 1);
  if (Skoff[*i] < Skoff[*j]) return(-1);
  if (Skoff[*i] > Skoff[*j]) return( 1);
  if (Average[*i] < Average[*j]) return(-1);
  if (Average[*i] > Average[*j]) return( 1);
  if (OppPerformance[*i] < OppPerformance[*j]) return(-1);
  if (OppPerformance[*i] > OppPerformance[*j]) return( 1);
  if (OppCumulative[*i] < OppCumulative[*j]) return(-1);
  if (OppCumulative[*i] > OppCumulative[*j]) return( 1);
  if (Sonneborn[*i] < Sonneborn[*j]) return(-1);
  if (Sonneborn[*i] > Sonneborn[*j]) return( 1);
  return(0);
 }

/* a useful formatting function */
static int Center(s,str,w)
 FILE *s;
 char *str;
 int w;
 {
  int l,pad,p;

  l = strlen(str);  pad = (w-l)/2;
  for(p=0;p<pad;p++)fprintf(s," ");  fprintf(s,"%s",str);
 }

int Tiebreaks(s,PLAYERS,ROUNDS,PairingCards)
 FILE *s;
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 {
  char N[NameLength], V[ValueLength];
  int p;
  int    *Index;

  if (!(Score = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Score");
    exit(-1);
   }
  byScore(PLAYERS,ROUNDS,PairingCards,Score);

  if (!(Harkness = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Harkness");
    exit(-1);
   }
  byHarkness(PLAYERS,ROUNDS,PairingCards,Harkness);

  if (!(Skoff = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Skoff");
    exit(-1);
   }
  bySkoff(PLAYERS,ROUNDS,PairingCards,Skoff);

  if (!(Solkoff = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Solkoff");
    exit(-1);
   }
  bySolkoff(PLAYERS,ROUNDS,PairingCards,Solkoff);

  if (!(Cumulative = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Cumulative");
    exit(-1);
   }
  byCumulative(PLAYERS,ROUNDS,PairingCards,Cumulative);

  if (!(Average = (int *)malloc(PLAYERS*sizeof(int))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Average");
    exit(-1);
   }
  byAverage(PLAYERS,ROUNDS,PairingCards,Average);

  if (!(OppPerformance = (int *)malloc(PLAYERS*sizeof(int))))
   {
    fprintf(stderr,"Tiebreaks: no memory for OppPerformance");
    exit(-1);
   }
  byOppPerformance(PLAYERS,ROUNDS,PairingCards,OppPerformance);

  if (!(OppCumulative = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for OppCumulative");
    exit(-1);
   }
  byOppCumulative(PLAYERS,ROUNDS,PairingCards,OppCumulative);

  if (!(Sonneborn = (double *)malloc(PLAYERS*sizeof(double))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Sonneborn");
    exit(-1);
   }
  bySonneborn(PLAYERS,ROUNDS,PairingCards,Sonneborn);



  /* print report */

  fprintf(s,"\n");
  strcpy(N,"TOURNAMENT");
  if (SUCCESS == GetDescriptor(N,V)) { Center(s,V,72); fprintf(s,"\n"); }
  strcpy(N,"DATE");
  if (SUCCESS == GetDescriptor(N,V)) { Center(s,V,72); fprintf(s,"\n"); }
  strcpy(N,"SECTION");
  if (SUCCESS == GetDescriptor(N,V)) { Center(s,V,72); fprintf(s,"\n"); }

  if (!(Index = (int *)malloc(PLAYERS*sizeof(int))))
   {
    fprintf(stderr,"Tiebreaks: no memory for Index");
    exit(-1);
   }
  for(p=0;p<PLAYERS;p++) Index[p] = p;
  qsort((char *)Index,PLAYERS,sizeof(int),TBCompare);

  fprintf(s,
"        Name         Score   Hark  Solk   Cum  Skof  Aver  OPer   OCum  Son-B\n");
/*
 ....................  hh.h  hhh.h sss.s ccc.c sss.s  aaaa  pppp cccc.c sss.ss
 */
  for(p=0;p<PLAYERS;p++)
   {
    int i;

    i = Index[PLAYERS-p-1]; /* print in decreasing order */     
    fprintf(s,
    "%20.20s  %4.1lf  %5.1lf %5.1lf %5.1lf %5.1lf  %4d  %4d %6.1lf %6.2lf\n",
              PairingCards[i].Name, 
              Score[i],
              Harkness[i],
              Solkoff[i],
              Cumulative[i],
              Skoff[i],
              Average[i],
              OppPerformance[i],
              OppCumulative[i],
              Sonneborn[i]);     
   }

  free((char *)Index);
  free((char *)Sonneborn);
  free((char *)OppCumulative);
  free((char *)OppPerformance);
  free((char *)Average);
  free((char *)Cumulative);
  free((char *)Solkoff);
  free((char *)Skoff);
  free((char *)Harkness);
  free((char *)Score);

  return(0);  /* all OK */
 }
