/* file: perfpoly2d - do performance timing on code processing 2d polygons */

/* This program facilitates deciding which algorithm is fastest for a given
 * processing task. A number of random polygons are generated, then each
 * algorithm is called to process the set of polygons.
 * To add a new test:
 *	1) add the routine name to the 'PROG_OPTIONS' macro
 *	2) put the routine in this file or another to be resolved at link time.
 *	   Your routine should return the following values:
 *		return 0 for degenerate polygons (lines and points)
 *		return 1 for convex polygons
 *		return 2 for non-convex non-degenerate polygons
 *
 * you can control the amount of output with the verbosity option, try:
 * 	perf < file 0			;only summary output
 *	perf < file 1			;more...
 *	perf < file 9			;the most
 *
 * if the file contains 999 for the first X coord, then program options
 * are read from the file and random polygons are generated. See code.
 *
 * specify the following in 'PROG_OPTIONS'
 * 	<routine name>
 *		this is the actual "C" routine name.
 *	<number of times to test routine with a given polygon> 
 *		this should be > 1 / (HZ * approx. single test time in seconds)
 */
#define	PROG_OPTIONS( macName )						\
	macName(isConvexFred,  NUM_TEST);				\
	macName(isConvexSteve, NUM_TEST);				\
	macName(isConvexKRSloan, NUM_TEST);				\

#define	DEBUG			1
#define	MAX_TEST_ROUTINE	50	/* max routines that may be specified */
#define	MAX_POLY_VERT		100	/* max vertices that may be generated */
#define	MAX_TIME_VERT		10	/* max verts to keep in summary table */
#define	NUM_TEST		100000
#undef	NUM_TEST
#define	NUM_TEST		100

#include "stdio.h"
#include <math.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/times.h>

/*****************************************************************************/
int	verbose    = 0;		/* verbosity for debugging */

static	int	numPolygon = 10;/* num of diff polygons to try for each nvert */
static	int	minVert	   = 3;	/* min number of polygon vertices to generate */
static	int	maxVert	   = 10;/* max number of polygon vertices to generate */

double	drand48();

#define	X	0
#define	Y	1

#if     DEBUG
#define GITrace(level, args)   if ( verbose >= level  ) { args };

#else
#define GITrace(level, args)
#endif

/*****************************************************************************/
long
convertStarStopToUSec( start, stop )
long		       start, stop;
/* make this a routine so you can test the timer function and make sure HZ
 * is defined right on your machine. Use verbosity option 't' to test time.
 */
{
    long uSec = 1000000.0 *(float)(stop - start) / (float)(HZ);
    return( uSec );
}

/*****************************************************************************/
testTimer()
{
    long        timestart, timestop;
    struct  tms timebuf;
    float       uSec;

    printf("Start timer test...\n");
    timestart = times( &timebuf );
    while ( 1 ) {
	timestop = times( &timebuf ); 
	if ( convertStarStopToUSec(timestart, timestop) / 1000000.0 > 10.0) 
	   break;
    }
    timestop = times( &timebuf );
    uSec = convertStarStopToUSec( timestart, timestop );
    printf("\7\7\7End test in %g uSec, %g seconds\n", uSec, uSec / 1000000.0 );
}

/*****************************************************************************/
readPolygon(stream, nvert, pVert )
FILE 	*stream;
int	*nvert;		/* return number of vertices read here */
float	pVert[][2];	/* return vertices here */
/* This routine reads in a stream of 2D polygon coords from the input stream */
{
    int		status, go = 1;

    if ( verbose >= 4 ) {
	printf("Read polygon data:\n");
    }
    *nvert = 0;
    while ( go ) {
	status = fscanf(stream,"%g%g",&(pVert[*nvert][X]), &(pVert[*nvert][Y]));
	switch ( status ) {
	    case EOF :
		go = 0;
		break;
	    case 2 : {
		if ( verbose >= 4 ) {
		    printf("%2d) %8g %8g\n", *nvert, 
					pVert[*nvert][X], pVert[*nvert][Y]);
		}
		(*nvert)++;
		break;
	    }
	    default : {
    		fprintf( stderr, "\7\7\7???BAD POLYGON DATA\n");
		usage();
		break;
	    }
	}
    }
}

/*****************************************************************************/
printPoly( nvert, pVert )
int        nvert;
float      pVert[][2];
{
    int	ii;

    for ( ii = 0;  ii < nvert;  ii++ ) {
	printf("%2d) %8g %8g\n", ii, pVert[ii][X], pVert[ii][Y]);
    }
}

/*****************************************************************************/
static	char	*progName = 0;
usage()
{
    fprintf(stderr, "Usage: %s [arg1 [arg2]]\n", progName );
    fprintf(stderr, "arg1 is either: 0-9 to specify verbosity level\n");
    fprintf(stderr, "                t   for timer test only\n");
    fprintf(stderr, "arg2 is the number of times to call test routine\n");
    fprintf(stderr, "     default value for arg2 is diff for each test\n");
}

/*****************************************************************************/
int
main(argc,argv)
int argc;  char *argv[];
{
    int		doRandom = 0, ii, jj, onVert, onPolygon, onRoutine, onTest, 
		timeIndex, status[MAX_TEST_ROUTINE], 
		useTableNumTest = 1, fixedNumTest, useNumTest;
    long	timestart, timestop;
    struct  tms	timebuf;
    float	uSec;
    int		nvertex;	/* number of vertices in current polygon */
    float	pgon[MAX_POLY_VERT][2];	/* current polygon */
    float	uSecPerTest[MAX_TEST_ROUTINE][MAX_TIME_VERT];

    progName = argv[0];
    if ( argc > 1 ) {
        verbose = argv[1][0] - '0';
	if ( argv[1][0] == 't' ) { testTimer(); verbose = 0; exit(0); }
	if ( verbose < 0  ||  verbose > 9 ) { usage(); exit(0); }
    }
    if ( argc > 2 ) {
	useTableNumTest = 0;			/* ignore table value */
	fixedNumTest = atoi( argv[2] );
	if ( fixedNumTest <= 0 ) { usage(); exit(0); }
    }
    srand( 12345 );
    for ( ii = 0;  ii < MAX_TEST_ROUTINE;  ii++ ) {
	for ( jj = 0;  jj < MAX_TIME_VERT;  jj++ ) {
	    uSecPerTest[ ii ][ jj ] = 0.0;
	}
    }

    readPolygon( stdin, &nvertex, pgon );
    if ( pgon[0][X] > 998.0 ) {
	/* generate random data */
	doRandom   = 1;
				   	numPolygon   = pgon[0][Y];
	minVert         = pgon[1][X]; 	maxVert      = pgon[1][Y];
	useTableNumTest = pgon[2][X];	fixedNumTest = pgon[2][Y];
	if ( minVert < 0 ) {
	    minVert = 0;
	}
    } else {
 	/* use polygon data from file for test */
	minVert    = maxVert = nvertex;
	numPolygon = 1;
    }

    if ( verbose ) {
        printf("Test %d polygons for each vertex count in the range [%d,%d]\n",
		numPolygon, minVert, maxVert );
    }

    /***** loop for each vertex size *****/
    for ( onVert = minVert;  onVert <= maxVert;  onVert++ ) {
	printf("********** test for %d vertices\n", onVert );
	if ( onVert >= MAX_TIME_VERT ) {
	    timeIndex = MAX_TIME_VERT - 1;
	} else {
	    timeIndex = onVert;
	}
	
	if ( verbose >= 3 ) {
printf("uSecs/Test is the ave of the average time for a given polygon size\n");
	    printf("uSecs/Test  totalSec onPolygon  numTest nameTest\n");
	    printf("---------- --------- --------- -------- --------\n");
	}

	/***** test for each polygon of the current vertex size *****/
	for ( onPolygon = 0; onPolygon < numPolygon; onPolygon++ ) {
	    if ( verbose >= 2 ) printf("onPolygon %2d\n", onPolygon );

	    if ( doRandom ) {
		/* make an arbitrary polygon fitting 0-1 range in x and y */
		for ( ii = 0; ii < onVert; ii++ ) {
		    pgon[ii][X] = drand48();
		    pgon[ii][Y] = drand48();
		}
	    }

	    /* define macro to run through all tests. It's true that this
	     * makes it hard to debug the code, but it's working now, 
	     * the type of code here doesn't need much more than visual
	     * inspection, and it makes it easier to add new tests.
	     */
#define	DO_TEST(rname, numtest)						\
	    if ( useTableNumTest ) useNumTest = numtest;		\
	    else useNumTest = fixedNumTest;				\
	    timestart = times( &timebuf );				\
	    for ( onTest = 0;  onTest < useNumTest;  onTest++ ) {	\
		status[onRoutine] = rname( onVert, pgon );		\
	    }								\
	    timestop = times( &timebuf );				\
	    uSec = convertStarStopToUSec(timestart, timestop);		\
	    uSecPerTest[ onRoutine ][timeIndex] += uSec / useNumTest;	\
	    if ( status[onRoutine] == -1 ) status[onRoutine] = 1;	\
	    if ( status[onRoutine] != 1  ) status[onRoutine] = 0;	\
	    if ( onRoutine  &&  status[onRoutine] != status[onRoutine-1] ) { \
		printf(							\
		"\7\7\7???ROUTINE 'rname' disagress with previous routine\n");\
		printf(							\
"    onVert = %d, onPolygon = %d, prev stat = %d, cur stat = %d\n",	\
		   onVert,onPolygon, status[onRoutine-1], status[onRoutine] ); \
		printPoly( onVert, pgon );				\
	    }								\
	    if ( verbose >= 3 ) {					\
	        /* print results of this one test */			\
	        printf("%10g %9g %9d %8d ", uSec/useNumTest,		\
			uSec / 1000000.0, onPolygon, useNumTest, "rname" );\
	    }								\
	    if ( verbose >= 2 ) {					\
	        printf( "%-17s says %s\n", "rname", status[onRoutine] == 1 ? \
			"Convex" : "Non-Convex" );                      \
	    }								\
	    onRoutine++;						\

	    onRoutine 	= 0;
	    PROG_OPTIONS( DO_TEST );
	}

#define	PRINT_RESULTS(rname, numtest)					\
	if ( useTableNumTest ) useNumTest = numtest;			\
	else useNumTest = fixedNumTest;					\
	printf("%10g %9g  aveOfAve %8d ", 				\
		uSecPerTest[ onRoutine ][timeIndex] / (float)(numPolygon),\
		uSecPerTest[ onRoutine ][timeIndex] / 1000000.0, useNumTest);\
	printf( "%-17s says %s\n", "rname", status[onRoutine] == 1 ?	\
			"Convex" : "Non-Convex" );  			\
	onRoutine++;						 	\

	onRoutine = 0;
	PROG_OPTIONS( PRINT_RESULTS );
    }

    if ( verbose )
        printf("\nSummary of test results ordered by number of vertices:\n");
    for ( onVert = minVert;  onVert <= maxVert && verbose;  onVert++ ) {
	printf("********** Summary for %d vertices\n", onVert );
	if ( onVert >= MAX_TIME_VERT ) {
	    timeIndex = MAX_TIME_VERT - 1;
	} else {
	    timeIndex = onVert;
	}

	onRoutine   = 0;
	PROG_OPTIONS( PRINT_RESULTS );
    }
    if ( status[0] == 1 ) return(0);
    else return(1);
}
