/*
   File: FFTwff.c 
   Authors: Mike Schmidt,
            K.R. Sloan
   Last Modified: 11 September 1990
   Purpose: Take an Intensity image file from stdin, perform Fourier transform
            on it, output Complex image file to stdout. 
   Assumptions: Height and width of input image must be powers of 2.
                if not, we create an image with height and width the next
                higher power of two, with the original image surrounded
                by a field of average gray.  This should not do TOO much
                violence to the FFT.
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>
#include <Complex.h>

extern char **malloc_2d(); 

static int CeilLog2(M)
 int M;
 {
  int LogN, N;

  for (LogN = 0, N = 1; N < M ; N *= 2) LogN++;
  return (LogN);
 }

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s\n",
           RoutineName);
 }

static void
Pass(fdIn,fdOut)
 FILE *fdIn, *fdOut;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right;
  char Name[NameLength], Value[ValueLength], WhatBands[NameLength];
  char MaxMagString[ValueLength];
  int x,y,n,BitsPerBand,MaxValue;
  double MaxMag,LogMax,epsilon = 0.0000001;
  unsigned short Pixel[2];
  int wBottom, wLeft, wTop, wRight, wWidth, wHeight, xBorder, yBorder;
  int wX, wY;
  unsigned short Gray;
  Complex **ComplexArray;
  int passed = 0;
  
  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }
  if (FAILURE == OpenFB(&FBout))          { (void)CloseFB(&FBin);  return; }
 
  GetColorSystem(FBin,WhatBands,&BitsPerBand);
  if (0 != strcmp(WhatBands,"I"))
   {
    fprintf(stderr,"Input file type is not I!\n");
    (void)CloseFB(&FBin);
    (void)CloseFB(&FBout);
    return;
   }

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Read in the pixels into local array of complex numbers. */
  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Note original dimensions - so inverse FFT can remove the border */

  sprintf(Value,"%d",Bottom);
  SetDescriptor(FBout, "X-Original-Bottom", Value);
  sprintf(Value,"%d",Left);
  SetDescriptor(FBout, "X-Original-Left", Value);
  sprintf(Value,"%d",Top);
  SetDescriptor(FBout, "X-Original-Top", Value);
  sprintf(Value,"%d",Right);
  SetDescriptor(FBout, "X-Original-Right", Value);

  /* calculate dimensions of Fourier domain image */
  wWidth  = (int) pow(2.0, (double) CeilLog2(Right-Left+1));  
  wHeight = (int) pow(2.0, (double) CeilLog2(Top-Bottom+1));  
  xBorder = (wWidth  - (Right-Left+1)) / 2;
  yBorder = (wHeight - (Top-Bottom+1)) / 2;
  Gray = (1 << (BitsPerBand-1)) - 1;

  ComplexArray = (Complex **) malloc_2d( wWidth, wHeight,
                                         sizeof (Complex) );
  if (ComplexArray == (Complex **)NULL) 
   {
    fprintf (stderr,"Insufficient memory for transform!\n");
    (void)CloseFB(&FBin);
    return;
   }

  /* move image to complex array - embedded in gray border */
  for (y=0; y<yBorder; y++)
   for (x=0; x<wWidth; x++)
    {
     ComplexArray[y][x].real = (double) Gray;
     ComplexArray[y][x].imag = 0.0;
    }  
  for (;y<yBorder+(Top-Bottom+1);y++)
   {
    for (x=0;x<xBorder;x++)
     {
      ComplexArray[y][x].real = (double) Gray;
      ComplexArray[y][x].imag = 0.0;
     }  
    for (;x<xBorder+(Right-Left+1);x++)
     {
      if (FAILURE == NextPixelIn(FBin,Pixel))
       { (void)CloseFB(&FBin); return; }
      ComplexArray[y][x].real = (double)Pixel[0];
      ComplexArray[y][x].imag = 0.0;
     } 
    for (;x<wWidth;x++)
     {
      ComplexArray[y][x].real = (double) Gray;
      ComplexArray[y][x].imag = 0.0;
     }  
   }
  for (;y<wHeight; y++)
   for (x=0; x<wWidth; x++)
    {
     ComplexArray[y][x].real = (double) Gray;
     ComplexArray[y][x].imag = 0.0;
    }  

  /* Perform Fast Fourier Transform. */
  FFT_2d (1, wWidth, wHeight, ComplexArray[0]);

  /* Find maximum magnitude^2 of output pixels. */
  MaxMag = 0.0;
  for (y=0;y<wHeight;y++)
   for (x=0;x<wWidth;x++)
    {
     double MM, Real, Imaginary;

     Real      = ComplexArray[y][x].real; 
     Imaginary = ComplexArray[y][x].imag;
     MM = Real*Real + Imaginary*Imaginary;
     if(MM > MaxMag) MaxMag = MM; 
    }

  /* Set output file for mA image and set up local variables. */ 
  SetColorSystem(FBout,"mA",BitsPerBand);
  SetBounds(FBout, 0, 0, wHeight-1, wWidth-1);
  MaxValue = (1 << BitsPerBand) - 1;
  LogMax = log (MaxMag + 2.0);
  sprintf (MaxMagString,"%f",MaxMag);
  SetDescriptor (FBout,"X-MaxMag",MaxMagString);

  /* Start the output stream. */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* shuffle the quadrants as we stream through the file */
  {
   int HalfWidth, HalfHeight, cy, cx;
   double LogMagnitude, Phase;

   HalfWidth  = wWidth / 2;
   HalfHeight = wHeight / 2;

   for (cy=0, y=HalfHeight; cy<wHeight; cy++, y++)
    {
     if (y == wHeight) y = 0;
     for (cx=0, x=HalfWidth; cx<wWidth; cx++, x++)
      {
       double Real, Imaginary;
       if (x == wWidth) x = 0;
       Real      = ComplexArray[y][x].real;
       Imaginary = ComplexArray[y][x].imag;
       LogMagnitude = log(Real*Real + Imaginary*Imaginary + 2.0) / LogMax;
       /* Sun's math library sucks dead goats */
       if ((1.0 == (1.0+Imaginary)) && (1.0 == (1.0+Real)))
        Phase = 0.0;
       else
        Phase =  atan2(Imaginary, Real) / (2.0*M_PI);
       if (Phase<0.0) Phase += 1.0;
       Pixel[0] = MaxValue*LogMagnitude;
       Pixel[1] = MaxValue*Phase;
       if (FAILURE == NextPixelOut(FBout,Pixel))
        { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
      }
     wffFlush(FBout);
    }
  } 
  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);
 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
 
  RoutineName = argv[ArgsParsed++];
  if (ArgsParsed < argc)  { usage(); exit (-1); }	

  Pass(stdin,stdout);

  exit (0);
}

