/*
  File: sun2wff.c
  Authors: K.R. Sloan
           Yun Wang
  Last Modified: 21 March 1991
  Purpose: Convert a Sun Rasterfile format file (.ras) to .WFF format
  NOTE: this is a very simple first attempt - it doesn't come close
        to handling all possible rasterfile options.  
        It handles colormap now.
*/
#include <assert.h>
#include <stdio.h>
#include "rasterfile.h"
#include "wff.h"

char *calloc();
unsigned char Red[256], Green[256], Blue[256];

int VERBOSE = 0;
static char *RoutineName;
static usage()
 {
  fprintf( stderr, "Usage: %s < sun > wff\n", RoutineName );
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s\n",s); exit(1);
 }

/* swap n PAIRS of bytes */
static void ByteSwap(p,n)
 char *p;
 int n;
 {
  register char *p0, *p1; 
  register char t;
  register int i;

  p0 = p; p1 = p+1;
  for (i=0;i<n;i++)
   {
    t = *p0; *p0 = *p1; *p1 = t;
    p0 += 2; p1 += 2;
   }
 }

/* swap n PAIRS of halfwords */
static void HWSwap(p,n)
 char *p;
 int n;
 {
  register char *p0, *p1; 
  register char t;
  register int i;

  p0 = p; p1 = p+2;
  for (i=0;i<n;i++)
   {
    t = *p0; *(p0++) = *p1; *(p1++) = t;
    t = *p0; *(p0++) = *p1; *(p1++) = t;
   }
 }

/* Sun Rasterfile stuff */

struct rasterfile Sun;
int BytesPerScanline, ImageSize;
unsigned char *Raster;

static void SunLoad(s)
 FILE *s;
 {
int num, i, j,length;

  /* read header */
  fread(&Sun.ras_magic,sizeof(int),1,s);
  fread(&Sun.ras_width,sizeof(int),1,s);
  fread(&Sun.ras_height,sizeof(int),1,s);
  fread(&Sun.ras_depth,sizeof(int),1,s);
  fread(&Sun.ras_length,sizeof(int),1,s);
  fread(&Sun.ras_type,sizeof(int),1,s);
  fread(&Sun.ras_maptype,sizeof(int),1,s);
  fread(&Sun.ras_maplength,sizeof(int),1,s);
  if (  ( 0      != Sun.ras_maptype)
       &&(0   != Sun.ras_maplength) ) 
  {
     length=Sun.ras_maplength/3;
     fread(Red,1,length,s);
     fread(Green,1,length,s);
     fread(Blue,1,length,s);
  }
  if (VERBOSE)
   {
    fprintf(stderr,"ras_magic %x\n",Sun.ras_magic);
    fprintf(stderr,"ras_width %d\n",Sun.ras_width);
    fprintf(stderr,"ras_height %d\n",Sun.ras_height);
    fprintf(stderr,"ras_depth %d\n",Sun.ras_depth);
    fprintf(stderr,"ras_length %d\n",Sun.ras_length);
    fprintf(stderr,"ras_type %d\n",Sun.ras_type);
    fprintf(stderr,"ras_maptype %d\n",Sun.ras_maptype);
    fprintf(stderr,"ras_maplength %d\n",Sun.ras_maplength);
   }

  if(ferror(s)) FatalError("Error reading header");

  /* check to make sure we can handle this image */

  if (RAS_MAGIC != Sun.ras_magic)
   {
    ByteSwap(&Sun.ras_magic,2);
    ByteSwap(&Sun.ras_width,2);
    ByteSwap(&Sun.ras_height,2);
    ByteSwap(&Sun.ras_depth,2);
    ByteSwap(&Sun.ras_length,2);
    ByteSwap(&Sun.ras_type,2);
    ByteSwap(&Sun.ras_maptype,2);
    ByteSwap(&Sun.ras_maplength,2);
   }
   if (RAS_MAGIC != Sun.ras_magic)
   {
    HWSwap(&Sun.ras_magic,1);
    HWSwap(&Sun.ras_width,1);
    HWSwap(&Sun.ras_height,1);
    HWSwap(&Sun.ras_depth,1);
    HWSwap(&Sun.ras_length,1);
    HWSwap(&Sun.ras_type,1);
    HWSwap(&Sun.ras_maptype,1);
    HWSwap(&Sun.ras_maplength,1);
   }
  if (RAS_MAGIC != Sun.ras_magic)      FatalError("Bad magic");

  if (  (RT_OLD      != Sun.ras_type)
      &&(RT_STANDARD != Sun.ras_type)
     )                    FatalError("I do not do RT_BYTE_ENCODED type");

  if (  (RMT_NONE         != Sun.ras_maptype) 
      &&(RMT_EQUAL_RGB  != Sun.ras_maptype)
     )                    FatalError("I do not do RMT_RAW"); 

  /* lucky us - an easy case */

  BytesPerScanline = (((Sun.ras_width * Sun.ras_depth) + 15) / 16) * 2;
  ImageSize = BytesPerScanline * Sun.ras_height;

  Raster = (unsigned char *)calloc(1, ImageSize*sizeof(unsigned char));
  if (!Raster) FatalError("no memory for Raster");  

  fread(Raster, sizeof(unsigned char), ImageSize, s);
  if(ferror(s)) FatalError("Error reading image");

  /* Raster and Sun are filled in and ready to go */
 }

/* These function deal with WFF files */

static FrameBufferType *SetUpWFFFile(stream, xsize, ysize, zsize)
 FILE *stream;
 int xsize, ysize, zsize;
 {
  FrameBufferType *FrameBuffer;
  int Bottom, Left, Top, Right;
  int BitsPerBand;
  char WhatBands[10], Name[NameLength], Value[ValueLength];

  FrameBuffer = (FrameBufferType *)0;

  OpenFB(&FrameBuffer);
  assert(FrameBuffer);

  Bottom = 0; Left = 0; Top = ysize-1; Right = xsize-1;
  SetBounds(FrameBuffer, Bottom, Left, Top, Right);

  /* we only handle the simple cases...*/
  if ((Sun.ras_maplength != 0) || ( 24 == zsize ))
   { BitsPerBand = 8; strcpy(WhatBands,"RGB"); }
  else             { BitsPerBand = zsize; strcpy(WhatBands,"I"  ); }
  SetColorSystem(FrameBuffer, WhatBands,  BitsPerBand);


  strcpy(Name,"X-CreatedBy");
  strcpy(Value,RoutineName);
  SetDescriptor(FrameBuffer, Name, Value);

  strcpy(Name,"Encoding");
  strcpy(Value,"AIS");
  SetDescriptor(FrameBuffer, Name, Value);

  /* Header operations over, now we can start the output stream */
  PassImageOut(stream, FrameBuffer);
  return (FrameBuffer);
 }  

static void WriteWFFScanLine(FB, xsize, ScanLine)
 FrameBufferType *FB;
 int xsize;
 unsigned short *ScanLine;
 {
  assert (SUCCESS == NextNPixelsOut(FB, xsize, ScanLine));
 }


static void CloseWFFFile(FB)
 FrameBufferType *FB;
 {
  CloseFB(&FB);
 }

/*
  The Main Line (are we in Philadelphia, yet?)
 */
int
main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  FrameBufferType *FB;
  unsigned char *Iptr;
  int mask, bits, octet;
  int x,y;

  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': usage(); exit(-1);
     }
   }
  
  SunLoad(stdin);

  FB = SetUpWFFFile(stdout, Sun.ras_width, Sun.ras_height, Sun.ras_depth);
  assert(FB);

  if (VERBOSE) fprintf(stderr,"%s: Sun.Ras-depth = %d\n",
                 RoutineName,Sun.ras_depth);

  mask = (1 << Sun.ras_depth) - 1;
  for(y=0;y<Sun.ras_height;y++)
   {
    unsigned short Pixel[3];
    register unsigned char c;

    Iptr = &Raster[(Sun.ras_height-1-y)*BytesPerScanline];
    bits = 0;
    
    /* this is slow, but you'll find it easier to extend...*/  
    for(x=0; x<Sun.ras_width; x++)
     {
      if (24 == Sun.ras_depth)
       {
        Pixel[0] = (unsigned short) *(Iptr++);
        Pixel[1] = (unsigned short) *(Iptr++);
        Pixel[2] = (unsigned short) *(Iptr++);
       }
      else if ( 0 != Sun.ras_maplength )
       {
        c =  *(Iptr++);
        Pixel[0] = Red[c];    /* Insert colormap here */
        Pixel[1] = Green[c];
        Pixel[2] = Blue[c];
       }
      else if (8 == Sun.ras_depth)
       {
        Pixel[0] = (unsigned short) *(Iptr++);
       }
      else
       {
        if (Sun.ras_depth > bits) { octet = *(Iptr++); bits = 8; }
        bits -= Sun.ras_depth;
        Pixel[0] = (unsigned short)(mask & (octet >> bits));
       }
      NextPixelOut(FB, (unsigned short *)Pixel);
     }
    wffFlush(FB);
   }
  CloseWFFFile(FB);
  exit(0);
 }
