/*
  File: UNC2wff.c
  Author: K.R. Sloan
  Last Modified: 22 April 1991
  Purpose: convert UNC file to .wff
 */

#include <assert.h>
#include <stdio.h>
#include <sys/time.h>
#include "wff.h"
#include "image.h"

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"usage is:\n\t%s [-h][-v][-f UNC] >wff\n", RoutineName);
  exit(-1);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(1);
 }

/* These function deal with WFF files */

static FrameBufferType *SetUpWFFFile(stream, xsize, ysize, zsize, BitsPerBand)
 FILE *stream;
 int xsize, ysize, zsize, BitsPerBand;
 {
  FrameBufferType *FrameBuffer;
  int Bottom, Left, Top, Right;
  char WhatBands[10], Name[NameLength], Value[ValueLength];

  FrameBuffer = (FrameBufferType *)0;

  OpenFB(&FrameBuffer);
  assert(FrameBuffer);

  Bottom = 0; Left = 0; Top = ysize-1; Right = xsize-1;
  SetBounds(FrameBuffer, Bottom, Left, Top, Right);

  switch (zsize)
   {
    case 1: strcpy(WhatBands,"I"  ); break;
    case 2: strcpy(WhatBands,"IA"  ); break;
    case 3: strcpy(WhatBands,"RGB");  break;
    case 4: strcpy(WhatBands,"RGBA");  break;
    default: FatalError("Bad zsize");
   }

  SetColorSystem(FrameBuffer, WhatBands,  BitsPerBand);

  strcpy(Name,"X-CreatedBy");
  strcpy(Value,RoutineName);
  SetDescriptor(FrameBuffer, Name, Value);

  strcpy(Name,"Encoding");
  strcpy(Value,"AIS");
  SetDescriptor(FrameBuffer, Name, Value);

  /* Header operations over, now we can start the output stream */
  PassImageOut(stream, FrameBuffer);
  return (FrameBuffer);
 }  

static void WriteWFFScanline(FB, xsize, Scanline)
 FrameBufferType *FB;
 int xsize;
 unsigned short *Scanline;
 {
  assert (SUCCESS == NextNPixelsOut(FB, xsize, Scanline));
 }

static void Pass(UNCFileName, fdOut)
 char *UNCFileName;
 FILE *fdOut;
 {
  FrameBufferType *FB;
  int Left, Bottom, Top, Right, BitsPerBand;
  int BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  double AspectRatio;
  int width, height, depth;
  unsigned short *WFFScanline;
  int WFFScanlineLength;
  IMAGE *UNCimage;
  int UNCformat, UNCsize, UNCcnt, UNCdimc;
  int UNCdimv[nDIMV];
  int UNCmaxmin[2];
  int Corners[2][2];
  int Coarseness[2];
  unsigned char *UNCScanline;
  int UNCScanlineLength;
  int x,y;

  if (VERBOSE) fprintf(stderr,"%s: opening %s as UNC image\n",
                        RoutineName, UNCFileName);
  UNCimage = imopen(UNCFileName, READ);
  if ((IMAGE *)0 == UNCimage) FatalError("opening UNC file");

  if (INVALID == imheader(UNCimage,
                           &UNCformat, &UNCsize, &UNCcnt,
                           &UNCdimc, UNCdimv, UNCmaxmin)) 
   FatalError("reading UNC header");

  if (VERBOSE) fprintf(stderr,"%s: %d dimensions\n",RoutineName, UNCdimc);
  switch (UNCdimc)
   {
    case 2: height = UNCdimv[0];
            width = UNCdimv[1];
            Coarseness[0] = 1; Coarseness[1] = 1;
            break;
    default: FatalError("Unsupported dimensionality");
   }

  switch (UNCformat)
   {
    case GREY:        BandsPerPixel = 1;
                      BitsPerBand = 16;
                      UNCScanlineLength = width*2;
                      break;
    case COLORPACKED: BandsPerPixel = 4;
                      BitsPerBand = 8;
                      UNCScanlineLength = width*4;
                      break; 
    default:
    case COLOR:
    case USERPACKED:  FatalError("Unsupported format");
   }


  if (VERBOSE)
   fprintf(stderr,"%s: %s is %d x %d x %d x %d\n",
      RoutineName, UNCFileName, width, height, BandsPerPixel, BitsPerBand);

  FB = SetUpWFFFile(stdout, width, height, BandsPerPixel, BitsPerBand);
  assert(FB);

  UNCScanline = (unsigned char *) malloc(UNCScanlineLength);
  if ((unsigned char *)0 == UNCScanline)
   FatalError("no memory for UNCScanline");

  WFFScanlineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanline = (unsigned short *) malloc(WFFScanlineLength);
  if ((unsigned short *)0 == WFFScanline)
   FatalError("no memory for WFFScanline");

  Bottom = 0; Top = height-1;
  Left = 0; Right = width-1;

  for (y=Bottom; y<=Top; y++)
   {
    unsigned char *UNC;
    unsigned short *WFF;
    int p;

    Corners[0][0] = height-1-y; Corners[0][1] = height-1-y;
    Corners[1][0] = 0; Corners[1][1] = width-1;
    if (INVALID == imgetpix(UNCimage, Corners, Coarseness, UNCScanline))
     FatalError("reading UNC Scanline");

    UNC = UNCScanline;
    WFF = WFFScanline;
    for(x=Left;x<=Right;x++)
     {
      switch (UNCformat)
       {
        case GREY: p  = *UNC++ << 8;
                   p |= *UNC++;
                   *WFF++ = (unsigned short)p;
                   break;
        case COLORPACKED: *WFF++ = *UNC++; /* r */
                          *WFF++ = *UNC++; /* g */
                          *WFF++ = *UNC++; /* b */
                          *WFF++ = *UNC++; /* a */
                          break; 
        default:
        case COLOR:
        case USERPACKED:  FatalError("Unsupported format");
       }
     }
    if (FAILURE == NextNPixelsOut(FB,width, WFFScanline))
     FatalError("NextNPixelsOut failed");
   }

  imclose(UNCimage);
  CloseFB(&FB);
  free(WFFScanline); free(UNCScanline);
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *UNCFileName;

  UNCFileName = "wff.UNC";
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'f': if (ArgsParsed >= argc)  { usage(); exit(-1);}
                UNCFileName = argv[ArgsParsed++];
                break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h':  usage(); exit(-1);
     }
   }

  Pass(UNCFileName, stdout);
  
  exit(0);
 }
