/*
  File: targa2wff.c
  Author: K.R. Sloan
  Last Modified: 16 April 1991
  Purpose: Convert .tga image files to .wff format.
           This version only attempts to deal with a particular
           subset of targa images - feel free to extend it as needed.

           We can handle:
            RGB images (or RGBA, but we discard the A)
            8 bits per band
            origin at lower left
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s\n",
           RoutineName);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(-1);
 }

/* Targa format routines */

unsigned char Header[18] =
 {                    /* Header for targa file                            */
  0,                  /* [0] Number of Characters in ID field             */
  0,                  /* [1] Color Map Type  0 means NONE                 */
  2,                  /* [2] Image Type      2 means RGB                  */
  0, 0, 0, 0, 0,      /* [3,4,5,6,7] color map specs - we don't have one  */
  0, 0,               /* [ 8, 9] xorigin                                  */
  0, 0,               /* [10,11] yorigin                                  */
  0, 0,               /* [12,13] width                                    */
  0, 0,               /* [14,15] height                                   */ 
  24,                 /* [16] bits per pixel                              */
  0                   /* [17] bits 3-0: number of attribute bits per pixel 
                                     4: 0/1 = origin at (left/right)
                                     5: 0/1 = origin at (bottom/top)
                                   7-6: interleaving (0 means no interleave)

                         I'm not sure if 3-0 ought to be 0 or 8, but
                         the code says 0 and it works, so...
                       */

 };

static void ReadTargaHeader(targaFile, xsize, ysize, zsize, xoffset, yoffset)
 FILE *targaFile;
 int *xsize, *ysize, *zsize;
 int *xoffset,*yoffset;
 {
  if (18 != fread(Header, 1, 18, targaFile))
   FatalError("reading targa header");

  if (   ( 2 != Header[ 2])
      || ((32 != Header[16]) && (24 != Header[16])) )
   FatalError("strange targa format");

  *xoffset = Header[ 8] | (Header[ 9] << 8);
  *yoffset = Header[10] | (Header[11] << 8);
  *xsize   = Header[12] | (Header[13] << 8);
  *ysize   = Header[14] | (Header[15] << 8);
  *zsize   = Header[16] >> 3;  /* bits / 8 */
 }

static void ReadTargaScanline(targaFile, Scanline, CharsPerScanline)
 FILE *targaFile;
 unsigned char *Scanline;
 int CharsPerScanline;
 {
  if (CharsPerScanline != fread(Scanline, 1, CharsPerScanline, targaFile))
   FatalError("reading targa image Scanline");
 }

static void
Pass(fdIn, fdOut)
 FILE *fdIn, *fdOut;
 {
  FrameBufferType *FBout = NULL;
  int Bottom, Left, Top, Right;
  char Name[NameLength], Value[ValueLength];
  char WhatBands[ValueLength];
  int  BitsPerBand, BandsPerPixel;
  int x,y,b;
  unsigned short *Pixel;
  unsigned char *TargaScanline;
  unsigned char *pin;
  unsigned short *pout;
  int xsize, ysize, zsize;
  int xoffset, yoffset;

  /* Read the targa file header */
  ReadTargaHeader(fdIn, &xsize, &ysize, &zsize, &xoffset, &yoffset);
  
  /* Create and write out .wff header */
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBout))  FatalError("OpenFB failed");
  Bottom = xoffset;             Left  = yoffset;
  Top    = Bottom + ysize - 1;  Right = Left + xsize - 1;
  SetBounds(FBout, Bottom, Left, Top, Right);
  (void) strcpy(WhatBands,"RGB"); BitsPerBand = 8;
  SetColorSystem(FBout, WhatBands,  BitsPerBand);
  BandsPerPixel = 3;

  (void) strcpy(Name,"X-CreatedBy");
  (void) strcpy(Value,RoutineName);
  SetDescriptor(FBout, Name, Value);

  if (FAILURE == PassImageOut(fdOut, FBout))
   FatalError("PassImageOut failed");

  /* Allocate memory for output scan line buffer */
  Pixel = (unsigned short *) 
    malloc( sizeof(unsigned short)*(unsigned)(xsize*BandsPerPixel));
  if ((unsigned short *)0 == Pixel)
   FatalError("Unable to allocate memory for output scan line");

  /* Allocate memory for input scan line buffer */
  TargaScanline = (unsigned char *) 
    malloc( sizeof(unsigned short)*(unsigned)(xsize*zsize));
  if ((unsigned char *)0 == TargaScanline)
   FatalError("Unable to allocate memory for input scan line\n");

  for (y=Bottom;y<=Top;y++) 
   {
    ReadTargaScanline(fdIn, TargaScanline, xsize*zsize);
    pin = TargaScanline;
    pout = Pixel;
    for (x=Left;x<=Right;x++, pout += BandsPerPixel ) 
     {
      pout[2] = *pin++;
      pout[1] = *pin++;
      pout[0] = *pin++;
      if (4 == zsize) pin++; 
     }
    if (FAILURE == NextNPixelsOut(FBout, xsize, Pixel))
     FatalError("NextNPixelsOut failed");
   }
  (void)CloseFB(&FBout);
 }

int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
 
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) {usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'v': VERBOSE = -1; break;
      case 'h':
      default:  usage(); exit(-1);  
     }         
   }

  Pass(stdin,stdout);
  exit(0);
 }



