/***************************************************************************/
/* FUNCTION: int img24_tobw()
/*
/* DESCRIPTION:
/*    Converts a 24 bit image into an 8-bit black and white image 
/*    which uses a 8-bit gray color map.
/*
/* USAGE:
/*    error_ret = img24_tobw(buf24,buf8,xsiz,ysiz);
/*
/* ARGUMENTS:
/*    int            *xsiz   (in) : X dimension of image
/*    int            *ysiz   (in) : Y dimension of image
/*    unsigned char  *buf24  (in) : input 24-bit image array
/*    unsigned char  *buf8   (in) : output 8-bit image array
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*
/* COMMENTS:
/*    The 24-bit image input array must be in pixel interleaved format.
/*
/* EXAMPLE:
/* SEE ALSO:
/* INFO:
/*    Author : Ray Idaszak
/*    Date   : May 8, 1989
/*    email  : rayi@ncsa.uiuc.edu
/***************************************************************************/

#include "rtep.h"

#define MAX_BITS 8

#define RP  77
#define GP 151
#define BP  28

int img24_tobw(buf24,buf8,xsiz,ysiz)
unsigned char *buf24,*buf8;
int xsiz,ysiz;
{
   int err_ret;
   int ival,num_pixels;
   unsigned char *buf_ptr;
   register int i;

   err_ret = 0;

   num_pixels = xsiz * ysiz;

   buf_ptr = &buf24[0];
   for(i=0; i<num_pixels; i++){
      ival = (RP*(*buf_ptr++) + GP*(*buf_ptr++) + BP*(*buf_ptr++))>>8;
      buf8[i] = (unsigned char)ival;
   }

error:
#if RTE_PRERR
   err_msg("img24_tobw",err_ret);
#endif
   return(err_ret);
}

/***************************************************************************/
/* FUNCTION: int seqpal_tobw()
/*
/* DESCRIPTION:
/*    Converts the RGB color entries in a color map to their
/*    black and white equivalent colors.
/*
/* USAGE:
/*    error_ret = seqpal_tobw(in_pal,out_pal);
/*
/* ARGUMENTS:
/*    unsigned char  in_pal[][3]     (in) : RGB values for input color map
/*    unsigned char  out_pal[][3]   (out) : RGB values for output color map
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*
/* COMMENTS:
/*    Currently this subroutine assumes an 8-bit color map with 256
/*    entries.
/*
/* EXAMPLE:
/* SEE ALSO:
/* INFO:
/*    Author : Ray Idaszak
/*    Date   : May 8, 1989
/*    email  : rayi@ncsa.uiuc.edu
/***************************************************************************/

#define PALBUFSIZE 256

int seqpal_tobw(ipal,opal)
unsigned char ipal[PALBUFSIZE][3];
unsigned char opal[PALBUFSIZE][3];
{
	int err_ret;
	int ival;
	register int i;

	err_ret = 0;

	for(i=0; i<PALBUFSIZE; i++){
		ival = (RP*ipal[i][0] + GP*ipal[i][1] + BP*ipal[i][2])>>8;
		opal[i][0] = ival;
		opal[i][1] = ival;
		opal[i][2] = ival;
	}

error:
#if RTE_PRERR
	err_msg("seqpal_tobw",err_ret);
#endif
	return(err_ret);
}

/***************************************************************************/
/* FUNCTION: int img8_tobw()
/*
/* DESCRIPTION:
/*    Converts an 8-bit image into an 8-bit black and white image 
/*    which uses a 8-bit gray color map.
/*
/* USAGE:
/*    error_ret = img8_tobw(ibuf,obuf,xsiz,ysiz,palette);
/*
/* ARGUMENTS:
/*    int            *xsiz   (in) : X dimension of image
/*    int            *ysiz   (in) : Y dimension of image
/*    unsigned char  *ibuf   (in) : input 8-bit image array
/*    unsigned char  *obuf  (out) : output 8-bit image array
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*
/* COMMENTS:
/*    The output image array depends on an 8-bit gray scale color
/*    map to be displayed correctly (i.e. index 0 = BLACK, index 
/*    255 = WHITE).
/*
/* EXAMPLE:
/* SEE ALSO:
/* INFO:
/*    Author : Ray Idaszak
/*    Date   : May 8, 1989
/*    email  : rayi@ncsa.uiuc.edu
/***************************************************************************/

int img8_tobw(ibuf,obuf,xsiz,ysiz,palette)
unsigned char *ibuf,*obuf;
int xsiz,ysiz;
unsigned char palette[][3];
{

	int err_ret;
	unsigned char tpal[PALBUFSIZE][3];
	register int i;
	int ival,num_pixels;
	unsigned char *ibuf_ptr,*obuf_ptr;

	seqpal_tobw(palette,tpal);

	err_ret = 0;

	ibuf_ptr = &ibuf[0];
	obuf_ptr = &obuf[0];
	num_pixels = xsiz * ysiz;
	i = num_pixels;
	do{
		i--;
		ival = (int)(*ibuf_ptr++);
		*obuf_ptr++ = *tpal[ival]; 
	}while(i);

error:
#if RTE_PRERR
	err_msg("img8_tobw",err_ret);
#endif
	return(err_ret);
}
