/***************************************************************************/
/* FUNCTION: int make_rgbpal()
/*
/* DESCRIPTION:
/*    Makes a color palette based on the true color interpretation
/*    of the bits used in the color palette index.
/*
/* USAGE:
/*    error_ret = make_rgbpal(palette,bits1,bits2,bits3);
/*
/* ARGUMENTS:
/*    unsigned char  palette[][3]  (out) : output color palette
/*    int            bits1          (in) : bits per component 1 
/*    int            bits2          (in) : bits per component 2 
/*    int            bits3          (in) : bits per component 3 
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*    1 : illegal value for bits1,bits2,bits3
/*    2 : bits1_bits2_bits3 exceeds quantization bits available
/*
/* COMMENTS:
/*    Currently this subroutine assumes an 8-bit color map with 256
/*    entries.
/*
/* EXAMPLE:
/* SEE ALSO:
/*    img_quant()
/*
/* INFO:
/*    Author : Ray Idaszak
/*    Date   : May 8, 1989
/*    email  : rayi@ncsa.uiuc.edu
/***************************************************************************/

#include "rtep.h"

#define MIN_BITS 0 	/* minimum bits per component		*/
#define MAX_BITS 8	/* maximum bits, all components combined */

#define MAX_VAL 256	/* 2^MAX_BITS	*/
#define MAX_VAL1 255	/* MAX_VAL - 1	*/

int make_rgbpal(palette,bits1,bits2,bits3)
unsigned char palette[][3];
int bits1,bits2,bits3;
{
	int err_ret;
	int i;
	int cur_val,last_val,ival,ictr;
	int parts1,parts2,parts3;
	int parts11,parts21,parts31;
	int mask1,mask2,mask3;
	int tbits;

	err_ret = 0;

	if ((bits1 < 0) || (bits2 < 0) || (bits3 < 0)){
		err_ret = 1;
		goto error;
	}

	tbits = bits1 + bits2 + bits3;
	if (tbits > MAX_BITS){
		err_ret = 2;
		goto error;
	}

	parts1 = 0;
	parts2 = 0;
	parts3 = 0;

	mask1 = 0;
	mask2 = 0;
	mask3 = 0;

	if (bits1){
		parts1 = 1 << bits1;
		mask1 = 1 << (tbits - 1);
		for (i=0; i<(bits1-1); i++)
			mask1 = (mask1 >> 1) | mask1;
	}

	if (bits2){
		parts2 = 1 << bits2;
		mask2 = 1 << (tbits - 1);
		for (i=0; i<(bits2-1); i++)
			mask2 = (mask2 >> 1) | mask2;
	}
	mask2 = mask2 >> bits1;
	parts21 = parts2 - 1;

	if (bits3){
		parts3 = 1 << bits3;
		mask3 = 1 << (tbits - 1);
		for (i=0; i<(bits3-1); i++)
			mask3 = (mask3 >> 1) | mask3;
	}
	mask3 = mask3 >> (bits1 + bits2);
	parts31 = parts3 - 1;

	last_val = 0;
	ival = 0;
	ictr = 0;
	for(i=0; i<(MAX_VAL); i++){
		cur_val = i & mask1;
		if (cur_val != last_val){
			ictr++;
		ival = (int)((float)MAX_VAL1*((float)ictr/(float)(parts1-1)));
		}
		palette[i][0] = (unsigned char)ival;
		last_val = cur_val;
	}

	last_val = 0;
	ival = 0;
	ictr = 0;
	for(i=0; i<(MAX_VAL); i++){
		cur_val = i & mask2;
		if (cur_val != last_val){
			ictr++;
			if (ictr > parts21)
				ictr = 0;
		ival = (int)((float)MAX_VAL1*((float)ictr/(float)(parts2-1)));
		}
		palette[i][1] = (unsigned char)ival;
		last_val = cur_val;
	}

	last_val = 0;
	ival = 0;
	ictr = 0;
	for(i=0; i<(MAX_VAL); i++){
		cur_val = i & mask3;
		if (cur_val != last_val){
			ictr++;
			if (ictr > parts31)
				ictr = 0;
		ival = (int)((float)MAX_VAL1*((float)ictr/(float)(parts3-1)));
		}
		palette[i][2] = (unsigned char)ival;
		last_val = cur_val;
	}
	

error:
#if RTE_PRERR
	err_msg("make_rgbpal",err_ret);
#endif
	return(err_ret);
}
