/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <sys/poll.h>

typedef struct _GTimeVal GTimeVal;
typedef struct _GSourceFuncs GSourceFuncs;
typedef struct _GIORiver GIORiver;
typedef struct _GIOFuncs GIOFuncs;
typedef struct _GPollFD GPollFD;

/* Opaque structure */
typedef struct _GMainLoop GMainLoop;

/* Types */

struct _GTimeVal {
  glong tv_sec;
  glong tv_usec;
};

struct _GSourceFuncs {
  gboolean (*prepare)  (gpointer  source_data, 
			GTimeVal *current_time,
			gint     *timeout);
  gboolean (*check)    (gpointer  source_data,
			GTimeVal *current_time);
  gboolean (*dispatch) (gpointer  source_data, 
			GTimeVal *current_time,
			gpointer  user_data);
  GDestroyNotify destroy;
};

typedef gboolean (*GSourceFunc) (gpointer        data);



typedef enum {
  G_IO_ERROR_NONE,
  G_IO_ERROR_AGAIN,
  G_IO_ERROR_INVAL,
  G_IO_ERROR_UNKNOWN
} GIOError;

typedef enum {
  G_SEEK_CUR,
  G_SEEK_SET,
  G_SEEK_END
} GSeekType;

typedef enum {
  G_IO_IN  = POLLIN,
  G_IO_OUT = POLLOUT,
  G_IO_PRI  = POLLPRI,

  G_IO_ERR = POLLERR,
  G_IO_HUP = POLLHUP,
  G_IO_NVAL = POLLNVAL
} GIOCondition;

typedef gboolean (*GIOFunc) (GIORiver       *source,
			     GIOCondition condition,
			     gpointer        data);

struct _GIOFuncs {
  GIOError (*io_read)   (GIORiver *channel, 
		         gchar      *buf, 
		         guint       count,
			 guint      *bytes_read);
  GIOError (*io_write)  (GIORiver *channel, 
		 	 gchar      *buf, 
			 guint       count,
			 guint      *bytes_written);
  GIOError (*io_seek)   (GIORiver   *channel, 
		 	 gint        offset, 
		  	 GSeekType   type);
  void (*io_close)      (GIORiver *channel);
  guint (*io_add_watch) (GIORiver      *channel,
			 gint           priority,
			 GIOCondition   condition,
			 GIOFunc        func,
			 gpointer       user_data,
			 GDestroyNotify notify);
  GDestroyNotify io_free;
};

struct _GIORiver {
  gpointer channel_data;
};

struct _GPollFD {
   gint fd;
   gushort events;
   gushort revents;
};

typedef gint (*GPollFunc) (GPollFD *ufds, guint nfsd, gint timeout);

/* Hooks for adding to the main loop */

guint g_source_add                  (gint           priority, 
				     GSourceFuncs  *funcs,
				     gpointer       source_data, 
				     gpointer       user_data,
				     GDestroyNotify notify);
void g_source_remove                (guint          tag);
void g_source_remove_by_user_data   (gpointer       user_data);
void g_source_remove_by_source_data (gpointer       source_data);


void g_get_current_time (GTimeVal *result);

/* Running the main loop */

GMainLoop *g_main_new ();
void g_main_run (GMainLoop *loop);
void g_main_quit (GMainLoop *loop);
void g_main_destroy (GMainLoop *loop);

/* Run a single iteration of the mainloop. If block is FALSE,
 * will never block
 */
void g_main_iteration (gboolean block);

/* See if any events are pending
 */
gboolean g_main_pending ();




/********************************/

/* Low level unix-centric file poll framework */

GPollFD *g_main_poll_add      (gint   priority, 
			       gint   fd, 
			       gshort events);
void     g_main_poll_remove   (GPollFD *fd);

void     g_main_set_poll_func (GPollFunc func);

 
/***************
 * IO Channels *
 ***************/

GIORiver *g_io_channel_new      (GIOFuncs      *funcs,
			         gpointer       channel_data);
void      g_io_channel_ref      (GIORiver      *channel);
void      g_io_channel_unref    (GIORiver      *channel);
GIOError  g_io_channel_read     (GIORiver      *channel, 
			         gchar         *buf, 
			         guint          count,
			         guint         *bytes_read);
GIOError  g_io_channel_write    (GIORiver      *channel, 
			         gchar         *buf, 
			         guint          count,
			         guint         *bytes_written);
GIOError  g_io_channel_seek     (GIORiver      *channel,
			         gint           offset, 
			         GSeekType      type);
void      g_io_channel_close    (GIORiver      *channel);
guint     g_io_add_watch_full   (GIORiver      *channel,
			         gint           priority,
			         GIOCondition   condition,
			         GIOFunc        func,
			         gpointer       user_data,
			         GDestroyNotify notify);
guint    g_io_add_watch         (GIORiver      *channel,
			         GIOCondition   condition,
			         GIOFunc        func,
			         gpointer       user_data);

GIORiver *g_io_channel_unix_new    (int       fd);
gint      g_io_channel_unix_get_fd (GIORiver *channel);

/* GIORiver *g_io_std_new (FILE *file); */

/**********************
 * Idles and timeouts *
 **********************/

guint g_timeout_add_full      (gint           priority,
			       guint          interval, 
			       GSourceFunc    function,
			       gpointer       data,
			       GDestroyNotify notify);
guint g_timeout_add           (guint          interval,
			       GSourceFunc    function,
			       gpointer       data);

guint	   g_idle_add		 (GSourceFunc          function,
				  gpointer	       data);
guint	   g_idle_add_full       (gint   	       priority,
				  GSourceFunc	       function,
				  gpointer	       data,
				  GDestroyNotify       destroy);

