/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvthd.c
 >>>>
 >>>>      Program Name: vthd
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:17:31 1991 
 >>>>
 >>>>          Routines: lvthd - the library call for vthd
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvthd - library call for vthd
*
* Purpose:
*    
*    Compute total harmonic distortion from FFT spectrum
*    
*    
* Input:
*    
*    image          pointer to the input image
*    
*    type"          indicates what the spectrum type of "image" is:
*    
*                                          1  - amplitude spectrum
*                                          2  - power spectrum
*    
*    thd            pointer to a float variable to  receive  the  com-
*                   puted THD
*    
*    
* Output:
*    
*    Return Value:  1 on success, 0 on failure.
*    
*    
*
* Written By: Scott Wilson, Mike Murillo
*    
*    
****************************************************************/


/* -library_def */
int
lvthd(image,type,thd)
struct xvimage *image;
int type;
float *thd;
/* -library_def_end */

/* -library_code */
{
  int rows,cols,i,j,rowsd2,colsd2;
  float *f,x,fmax,sum,ftmp;
  int imax,jmax,dx,dy;

  if (image->data_storage_type != VFF_TYP_FLOAT)
    {
      fprintf(stderr,"vthd: Image not a spectra!\n");
      return(0);
    }

  rows = image->col_size;
  cols = image->row_size;
  rowsd2 = rows/2;
  colsd2 = cols/2;

  /* Hunt down highest non-DC component, assume its the fundamental */
  fmax = -1.0e+38;
  f = (float *)(image->imagedata);
  ftmp = f[rowsd2*cols+colsd2];          /* Save the DC level */
  f[rowsd2*cols+colsd2] = 0.0;           /* Set DC to zero */
  for (i=0; i<rows; i++)                 /* Look for highest component */
    for (j=0; j<(colsd2+1); j++)
      if (f[j*cols+i] > fmax)
        {
          fmax = f[j*cols+i];
          imax = i;
          jmax = j;
        }

  if (fmax == 0.0)
    {
      fprintf(stderr,"vthd: Spectrum has no power!\n");
      return(0);
    }

  dx = colsd2-imax;   /* Compute dx and dy for direction of harmonics */
  dy = rowsd2-jmax;

  /* Integrate along the higher harmonics */
  sum = 0.0;
  i = colsd2-2*dx;
  j = rowsd2-2*dy;
  while (i >= 0 && i < cols && j >= 0 && j < rows )
    {
      if (type == 1)      sum += f[j*cols+i]*f[j*cols+i];
      else if (type == 2) sum += f[j*cols+i];
      else
        {
          fprintf(stderr,"lvthd: Unknown spectrum type!\n");
          return(0);
        }
      i -= dx;  j -= dy;      /* Step to next harmonic */
    }

  if (type == 1 ) x = f[cols*jmax+imax]*f[cols*jmax+imax]; /* Get fundamental */
  if (type == 2 ) x = f[cols*jmax+imax];

  *thd = sum/x;

  f[rowsd2*cols+colsd2] = ftmp;   /* Restore the DC level */

  return(1);
}
/* -library_code_end */
