 /*
  * Khoros: $Id: lelas2viff.c,v 1.1 1991/05/10 15:43:29 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lelas2viff.c,v 1.1 1991/05/10 15:43:29 khoros Exp $";
#endif

 /*
  * $Log: lelas2viff.c,v $
 * Revision 1.1  1991/05/10  15:43:29  khoros
 * Initial revision
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lelas2viff.c
 >>>>
 >>>>      Program Name: elas2viff
 >>>>
 >>>> Date Last Updated: Sat Mar  2 14:22:07 1991 
 >>>>
 >>>>          Routines: lelas2viff - the library call for elas2viff
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
#include "file_formats/elas.h"
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lelas2viff - library call for elas2viff
*
* Purpose:
*    
*    Converts an ELAS file format to the VIFF image file format.
*    
*    
* Input:
*    
*         ELAS *elas_image - pointer to elas image structure
*    
*         int elas_fid - file descriptor
*    
*         struct  xvimage **viff_image - pointer to  a  pointer  of  a
*         VIFF image file
*    
*         int band_num - band number to process
*    
*         int x_coord - starting x location into ELAS image.
*    
*         int y_coord - starting y location into ELAS image.
*    
*         int width - desired width of VIFF image.
*    
*         int height - desired width of VIFF image.
*    
*         int color_model - the color space  model  specified  by  the
*         user.
*    
*         int subimage - flag specifying  whether  to  process  entire
*         band or only a specified portion of the band.
*    
*    
* Output:
*    
*         struct  xvimage **viff_image - malloced and copied imagedata
*         for VIFF image
*    
*    
*
* Written By: Charlie Gage
*    
*    
****************************************************************/


/* -library_def */
int lelas2viff(elas_image, elas_fid, viff_image, band_num, x_coord, y_coord, width, height, color_model, subimage)

ELAS *elas_image;
int  elas_fid;
struct  xvimage **viff_image;
int  band_num, x_coord, y_coord, width, height, color_model, subimage;
/* -library_def_end */

/* -library_code */
{
    struct  xvimage *img, *createimage();
    struct  elas_coords *elas_ptr;
    char *program = "lelas2viff";
    char *viff_img_ptr, *scan_buf;
   
    int nr, nc, bands;
    int num_recbytes;        /* number of bytes per ELAS image record */
    int num_padbytes;
    int num_skipbytes;
    int i, j, k;
    int row = 0;

    /*--------------------------------------------------
     * Determine dimensions of ELAS image 
     *-------------------------------------------------*/
    nr = (int) (elas_image->last_line - elas_image->init_line + 1);
    nc = (int) (elas_image->last_elem_scan_line - elas_image->init_elem_scan_line + 1);
    bands = (int) elas_image->num_channels;
    num_recbytes = elas_image->num_bytes_record;
    num_padbytes = (num_recbytes - nc);


    /*-----------------------------------------------------
     * If subimage is selected, determine the requested
     *  subimage dimensions from the specified width and
     *  height parameters, otherwise use the dimensions of
     *  the ELAS image for the resulting VIFF image.
     *----------------------------------------------------*/
    if (subimage)
    {
      num_skipbytes = nc - (width + x_coord);
    }
    else
    {
      num_skipbytes = 0;
      height = nr;
      width = nc;
      x_coord = 0;
      y_coord = 0;
    }
     
    /*-------------------------------------------------------------------
     * ERROR checking:
     *  If subimage coordinates are specified, ensure that all four 
     *  subimage parameters are valid and within the ELAS image bounds.
     *-------------------------------------------------------------------*/
  if (subimage)
  {
   if (x_coord  > nc )
    {
     (void) fprintf(stderr,"%s: input x-coordinate out of bounds.\n", program);
     (void) fprintf(stderr,"ELAS image width = %d bytes.\n", nc);
     close(elas_fid);
     return(0);
    }
   if (y_coord  > nr )
    {
     (void) fprintf(stderr,"%s: input y-coordinate out of bounds.\n", program);
     (void) fprintf(stderr,"ELAS image height = %d bytes.\n", nr);
     close(elas_fid);
     return(0);
    }
   if ( width > nc )
    {
     (void) fprintf(stderr,"%s: requested width exceeds ELAS image bounds.\n", program);
     (void) fprintf(stderr,"Use smaller x-coordinate and/or width values.\n");
     close(elas_fid);
     return(0);
    }
   if ( height > nr )
    {
     (void) fprintf(stderr,"%s: requested width exceeds ELAS image bounds.\n", program);
     (void) fprintf(stderr,"Use smaller x-coordinate and/or width values.\n");
     close(elas_fid);
     return(0);
    }
   if (band_num > bands)
    {
     (void) fprintf(stderr,"%s: requested band number exceeds\n", program);
     (void) fprintf(stderr,"the number of channels in the ELAS image.\n");
     close(elas_fid);
     return(0);
    }
  }

    /*----------------------------------------------------------------
     * Allocate space for VIFF output image data, based on
     * requested size and fill in the VIFF header info
     *-------------------------------------------------------------- */
   img = createimage( (unsigned long) height,          /* number of rows */
                      (unsigned long) width,           /* number of columns */
                      (unsigned long) VFF_TYP_1_BYTE,  /* data_storage_type */
                      (unsigned long) 1,               /* num_of_images */
                      (unsigned long) 1,               /* num_data_bands */
                      elas_image->comments,            /* comment */
                      (unsigned long) 0,               /* map_row_size */
                      (unsigned long) 0,               /* map_col_size */
                      (unsigned long) VFF_MS_NONE,     /* map_scheme */
                      (unsigned long) VFF_MAPTYP_NONE, /* map_storage_type */
                      (unsigned long) VFF_LOC_IMPLICIT, /* location_type */
                      (unsigned long) 0);               /* dimension */

   if ( img == NULL )
   {
       (void)fprintf(stderr,"%s: Unable to allocate new image!\n",program);
       close(elas_fid);
       return(0);
   }
   *viff_image = img;

    /*-----------------------------------------------------------------
     * Allocate space for one scan line of elas image,
     * according to requested width.  (a line buffer)
     *---------------------------------------------------------------- */
   scan_buf = (char *) malloc((unsigned int) width * sizeof(char));
   if (scan_buf == NULL)
   {
     (void) fprintf(stderr,"%s: can't allocate memory for scanline buffer\n", program);
     close(elas_fid);
     return(0);
   }

    /*------------------------------------------------------
     * Allocate space for elas_coord structure
     *----------------------------------------------------- */
   elas_ptr = (ECOORD *) malloc(sizeof(ECOORD));
   if (elas_ptr == NULL)
   {
     (void) fprintf(stderr,"%s: can't allocate memory for elas coords\n", program);
     close(elas_fid);
     return(0);
   }

     /* Put the data in VIFF format -
      *  Load the data according to the VIFF format.
      *  Note, ELAS file format uses band-interleaved-by-line and
      *  pads to 512 byte boundaries.
      */

   /* Assign image coordinate information to elas_coord structure */
   elas_ptr->x_init = x_coord;
   elas_ptr->skip = num_skipbytes;
   elas_ptr->wid = width;
   elas_ptr->rec_length = num_recbytes;
   elas_ptr->pad = num_padbytes;
   elas_ptr->fid = elas_fid;
   elas_ptr->buf = scan_buf;
   
   viff_img_ptr = img->imagedata;          /* pointer to viff image data */

   /*--------------------------------------------------------------
    * Read ELAS image from disk, one scan line at a time.
    *  Determine which channel to read and seek into ELAS image
    *  to the requested starting point.  That is, seek to the
    *  appropriate channel and row in the ELAS image, then call
    *  read_elas_line() with a structure containing starting coordinate
    *  information and a scan line buffer.  The appropriate ELAS
    *  image data is returned in the scan line buffer of the structure.
    *  This image data is then stored to a VIFF image structure in
    *  memory.  Note that only one channel of the ELAS image is
    *  processed.  A future enhancement would be to process all
    *  requested channels.
    *--------------------------------------------------------------*/

  for (i=0; i < nr; i++)
  {
    for (j=0; j < bands; j++)
    {
      if ( (j == band_num) && (i >= y_coord) && (row < height) )
      {
        if (! read_elas_line(elas_ptr))
         {
           (void)fprintf(stderr,"read_elas_line: unable to read scan line\n");
           close(elas_fid);
           return(0);
         }
        for (k=0; k < width; k++)  /* store scan line data in VIFF image */
         {
           viff_img_ptr[row * width + k] = elas_ptr->buf[k];
         }
         row++;         /* count number of rows processed */
      }
      else                  /* skip record of ELAS image */
        lseek(elas_fid, num_recbytes, 1);
    }
  }

      /* set the color space model as the user specified */
    set_color_space_model(img, color_model); 

    close(elas_fid);
    return(1);
}

/* -library_code_end */
