 /*
  * Khoros: $Id$
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1992, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1992 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvfdmax.c
 >>>>
 >>>>      Program Name: vfdmax
 >>>>
 >>>> Date Last Updated: Thu Mar 26 10:39:58 1992 
 >>>>
 >>>>          Routines: lvfdmax - the library call for vfdmax
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
#define WMAX 15

/* This is to find sub_pixel maxi, see vgetqtilt.h and vgetqtilt.prog */
typedef struct {
       double  x2slope, y2slope ,xyslope, xslope,
               yslope, piston, min, mean, max;
       int     nb_points;} STRUCT_TILT;


/* for direct H. input xy image: */
#define INIT_IMA_TYP     unsigned char
#define INIT_VVF_TYP     VFF_TYP_1_BYTE

/* for direct H. output and inverse H. input az image: */
#define IMA_TYP          unsigned long
#define VVF_HOUGH_TYP    VFF_TYP_4_BYTE

/* for inverse H. output xy image: */
#define IMA_INV_TYP      unsigned long
#define OUT_INV_VVF_TYP  VFF_TYP_4_BYTE
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvfdmax - library call for vfdmax
*
* Purpose:
*    
*    Finds maxima in the Fast Hough  domain  (see  vhough(3)  and
*    gives  one point result for each maximum in output image and
*    sub-pixel results for ascii file
*    
*    

* Input:
* Output:
*
* Written By: Jean-Pierre Guerin, Jean-Francois Larue, Marc Viala
****************************************************************/


/* -library_def */
int lvfdmax(inp_image_acc, out_image_max, data3, wd_size,
            threshold, accumPrint, print_dev )
struct xvimage *inp_image_acc, *out_image_max;
IMA_TYP        **data3;
int            wd_size, threshold;
char           accumPrint;
FILE           *print_dev;

/* -library_def_end */

/* -library_code */

/******************************************************************
 * lvfdMax()                                                      *
 * This routine finds maxima in the Hough accumulator. All pixels in
 * the Hough accumulator which are not maxima are cleared to zero.*
 * Maxima are left untouched. If the flag accumPrint is set,      *
 * information about the maxima is written to the file output.    *
 * The threshold parameter is used to threshold the peaks         *
 ******************************************************************/

{
    struct xvimage *window_image;

    int a_off, z_off, z_f, a_f, size, frame_off, size_frame, max_found;

    IMA_TYP   cellPeak, a, b, c, d, f, g, h, i ;
    IMA_TYP   *inp_acc, *iinp_acc, *out_acc, *iout_acc,
              *frame_acc, *iframe_acc, *window_acc, *iwindow_acc;
    FILE      *output=stderr;

    double    a_dbl, b_dbl, c_dbl, d_dbl, e_dbl, maxi, 
              a_subpx_max, z_subpx_max, delta;

/* INIT_IMA_TYP  * ...; */

    if ( (wd_size & 1) == 0 ) /* odd? */
    {
      wd_size++;
      (void)fprintf(stderr,
                    "lvfdmax : warning, window size must be odd, will be %d\n", 
                    wd_size );
    }
    if ( wd_size < 3 )
    {
      wd_size = 3;
      (void)fprintf(stderr,
                    "lvfdmax: warning, window size too small, will be %d \n", 
                    wd_size );
    }
    else if ( wd_size > WMAX )
    {
      wd_size = WMAX;
      (void)fprintf(stderr,
                    "lvfdmax : warning, window size too big, will be %d \n", 
                    wd_size );
    }
 
    if ( inp_image_acc->col_size != inp_image_acc->row_size )
    {
        (void)fprintf(stderr,
"lvfdmax : Input Image must be square (col_size = row_size) ...\n");
        return(0);
    }

    if ( (inp_image_acc->data_storage_type) != VVF_HOUGH_TYP)
    {
        (void)fprintf(stderr,
                      "lvfdmax : no proper format for Input Image ...\n");
        return(0);
    }

    size    = (int)      inp_image_acc->row_size;
    inp_acc = (IMA_TYP *)inp_image_acc->imagedata;
    out_acc = (IMA_TYP *)out_image_max->imagedata;

/*
... transmit the startx, startxy infos.
*/

    out_image_max->startx = inp_image_acc->startx;
    out_image_max->starty = inp_image_acc->starty;

/*
... get a new buffer with a frame around (wd_size/2).
*/
    frame_off  = (wd_size-1)/2;
    size_frame = size + wd_size-1;
    frame_acc  = (IMA_TYP *)malloc( size_frame*size_frame*sizeof(IMA_TYP) );
    if ( frame_acc == NULL)
    {
        (void)fprintf(stderr,"lvfdmax : no memory available\n");
        return(0);
    }

/*
... get a window image, wd_size * wd_size.
*/
    window_image = createimage(wd_size, wd_size, 
                     VVF_HOUGH_TYP, 1,
                     1,"", 0, 0, VFF_MS_NONE,
                     VFF_MAPTYP_NONE, VFF_LOC_IMPLICIT, 0);
   
    window_acc = (IMA_TYP *)window_image->imagedata;

/* Copy of inp_acc in temp_acc */
    iinp_acc  = inp_acc;
    iout_acc =  out_acc;
    for (z_off=0; z_off<size; z_off++)
       {
       for (a_off=0; a_off<size; a_off++)
         {
         *iout_acc++ = *iinp_acc++;
         }
       }

/* Clear frame_acc */
    iframe_acc = frame_acc;
    for (z_off=0; z_off<size_frame; z_off++)
    {
       for (a_off=0; a_off<size_frame; a_off++)
       {
          *iframe_acc++ = 0;
       }
    }

/* Copy of inp_acc in middle of frame_acc */
    iinp_acc  = inp_acc;
    for (z_off=0; z_off<size; z_off++)
    {
       iframe_acc = frame_acc + frame_off + size_frame*(frame_off+z_off);
       for (a_off=0; a_off<size; a_off++)
       {
          *iframe_acc++ = *iinp_acc++;
       }
    }

/* copy right part of image to left frame border (Top <==> Bottom) */
    for (z_off=0; z_off<size; z_off++)
    {
       iframe_acc = frame_acc + size_frame*(frame_off+z_off);
       iinp_acc   = inp_acc + (size*(size-z_off) - frame_off);
       for (a_off=0; a_off<frame_off; a_off++)
       {
          *iframe_acc++ = *iinp_acc++;
       }
    }

/*  copy left part of image to right frame border (Top <==> Bottom) */
    for (z_off=0; z_off<size; z_off++)
    {
       iframe_acc = frame_acc + (size_frame*(frame_off+z_off+1) - frame_off);
       iinp_acc   = inp_acc + size*(size-z_off-1);
       for (a_off=0; a_off<frame_off; a_off++)
       {
          *iframe_acc++ = *iinp_acc++;
       }
    }

/*
... find not sub-pixel maxima
*/
    for (z_off=0; z_off<size; z_off++)
    {
       for (a_off=0; a_off<size; a_off++)
       {
          iout_acc = out_acc + a_off + size*z_off;

          iframe_acc = frame_acc + 
                       (frame_off+a_off) + 
                       size_frame * (frame_off+z_off);

          cellPeak = *iframe_acc;

          max_found=TRUE;
          if (cellPeak<threshold)
          {
             *iout_acc = 0;
             max_found = FALSE;
          }
          else
          {
            iwindow_acc = window_acc;
            for (z_f=0; z_f<wd_size; z_f++)
            {
              for (a_f=0; a_f<wd_size; a_f++, iwindow_acc++)
              {
                *iwindow_acc = *(iframe_acc + 
                                 a_f -
                                 frame_off + 
                                 (z_f -frame_off)*size_frame);

                if ( *iwindow_acc > cellPeak )
                {
                  *(iout_acc) = 0;
                  max_found = FALSE;
                }
              }
            }
          }

          if (max_found)       /* calculate sub_pixel precision maximum */
          {
             if ( ! find_sub_pxl_max2(window_image, 
                                      &a_subpx_max, 
                                      &z_subpx_max, 
                                      &maxi))
             {
                (void) fprintf(stderr,
"lvfdmax: warning, find_sub_pxl_max2 failed, I try parabole method\n");
                a_subpx_max = 0;
                z_subpx_max = 0;
                maxi  = cellPeak;
             }

             if ( ! find_sub_pxl_max(window_image, 
                                     &a_subpx_max, 
                                     &z_subpx_max, 
                                     &maxi))
             {
                (void) fprintf(stderr,
"lvfdmax: warning, find_sub_pxl_max failed, no sub-pixel result\n");
                a_subpx_max = 0;
                z_subpx_max = 0;
                maxi  = cellPeak;
             }
             maxi  = cellPeak;

             /* check if max is still within window (weak max) */
             if (   (a_subpx_max < -wd_size/2.)   || 
                    (a_subpx_max >= wd_size/2.)   ||
                    (z_subpx_max < -wd_size/2.)   || 
                    (z_subpx_max >= wd_size/2.)   || 
                    (a_subpx_max+a_off < 0.   )   || 
                    (a_subpx_max+a_off >= size-1) || 
                    (z_subpx_max+z_off < 0.)      || 
                    (z_subpx_max+z_off >= size-1) )
             {
/*
 *                max_found = FALSE;
 */
                (void) fprintf(stderr,
    "lvfdmax: warning, max outside of window: no sub-pixel result\n");
                a_subpx_max = 0;
                z_subpx_max = 0;
                maxi  = cellPeak;
             }

             if ( (print_dev!=NULL) && max_found )
                 (void) fprintf(print_dev,
                                " %f %f %f\n",
                                (double)(a_subpx_max+a_off), 
                                (double)(z_subpx_max+z_off), 
                                maxi);

             if ( (accumPrint) && max_found)
             {
                (void) fprintf(stderr,
                               "window center: a_slope=%d, z=%d, max=%d\n",
                               a_off, 
                               z_off, 
                               cellPeak );
                (void) fprintf(stderr,
                "subpixel precision:  a_slope= %.3f, z=%.3f, max=%.3f\n",
                (double)(a_subpx_max+a_off), 
                (double)(z_subpx_max+z_off),  
                 maxi);

                iwindow_acc = window_acc;
                for (z_f=0; z_f<wd_size; z_f++)
                {
                   for (a_f=0; a_f<wd_size; a_f++, iwindow_acc++)
                   {
                      fprintf(output," %5d ", *iwindow_acc);
                   }
                   fprintf(output,"\n");
                }
             fprintf(output,"\n");
             }    /* end accumPrint */
          }       /* end max_found */
       }          /* end a_off */
    }             /* end z_off */
    *data3 = frame_acc;
    (void)free(window_acc);

    return(1);
}

/**************************************************************
*
* ROUTINE NAME: find_sub_pxl_max2
*
*     PURPOSE: Compute sub-pixel positions and value of maxima in
*        fast hough domain a,z.
*        The maximum is inside a window.
* note:  this routine often gives maximum too far (outside the window).
*        and is changed by find_sub_pxl_max
*
*        We use the lvgetqtilt routine and find the surface "center"
*
*       INPUT:
*              IMA_TYP *window;
*
*      OUTPUT: 
*              double *a_subpx_max, *z_subpx_max, *val_max 
*
*              find_sub_pxl_max2 will return a zero if failure and a
*              one upon success.
*
* ROUTINES CALLED: lvgetqtilt()
*
**************************************************************/

int find_sub_pxl_max2(window_image, a_subpx_max, z_subpx_max, val_max)

struct xvimage *window_image;
double *a_subpx_max, *z_subpx_max, *val_max;

{
    STRUCT_TILT   *q_par;      /* allocated and computed by lvgetqtilt */
    int           nb_str;      /* computed by lvgetqtilt */
    int           no_str;

    double    det, x, y, half_size;

  if (TRUE)
  {
    *z_subpx_max = 0; 
    *a_subpx_max   = 0;
    *val_max = 255;
    return(1);  
  }

/* no good enough results and uses NEW lvgetqtilt: suppress comments marks */

/*    if (! lvgetqtilt(window_image,NULL,NULL,
 *                     FALSE, FALSE,
 *                     &nb_str, &q_par) )
 *    {
 *      (void)fprintf(stderr,"find_sub_pxl_max2: lvgetqtilt failed!\n");
 *      return(0);
 *    }
*/

    half_size = (window_image->col_size -1)/2.0;

    if (nb_str!=1)
    {
       (void) fprintf(stderr,"lvdeqtilt: Warning, no subpixel precision\n");
       *z_subpx_max = 0; 
       *a_subpx_max   = 0;
       *val_max = 255;
       return(1);
    }

    /* find the center: */
    det = q_par->xyslope * q_par->xyslope 
          - 4.0 * q_par->x2slope * q_par->y2slope;

    x = (2.0*q_par->y2slope*q_par->xslope - q_par->yslope*q_par->xyslope)/det; 
    *a_subpx_max = x;

    y = (2.0*q_par->x2slope*q_par->yslope - q_par->xslope*q_par->xyslope)/det;

    *z_subpx_max = y;
    *val_max = q_par->x2slope*x*x +
               q_par->y2slope*y*y +
               q_par->xyslope*x*y +
               q_par->xslope*x +
               q_par->yslope*y +
               q_par->piston;

    (void)free(q_par);

    return(1);
}

/**************************************************************
*
* ROUTINE NAME: find_sub_pxl_max
*
*     PURPOSE: Compute sub-pixel positions and value of maxima in fast hough
*        domain a,z. (a is x axis and z is y axis)
*        The maximum is inside a window.
*        A least-squares method gives best fit parameters of a family of
*        paraboles for a vector along plans in the z axis:
*
*        Along the other axis (a axis), we find maximum of alpha
*        
*        Each parabole will be given by
*                f(val) = alpha*z*z + beta*x + gamma
*
*       INPUT:
*              int *window; int size ; 
*
*      OUTPUT: 
*              double *a_subpx_max, *z_subpx_max, *val_max 
*
*              find_sub_pxl_max will return a zero if failure and a
*              one upon success.
*         All the computation are done with origine in 1st corner but
*         results are given for the center point
*
*
* ROUTINES CALLED: lvgetpar, dgeco_(),  dgesl_(), Double_Epsilon()
*
**************************************************************/

int find_sub_pxl_max(window_image, a_subpx_max, z_subpx_max, val_max)
struct xvimage *window_image;

double *a_subpx_max, *z_subpx_max, *val_max;

{
   /* paraboles params : see above */
   double a_of_max, z_of_max, max, det, alpha1, beta1, gamma1;
   /* vectors: */
   double *alpha, *beta, *gamma, *wide, *z_subpx_max_pos, *val_m;

   double *a_array, *sig_array;
   double *vect, *ivect;
   IMA_TYP    *iwindow;
   int    i, ia;
   int    mwt,  size;
   double a_coeff,b_coeff,siga,sigb,chi2,q, half_size;

   *z_subpx_max = 0; 
   *a_subpx_max   = 0;
   *val_max = 255;

   size = window_image->col_size;
   half_size = (size-1.0)/2.0;

   vect = (double *)malloc(size*sizeof(double));
   if (vect==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   alpha = (double *)malloc(size*sizeof(double));
   if (alpha==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   beta = (double *)malloc(size*sizeof(double));
   if (beta==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   gamma = (double *)malloc(size*sizeof(double));
   if (gamma==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   wide = (double *)malloc(size*sizeof(double));
   if (wide==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   z_subpx_max_pos = (double *)malloc(size*sizeof(double));
   if (z_subpx_max_pos==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   val_m = (double *)malloc(size*sizeof(double));
   if (val_m==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   a_array = (double *)malloc(size*sizeof(double));
   if (a_array==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }
   sig_array = (double *)malloc(size*sizeof(double));
   if (sig_array==NULL) 
   {
      (void) fprintf(stderr,"\n error: no memory for find_sub_pxl_max\n");
      exit(1);
   }

   for (ia =0; ia<size ; ia++) /* get params of next parabole */
   {
       a_array[ia] = ia;
       ivect = vect;
       iwindow = (IMA_TYP *)window_image->imagedata+ia;
       for (i=0; i<size ; i++, iwindow=iwindow+size, ivect++ )
       {
          (double)(*ivect) = (IMA_TYP)(*iwindow); /* copy vals, verified OK */
       }

       lvgetpar(vect, size, &alpha[ia], &beta[ia], &gamma[ia]);
       z_subpx_max_pos[ia] = -beta[ia]/(2.0 * alpha[ia]); /* "z" maxi's position */
       val_m[ia]     = gamma[ia] - beta[ia]*beta[ia]/(4.0*alpha[ia]);

       det = beta[ia]*beta[ia] - 4.0*alpha[ia]*gamma[ia];
       if (det>=0)
       {
           wide[ia] = sqrt(det)/alpha[ia];
       }
       else
       {
           wide[ia] = half_size;
           (void)fprintf(stderr,"find_sub_pxl_max: warning, det<0 \n");
       }
   }

   /* now, process along "a" axis */

   lvgetpar(wide, size, &alpha1, &beta1, &gamma1 );
   a_of_max = -beta1/(2.0 * alpha1); /* "a" maxi's position */
   lvgetpar(val_m, size, &alpha1, &beta1, &gamma1  );
   a_of_max = -beta1/(2.0 * alpha1);  /* "a" maxi's position */
   max = gamma1 - beta1*beta1/(4.0*alpha1); /* its maxi's value */
   lvgetpar(alpha, size, &alpha1, &beta1, &gamma1 );
   a_of_max = -beta1/(2.0 * alpha1); /* "a" maxi's position */

/* fit: Given a set of points x[0..ndata-1], y[0..ndata-1] fit them to a
 *      straight line y = a + bx, by minimizing chi square.
 *      Returned are a, b.
 */
   fit(a_array,z_subpx_max_pos,size,NULL,0,&a_coeff,&b_coeff,&siga,&sigb,&chi2,&q);
   *z_subpx_max = a_coeff + b_coeff*a_of_max - half_size; 
   *a_subpx_max   = a_of_max - half_size;
   *val_max = max;

   (void)free(alpha);
   (void)free(beta);
   (void)free(gamma);
   (void)free(wide);
   (void)free(z_subpx_max_pos);
   (void)free(val_m);
   (void)free(vect);
   (void)free(a_array);
   (void)free(sig_array);

    return(1);  /* success */
}


/**************************************************************
*
* ROUTINE NAME: lvgetpar
*
*     PURPOSE: Compute the least-squares best fit parameters of a 
*                parabole for a vector.
*                The parabole will be given by
*                f(z) = ax*x + bx + C
*
*       INPUT:
*              double *vector; int size ; 
*
*      OUTPUT: 
*              double a,b,c 
*
*              lvgetpar will return a zero if failure and a
*              one upon success.
*
* ROUTINES CALLED:     dgeco_(),  dgesl_(), Double_Epsilon()
*
**************************************************************/

int lvgetpar(vector,size, a,b,c)

double    *vector;
int       size;
double    *a,*b,*c;
  {
    double x,x2,x3,x4,xz,x2z,z, det,deta,detb,detc, ixi;
    int i;
    double *vect;

    /* for Gauss method: */
#define DIM_X 3

    double matrix[DIM_X][DIM_X],
           vect_work[DIM_X],
           vect_result[DIM_X],
           cond, epsilon;
    int    job, matrix_size, vect_pivot[DIM_X];
    void   dgeco_(), dgesl_()   ; /* LINPACK Algo. */
    double Double_Epsilon();


    /* sums: */
    x  = 0.0; x2  = 0.0; x3 = 0.0; x4 = 0.0;
    xz = 0.0; x2z = 0.0; z  = 0.0;

    if (size <= 2)
      {
        (void) fprintf(stderr,"lvgetpar: Cannot find param on small vector\n");
        return(0);
      }

    vect = vector;    
    for (i=0; i<size; i++, vect++)
    { 
        ixi = i * i;
        x  += i;
        x2 += ixi;
        x3 += ixi * i;
        x4 += ixi * ixi;
        x2z += ixi * (*vect);
        xz += i * (*vect);
        z  += *vect;
    }

    matrix[0][0]= x4;
    matrix[1][1]= x2;
    matrix[2][2]= size;

    matrix[0][1]= matrix[1][0]= x3;
    matrix[0][2]= matrix[2][0]= x2;
    matrix[1][2]= matrix[2][1]= x;

    vect_result[0] = x2z;  /* sign - * - = + */
    vect_result[1] = xz;
    vect_result[2] = z;

    epsilon = Double_Epsilon();

    matrix_size = DIM_X;
    dgeco_(matrix, &matrix_size, &matrix_size, vect_pivot, &cond, vect_work);
    if( fabs(cond) <= epsilon )
    {
        (void)fprintf(stderr,
                      "\nlvgetpar:Warning, condition for dgeco_ too small\n");
        return(0);  /* failed */
    }

    job = 0;     /* to solve  a*x = b */
    dgesl_(matrix, &matrix_size, &matrix_size, vect_pivot, vect_result, &job);

    *a = vect_result[0];
    *b = vect_result[1];
    *c = vect_result[2];

    return(1);  /* success */
  }

/*************************************************************************
 * Proc : Double_Epsilon()
 *       Estimate unit roundoff error (see ratqr.f of eigpack).
 *************************************************************************/
 
  double Double_Epsilon()
  {
        double eps, a, c, b;

        a = (double)4.0/(double)3.0;
        do
        {
                b = a - (double)1.0;
                c = b + b + b;
                eps = fabs( c - (double)1.0 );
        }
        while( eps == 0.0 );
        return(eps);
 }

/* end Proc : Double_Epsilon() ***********************************************/


/* -library_code_end */
