 /*
  * Khoros: $Id$
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1992, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1992 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvhough.c
 >>>>
 >>>>      Program Name: vhough
 >>>>
 >>>> Date Last Updated: Thu Mar 26 10:42:38 1992 
 >>>>
 >>>>          Routines: lvhough - the library call for vhough
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */


/* for direct H. input xy image: */
#define INIT_IMA_TYP     unsigned char
#define INIT_VVF_TYP     VFF_TYP_1_BYTE

/* for direct H. output and inverse H. input az image: */
#define IMA_TYP          unsigned long
#define VVF_HOUGH_TYP    VFF_TYP_4_BYTE

/* for inverse H. output xy image: */
#define IMA_INV_TYP      unsigned long
#define OUT_INV_VVF_TYP  VFF_TYP_4_BYTE

/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvhough - library call for vhough
*
* Purpose:
*    
*    Fast Hough (Grey level) and Inverse Transforms.
*    
*    

* Input:
* Output:
*
* Written By: Patrick Arnoul, Jean-Pierre Guerin, Jean-Francois Larue
****************************************************************/


/* -library_def */
int lvhough(image1, image2, direct_fg, draw_fg, norm_fg)
struct xvimage *image1, *image2;
int    direct_fg, draw_fg, norm_fg;

/* -library_def_end */

/* -library_code */
{
    if ( image1->col_size != image1->row_size )
        {
        (void)fprintf(stderr,"lvhough : Input Image must be square (col_size = row_size) ...\n");
        return(0);
        }

    if ( direct_fg == 1 )                 /* do forward transform */
        return(lvhough_direct(image1, image2, draw_fg, norm_fg));
    else
        return(lvhough_inv(image1, image2, draw_fg, norm_fg));
}

/*********************************************
 * Fast Forward (Grey level) Hough Transform *
 *********************************************/
int lvhough_direct(image_xy,image_az,draw_fg, norm_fg)
struct xvimage *image_xy, *image_az;
int    draw_fg, norm_fg;  /* draw_fg is not used */

{
#define BORCOL  255
#define LINCOL  255

int x,y,x_off,y_off;

int a,b,c,z,a_off,z_off, pix_count;

int size, half_size;

double x_double, y_double, inc_double, fract, pix_tmp;

INIT_IMA_TYP  *image_pointer_xy;
INIT_IMA_TYP  *pixel_pointer_xy;
IMA_TYP       *image_pointer_az;
IMA_TYP       *pixel_pointer_az;

    size      = (int)image_xy->row_size;
    half_size = size/2;
    image_pointer_xy = (INIT_IMA_TYP *)image_xy->imagedata;
    image_pointer_az = (IMA_TYP *)image_az->imagedata;

    if ( (image_xy->data_storage_type) != INIT_VVF_TYP)
        {
        (void)fprintf(stderr,"lvhough : no proper format for Input Image ...\n");
        return(0);
        }

/*
... transmit the startx, startxy infos.
*/

image_az->startx = image_xy->startx;
image_az->starty = image_xy->starty;

/*
 * ... clear up the whole image_az (no usefull).
 * 
 * 
 * pixel_pointer_az = (IMA_TYP *)image_az->imagedata;
 * 
 * for ( y = 0 ; y < rowsxcols ; y++ )
 *         *pixel_pointer_az++ = 0;
 */

/*
... Process Forward Hough Transform
*/

    for ( z = 0 ; z < size  ; z++ )
        {

        z_off = size * (int)(z);   /* offset along z axis in the image */

        /* high slope descending lines (asc imag) (right most of hough image) */
        for ( a = half_size ; a < size ; a += 2 )
            {
            a_off = ( size + a ) / 2;             /* offset along a  axis in the image */
            b = size - a ;
            pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
            c = - z*size;
            pix_count = 0;
            pix_tmp = 0.0;
            x_double = (double)( -c) / (double)(a) + 2.0000001;  /* + 2.0001 to avoid fract<0 */
            inc_double = -(double)(b)/(double)(a);
            y_off = size * (size-1);

            for ( y = 0 ; y < size ; y++, x_double += inc_double, y_off-=size )    /* scan along y axis */
                    {
                    x = (int)(x_double) - 2;

                    if ( ( x >= 0 ) && ( x < size-1 ) )   /* if within image size */
                        {
                        fract = x_double-(double)(x)-2.0;
                        x_off = size-1 - x;            /* offset along x axis in the image */
                        pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                        pix_tmp += (1.0-fract)*(double)(*pixel_pointer_xy)
                                +  (fract)*(double)(*(pixel_pointer_xy - 1));
                        pix_count++;
                        }

                    else if ( x == size-1 )            /* if within image size */
                        {
                        x_off = size-1 - x;            /* offset along x axis in the image */
                        pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                        pix_tmp += (double)(*pixel_pointer_xy);
                        pix_count++;
                        }
                    }
            if (pix_count!=0) 
                {
                if (norm_fg)
                   {
                   if (  (pix_tmp * (double)(size) / (double)(pix_count) ) <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp * (double)(size) / (double)(pix_count));
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                else 
                   {
                   if ( pix_tmp <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp);
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                }
            else  fprintf(stderr,"A pix_count=0, a_off=%d, z_off=%d\n", a_off, z_off/size);
            }

        /* gentle slope descending lines (ascend in imag)  (middle right of hough  image) */
        for ( a = 0; a < half_size  ; a += 2 )
            {
            a_off = ( a + size ) / 2 ;              /* offset along a  axis in the image */
            b = size - a ;
            pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
            c = - z*size;
            pix_count = 0;
            pix_tmp = 0.0;
            y_double = (double)( -c) / (double)(b) + 2.0000001;  /* + 2.0 to avoid fract<0 */
            inc_double = -(double)(a)/(double)(b);
            x_off = size-1;

            for ( x = 0 ; x < size ; x++, y_double += inc_double,  x_off-- )      /* scan along x axis */
                    {
                    y = (int)(y_double)-2;

                    if ( ( y >= 0 ) && ( y < size-1 ) )     /* if within image size */
                        {
                        fract = y_double-(double)(y)-2.0;
                        y_off = size * ((size-1) - y);    /* offset along y axis in the image */
                        pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                        pix_tmp += (int)( (1.0-fract)*(double)(*pixel_pointer_xy)
                                +  (fract)*(double)(*(pixel_pointer_xy - size)) );
                        pix_count++;
                        }

                    else if ( y == size-1 )     /* if within image size */
                        {
                        y_off = size * ((size-1) - y);            /* offset along y axis in the image */
                        pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                        pix_tmp += (double)(*pixel_pointer_xy);
                        pix_count++;
                        }
                    }
            if (pix_count!=0) 
                {
                if (norm_fg)
                   {
                   if (  (pix_tmp * (double)(size) / (double)(pix_count) ) <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp * (double)(size) / (double)(pix_count));
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                else 
                   {
                   if ( pix_tmp <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp);
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                }
            else  fprintf(stderr,"B pix_count=0, a_off=%d, z_off=%d\n", a_off, z_off/size);
            }

        /* gentle slope climbing lines (desc imag)  (middle left of hough image)*/
        for ( a = 2 ; a <=  half_size ; a += 2 )
            {
            a_off = ( size - a ) / 2 ;             /* offset along "a" axis in hough image */
            b = a - size ;
            pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
            c = -size * (z + b) + b ;

            pix_count = 0;
            pix_tmp = 0.0;
            y_double = (double)( -c) / (double)(b) + 2.0000001;  /* + 2.0001 to avoid fract<0 */
            inc_double = -(double)(a)/(double)(b);
            x_off = 0;

            for ( x = 0 ; x < size ; x++, y_double += inc_double, x_off++ )     /* scan along x axis */
                    {
                    y = (int)(y_double)-2;

                    if ( ( y >= 0 ) && ( y < size-1 ) )       /* if within image size */
                        {
                        fract = y_double-(double)(y)-2.0;
                        y_off = y * size;   /* offset along y axis in the image */
                        pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                        pix_tmp += (int)( (1.0-fract)*(double)(*pixel_pointer_xy)
                                +  (fract)*(double)(*(pixel_pointer_xy + size)) );
                        pix_count++;
                        }
                    else if ( y == size-1 )     /* if within image size */
                        {
                        y_off = y * size;            /* offset along y axis in the image */
                        pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                        pix_tmp += (double)(*pixel_pointer_xy);
                        pix_count++;
                        }
                    }
            if (pix_count!=0) 
                {
                if (norm_fg)
                   {
                   if (  (pix_tmp * (double)(size) / (double)(pix_count) ) <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp * (double)(size) / (double)(pix_count));
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                else 
                   {
                   if ( pix_tmp <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp);
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                }
            else  fprintf(stderr,"C pix_count=0, a_off=%d, z_off=%d\n", a_off, z_off/size);
            }

        /* high slope climbing lines (desc imag)  ( left of hough  image) */
        for (a = half_size+2 ; a <= size ; a += 2)
            {
            a_off = ( size - a ) / 2 ;              /* offset along a  axis in the image */
            b = a - size ;
            pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
            c = -size * (z + b) + b ;
            pix_count = 0;
            pix_tmp = 0.0;
            x_double = (double)( -c) / (double)(a) +2.0000001; /* +2.0 to avoid fract<0 */
            inc_double = -(double)(b)/(double)(a);
            y_off = 0;

            for ( y = 0 ; y < size ; y++, x_double += inc_double, y_off+=size )      /* scan along y axis */
                {
                x = (int)(x_double)-2;

                if ( ( x >= 0 ) && ( x < size-1 ) )       /* if within image size */
                    {
                    fract = x_double-(double)(x)-2.0;
                    x_off = x;                 /* offset along x axis in the image */
                    pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                    pix_tmp += (int)( (1.0-fract)*(double)(*pixel_pointer_xy)
                            +  (fract)*(double)(*(pixel_pointer_xy + 1)) );
                    pix_count++;
                    }

                else if ( x == size-1 )     /* if within image size */
                    {
                    x_off = x;            /* offset along x axis in the image */
                    pixel_pointer_xy = (INIT_IMA_TYP *)image_pointer_xy + y_off + x_off;
                    pix_tmp += (double)(*pixel_pointer_xy );
                    pix_count++;
                    }
                }
            if (pix_count!=0) 
                {
                if (norm_fg)
                   {
                   if (  (pix_tmp * (double)(size) / (double)(pix_count) ) <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp * (double)(size) / (double)(pix_count));
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                else 
                   {
                   if ( pix_tmp <= 4294967295.0 )
                       *pixel_pointer_az = (int)(pix_tmp);
                   else
                       *pixel_pointer_az = 4294967295;
                   }
                }
            else  fprintf(stderr,"D pix_count=0, a_off=%d, z_off=%d\n", a_off, z_off/size);
            }
        }

    return(1);

}

/*****************************************
 *  Inverse (Grey level) Hough Transform *
 *****************************************/
int lvhough_inv(image_az,image_xy,draw_fg,norm_fg)
struct xvimage *image_az,*image_xy;
int    draw_fg,norm_fg;
{
#define BORCOL  255
#define LINCOL  255

int x,y,x_off,y_off;

int a, b, c, z, a_off, z_off, pix_count;

int size, half_size, rowsxcols;
double x_double, y_double, z_double, inc_double, fract, pix_tmp;

IMA_INV_TYP   *image_pointer_xy;
IMA_INV_TYP   *pixel_pointer_xy;
IMA_TYP       *image_pointer_az;
IMA_TYP       *pixel_pointer_az;
double        *db_pixel_pointer_xy, *i_pixel_pointer_xy;

    size      = (int)image_az->row_size;
    rowsxcols = size * size;
    half_size = size/2;
    image_pointer_xy = (IMA_INV_TYP *)image_xy->imagedata;
    image_pointer_az = (IMA_TYP *)image_az->imagedata;

    if ( (image_az->data_storage_type) != VVF_HOUGH_TYP)
        {
        (void)fprintf(stderr,"lvhough : no proper format for Input Image ...\n");
        return(0);
        }

/*
... transmit the startx, startxy infos.
*/

image_xy->startx = image_az->startx;
image_xy->starty = image_az->starty;

/*
... first of all clear up the whole image_xy.
*/

   pixel_pointer_xy = (IMA_INV_TYP *)image_pointer_xy;

   for ( y = 0 ; y < rowsxcols ; y++ )
        *pixel_pointer_xy++ = 0;

/*
... Then process Inverse Hough Transform
*/

/* descending lines (ascending in xy image) (right of hough image)
 * equation of the line, with 0 at left up in xy image:
 *  (size-2a)x + 2(a-size)y + size(size-1-z) = 0
 */
    for ( x = 0 ; x < size ; x++ )        /* scan along x axis */
        {
        x_off = size-1 - x;               /* offset along x axis in the image */
        for ( y = 0 ; y < size ; y++ )    /* scan along y axis */
            {
            y_off = size * (size-1 -y) ;
            pixel_pointer_xy = (IMA_INV_TYP *)image_pointer_xy + y_off + x_off;
            pix_count = 0;
            pix_tmp = 0.0;
            z_double  = (double)(y) + 2.0000001;
            inc_double = 2.0*(double)(x-y) / (double)(size);

            for ( a = 0 ; a < size ; a += 2, z_double += inc_double )
                {
                a_off = (size + a) / 2;        /* offset along a axis in hough image */
/*  initialy, we had:
 *              a = 2*a_off - size;
 *              b = size - a; or b = 2*(size - a_off);
 *              c = - (a*x + b*y); or   c = -z*size;
 *              z_double = (double)(-c) / (double)(size) +(2.00001) ;
 */
                z = (int)(z_double) - 2;
                if ( (z>=0) && ( z<size-1) )
                    {
                    fract = z_double - (double)(z) - 2.0;
                    z_off = size * z;       /* offset along z axis in the image */
                    pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
                    pix_tmp += (1.0-fract)*(double)(*pixel_pointer_az)
                                +  (fract)*(double)(*(pixel_pointer_az + size));
                    pix_count++;
                    }
                else if ( z == size-1 )     /* if within image size */
                    {
                    z_off = size * z;       /* offset along z axis in the image */
                    pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
                    pix_tmp += (double)(*pixel_pointer_az);
                    pix_count++;
                    }
                }
            if (pix_count!=0) 
                {
                if (norm_fg)
                   *pixel_pointer_xy = pix_tmp / (double)(2.0*pix_count);
                else 
                   *pixel_pointer_xy = pix_tmp;
                }
            else  fprintf(stderr,"A pix_count=0, x_off=%d, y_off=%d\n", x_off, y_off/size);
            }  /* end for y */
        }      /* end for x */

    /* climbing lines (descending in xy image)  (left of hough image) */
    for ( x = 0 ; x < size ; x++ )        /* scan along x axis */
        {
        x_off = x;                        /* offset along x axis in the image */
        for ( y = 0 ; y < size ; y++ )    /* scan along y axis */
            {
            y_off = size * y ;
            pixel_pointer_xy = (IMA_INV_TYP *)image_pointer_xy + y_off + x_off;
            pix_count = 0;
            pix_tmp = 0.0;
            inc_double = 2.0*(double)(1+x+y) / (double)(size) - 2.0;
            z_double  = (double)(size - y) + 1.0000001 + inc_double;   
            for ( a = 2 ; a <= size ; a += 2, z_double += inc_double)
                {
                a_off = (size - a) / 2;      /* offset along a axis in hough image */
/*
                b = a - size;
                c = - (a*x + b*y);
        details:
                c = - (a*x + a*y - size*y);
                z_double = (double)(b-c) / (double)(size) - (double)(b) + 2.0000001;
                z_double = (double)(a - size -c) / (double)(size) - (double)(a-size) + 2.0000001;
                z_double = (double)(a - size + a*x + b*y) / (double)(size) - a + size + 2.0000001;
                z_double = (double)(a - size + a*x + (a - size)*y) / (double)(size) - a + size + 2.0000001;
                z_double = (double)(a - size + a*x + a*y - size*y) / (double)(size) - a + size + 2.0000001;
                z_double = (double)(a*(1+x+y)) / (double)(size) -1 -y - a + size + 2.0000001;
                z_double = (double)(a*(1+x+y)) / (double)(size) -y - a + size + 1.0000001;

                z_double = (double)(b-c) / (double)(size) - (double)(b) + 2.0000001;
                z = (int)(z_double) - 2;
*/
                z = (int)(z_double) - 2;

                if ( (z>=0) && (z<size-1) )
                    {
                    fract = z_double - (double)(z) - 2.0;
                    z_off = size * z;      /* offset along z axis in the image */
                    pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
                    pix_tmp += (1.0-fract)*(double)(*pixel_pointer_az)
                                +  (fract)*(double)(*(pixel_pointer_az + size));
                    pix_count++;
                    }
                else if ( z == size-1 )     /* if within image size */
                    {
                    z_off = size * z;       /* offset along z axis in the image */
                    pixel_pointer_az = (IMA_TYP *)image_pointer_az + z_off + a_off;
                    pix_tmp += (double)(*pixel_pointer_az );
                    pix_count++;
                    }
                }
            if (pix_count!=0) 
                {
                if (norm_fg)
                   *pixel_pointer_xy += pix_tmp / (double)(2.0*pix_count);
                else 
                   *pixel_pointer_xy += pix_tmp;
                }
            else  fprintf(stderr,"B pix_count=0, x_off=%d, y_off=%d\n", x_off, y_off/size);
            }  /* end for y */
        }      /* end for x */

/*
... draw surrounding lines if asked for.
*/

  if ( draw_fg != 0 )
    {
    pixel_pointer_xy = (IMA_INV_TYP *)image_pointer_xy;

    for ( x = 0 ; x < size ; x++ )      /* upper and lower lines */
        {
        *pixel_pointer_xy = BORCOL;                  /* grey value */
        *(pixel_pointer_xy+size*(size-1)) = BORCOL;
        pixel_pointer_xy++;
        }

    pixel_pointer_xy = (IMA_INV_TYP *)image_pointer_xy;

    for ( y = 0 ; y < size ; y++ )      /* left and right lines */
        {
        *pixel_pointer_xy = BORCOL;                  /* grey value */
        *(pixel_pointer_xy+size-1) = BORCOL;
        pixel_pointer_xy += size;
        }
    }
   return(1);
}

/* -library_code_end */
