/*******************************************************************************
+
+  LEDA  3.0
+
+
+  graph.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef LEDA_GRAPH_H
#define LEDA_GRAPH_H

//------------------------------------------------------------------------------
//   directed graphs
//------------------------------------------------------------------------------


#include <LEDA/basic.h>
#include <LEDA/list.h>


//------------------------------------------------------------------------------
/* declare node, edge, list<node> and list<edge>                              */
//------------------------------------------------------------------------------

class i_node;
class i_edge;

// node = pointer to internal node
typedef i_node* node;

// edge = pointer to internal edge
typedef i_edge* edge;


/* list<node> = doubly linked list of nodes */

/* list<edge> = doubly linked list of edges */



//------------------------------------------------------------------------------
// class i_node: internal representation of nodes
//------------------------------------------------------------------------------

class i_node {  

   friend class graph;
   friend class ugraph;
   friend class planar_map;
   

   graph* g;             // pointer to graph 
   int i_name;           // internal name (index)  
   list_item loc;        // location in V
   int indegree;        
   GenPtr contents;         // user defined pointer-type
   list<edge> adj_edges;   /* list of adjacent edges */ 

   i_node(GenPtr i)
   { 
     contents=i;
     i_name = -1; 
     g = 0; 
     loc = 0; 
     indegree = 0;
   }

  LEDA_MEMORY(i_node)

friend graph* graph_of(node);
friend graph* graph_of(edge);
friend int    indeg(node);
friend int    outdeg(node);
friend int    degree(node);
friend int    index(node);

/* space :  5 words + 1 list = 20 + 20 bytes = 40 bytes */
};


inline graph* graph_of(node v)   { return v->g;         }
inline int    indeg(node v)      { return v->indegree;  }
inline int    outdeg(node v)     { return v->adj_edges.length(); }
inline int    degree(node v)     { return v->adj_edges.length(); }
inline int    index(node v)      { return v->i_name;    }


//------------------------------------------------------------------------------
// class i_edge: internal representation of edges
//------------------------------------------------------------------------------

class i_edge { 
   
   friend class graph;
   friend class ugraph;
   friend class planar_map;


   int i_name;           // internal name (index)  
   node s;               // edge from "source" 
   node t;               // to "target"
   list_item loc;        // location in |E|
   list_item adj_loc;    // location in adjacencylist of source(e)
   list_item adj_loc2;   // location in adjacencylist of target(e) (ugraph)
   GenPtr contents;         // user defined pointer type

   i_edge(node v, node w, GenPtr i)
   { contents=i;
     i_name=-1;
     s=v;
     t=w;
     loc=0;
     adj_loc=0;
     adj_loc2=0;
   }

  LEDA_MEMORY(i_edge)

friend graph* graph_of(edge);
friend node   source(edge);
friend node   target(edge);
friend int    index(edge);

//space :  7 words = 28 bytes
};


inline graph* graph_of(edge e)   { return e->s->g;      }
inline node   source(edge e) { return e->s;         }
inline node   target(edge e) { return e->t;         }
inline int    index(edge e)  { return e->i_name;    }




//------------------------------------------------------------------------------
// Macro zur Definition von Feldern mit Indextyp node oder edge
//------------------------------------------------------------------------------

// declaration (itype = node/edge):

#define graph_array(itype) name2(itype,_graph_array)

#define declare_graph_array(itype)\
\
class graph_array(itype) {\
friend class node_partition;\
graph* g;  /* array is declared for graph *g */  \
int max_i; /* maximal node index in g at declaration time */  \
protected:\
GenPtr* arr;  \
virtual void clear_entry(GenPtr&) const {}\
virtual void copy_entry(GenPtr&)  const {}\
virtual void init_entry(GenPtr&)  {}\
\
public:\
virtual int cmp_entry(itype x, itype y) const { return int(inf(x))-int(inf(y)); }\
 GenPtr& entry(itype x)\
 { if (g != graph_of(x) || index(x) > max_i)\
          error_handler(102,"(node/edge)_array[x]: not defined for x");\
  return arr[index(x)];\
 }\
 GenPtr  inf(itype x) const\
 { if (g != graph_of(x) || index(x) > max_i)\
          error_handler(102,"(node/edge)_array[x]: not defined for x");\
  return arr[index(x)];\
 }\
 GenPtr& entry(int i) { return arr[i]; }\
 void init(const graph&, int, GenPtr);\
 void init(const graph&, GenPtr);\
 void init(const graph&);\
 void init(const graph_array(itype)&);\
 void clear();\
 int  size()               const     { return max_i+1;}\
 graph_array(itype)()                { g=0; max_i=(-1); arr=0;}\
virtual ~graph_array(itype)()        { clear(); }\
};

declare_graph_array(node)
declare_graph_array(edge)

typedef int  (*cmp_graph_node) (node&, node&);
typedef int  (*cmp_graph_edge) (edge&, edge&);

//------------------------------------------------------------------------------
// graph: base class for all graphs
//------------------------------------------------------------------------------

class graph {

friend class ugraph;
friend class planar_map;
friend class graph_array(node); 
friend class graph_array(edge); 

list<node> V;              /* list of all nodes */
list<edge> E;              /* list of all edges */

int max_node_index;      // maximal node index 
int max_edge_index;      // maximal edge index

bool   undirected;       // true iff graph undirected

/* space: 2 lists + 4 words + "virtual" = 2*20 + 5*4 bytes = 60 bytes */

virtual void init_node_entry(GenPtr& x)  { x=0; }
virtual void init_edge_entry(GenPtr& x)  { x=0; }

virtual void copy_node_entry(GenPtr& x)  const { x=x; }
virtual void copy_edge_entry(GenPtr& x)  const { x=x; }

virtual void clear_node_entry(GenPtr& x) const { x=0; }
virtual void clear_edge_entry(GenPtr& x) const { x=0; }

virtual void read_node_entry(istream& in, GenPtr& x)  { Read(*(int*)&x,in); }
virtual void read_edge_entry(istream& in, GenPtr& x)  { Read(*(int*)&x,in); }

virtual void write_node_entry(ostream& o, GenPtr& x)  const { Print(*(int*)&x,o); }
virtual void write_edge_entry(ostream& o, GenPtr& x)  const { Print(*(int*)&x,o); }

virtual void print_node_entry(ostream&, GenPtr&)  const {}
virtual void print_edge_entry(ostream&, GenPtr&)  const {}

virtual string node_type()  const { return "void"; }
virtual string edge_type()  const { return "void"; }

protected:

graph* parent;           // for subgraphs

void copy_all_entries() const;
void clear_all_entries() const;

public:

virtual int cmp_node_entry(node, node) const { return 0; }
virtual int cmp_edge_entry(edge, edge) const { return 0; }

   int  space() const;
   list<node> adj_nodes(node)   const;
   void reset() const;


   int  indeg(node v)     const;
   int  outdeg(node v)    const;
   node source(edge e)    const;
   node target(edge e)    const;

   graph* super()         const;
   int max_i_node()       const;
   int max_i_edge()       const;

   int  number_of_nodes() const;
   int  number_of_edges() const;

   list<node> all_nodes()       const;
   list<edge> all_edges()       const;
   list<edge> adj_edges(node v) const;

   GenPtr& entry(node v);
   GenPtr& entry(edge e);
   GenPtr  inf(node v) const;
   GenPtr  inf(edge e) const;


   void init_node_iterator() const;
   void init_edge_iterator() const;
   int  next_node(node& v)   const;
   int  prev_node(node& v)   const;
   int  next_edge(edge& e)   const;
   int  prev_edge(edge& e)   const;

   edge first_adj_edge(node v)  const;
   edge last_adj_edge(node v)   const;

   void init_adj_iterator(node v)        const;
   int  next_adj_edge(edge& e,node v)    const ;
   int  current_adj_edge(edge& e,node v) const ;
   int  next_adj_node(node& w,node v)    const;
   int  current_adj_node(node& w,node v) const;
   
   node first_node()      const;
   edge first_edge()      const;
   node last_node()       const;
   edge last_edge()       const;
   node choose_node()     const;
   edge choose_edge()     const;
   node succ_node(node v) const;
   node pred_node(node v) const;
   edge succ_edge(edge e) const;
   edge pred_edge(edge e) const;
   edge adj_succ(edge e)  const;
   edge adj_pred(edge e)  const;
   edge cyclic_adj_succ(edge e) const;
   edge cyclic_adj_pred(edge e) const;

#ifdef __ZTC__
public:
#else
protected:
#endif

   node new_node(GenPtr);
   edge new_edge(node, node, GenPtr);
   edge new_edge(edge, node, GenPtr, int dir);

   void assign(node v,GenPtr x);
   void assign(edge e,GenPtr x);

public:

   node new_node()   { GenPtr x; init_node_entry(x);  
                       return new_node(x); }

   edge new_edge(node v, node w) 
   { GenPtr x; init_edge_entry(x);
     return new_edge(v,w,x);}

   edge new_edge(edge e, node w, int dir=0) 
   { GenPtr x; init_edge_entry(x);
     return new_edge(e,w,x,dir); }

   void del_node(node);
   void del_edge(edge);
   void del_all_nodes(); 
   void del_all_edges(); 

   list<edge> insert_reverse_edges();

   void sort_edges(cmp_graph_edge);
   void sort_nodes(cmp_graph_node);

   void sort_edges(const graph_array(edge)&);
   void sort_nodes(const graph_array(node)&);

   void sort_edges();
   void sort_nodes();

   edge rev_edge(edge);
   graph& rev();

   void make_directed();
   void make_undirected();

   void write(ostream& = cout) const;
   void write(string) const;

   int  read(istream& = cin);
   int  read(string);

   void print_node(node,ostream& = cout)  const;

virtual void print_edge(edge,ostream& = cout)  const;  //different for ugraph's

   void print(string s, ostream& = cout) const;

   void print()           const { print("");   }
   void print(ostream& o) const { print("",o); }

   void clear();

   graph();
   graph(const graph&);
   graph& operator=(const graph&); 

  virtual ~graph(){ clear(); }


   //subgraph constructors

   graph(graph&, const list<node>&, const list<edge>&);
   graph(graph&, const list<edge>&);

};

inline int  graph::indeg(node v)     const   { return v->indegree; }
inline int  graph::outdeg(node v)    const   { return v->adj_edges.length(); }
inline node graph::source(edge e)    const   { return e->s; }
inline node graph::target(edge e)    const   { return e->t; }

inline graph* graph::super()         const   { return parent; }
inline int graph::max_i_node()       const   { return max_node_index; }
inline int graph::max_i_edge()       const   { return max_edge_index; }

inline int  graph::number_of_nodes() const   { return V.length(); }
inline int  graph::number_of_edges() const   { return E.length(); }

inline GenPtr& graph::entry(node v)        { return v->contents; }
inline GenPtr& graph::entry(edge e)        { return e->contents; }
inline void graph::assign(node v,GenPtr x) { v->contents = x; }
inline void graph::assign(edge e,GenPtr x) { e->contents = x; }

inline GenPtr  graph::inf(node v) const { return v->contents; }
inline GenPtr  graph::inf(edge e) const { return e->contents; }


inline void graph::init_node_iterator()   const { V.reset(); }
inline void graph::init_edge_iterator()   const { E.reset(); }
inline int  graph::next_node(node& v)     const { return V.next_element(v); }
inline int  graph::prev_node(node& v)     const { return V.prev_element(v); }
inline int  graph::next_edge(edge& e)     const { return E.next_element(e); }
inline int  graph::prev_edge(edge& e)     const { return E.prev_element(e); }
inline edge graph::first_adj_edge(node v) const { return v->adj_edges.head(); }
inline edge graph::last_adj_edge(node v)  const { return v->adj_edges.tail(); }


inline void graph::init_adj_iterator(node v) const 
{ v->adj_edges.reset(); }

inline int  graph::next_adj_edge(edge& e,node v) const 
{ return v->adj_edges.next_element(e); }

inline int  graph::current_adj_edge(edge& e,node v) const 
{ return v->adj_edges.current_element(e);}


inline int graph::next_adj_node(node& w,node v)  const
{ edge e;
  if (next_adj_edge(e,v))
  { w = (v==e->s) ? e->t : e->s;
    return 1; 
   }
  else { w = 0; return 0;}
 }
   
inline int graph::current_adj_node(node& w,node v)  const
{ edge e;
  if (current_adj_edge(e,v))
  { w = (v==e->s) ? e->t : e->s; 
    return 1; }
  else { w = 0; return 0; }
}
   
inline node graph::first_node()   const { return V.head(); }
inline edge graph::first_edge()   const { return E.head(); }
inline node graph::last_node()    const { return V.tail(); }
inline edge graph::last_edge()    const { return E.tail(); }
inline node graph::choose_node()  const { return V.head(); }
inline edge graph::choose_edge()  const { return E.head(); }

inline node graph::succ_node(node v)  const  
{ list_item loc = V.succ(v->loc);
  return (loc) ? V.inf(loc) : 0;
 }

inline node graph::pred_node(node v)  const  
{ list_item loc = V.pred(v->loc);
  return (loc) ? V.inf(loc) : 0;
 }

inline edge graph::succ_edge(edge e)  const  
{ list_item loc = E.succ(e->loc);
  return (loc) ? E.inf(loc) : 0;
 }

inline edge graph::pred_edge(edge e)  const  
{ list_item loc = E.pred(e->loc);
  return (loc) ? E.inf(loc) : 0;
 }


inline edge graph::adj_succ(edge e) const 
{ list_item loc = e->s->adj_edges.succ(e->adj_loc);
  return (loc) ? e->s->adj_edges[loc] : 0; 
 } 

inline edge graph::adj_pred(edge e) const 
{ list_item loc = e->s->adj_edges.pred(e->adj_loc);
  return (loc) ? e->s->adj_edges[loc] : 0; 
 } 

inline edge graph::cyclic_adj_succ(edge e) const 
{ list_item loc = e->s->adj_edges.cyclic_succ(e->adj_loc);
  return e->s->adj_edges[loc];
 }

inline edge graph::cyclic_adj_pred(edge e)  const
{ list_item loc = e->s->adj_edges.cyclic_pred(e->adj_loc);
  return e->s->adj_edges[loc];
 }



//------------------------------------------------------------------------------
// GRAPH: generic directed graphs
//------------------------------------------------------------------------------


template<class vtype, class etype> 

class _CLASSTYPE GRAPH : public graph 
{

vtype X;
etype Y;

void copy_node_entry(GenPtr& x) const  { x=Copy(ACCESS(vtype,x)); }
void copy_edge_entry(GenPtr& x) const  { x=Copy(ACCESS(etype,x)); }

void clear_node_entry(GenPtr& x) const { Clear(ACCESS(vtype,x)); }
void clear_edge_entry(GenPtr& x) const { Clear(ACCESS(etype,x)); }

void write_node_entry(ostream& o, GenPtr& x) const {Print(ACCESS(vtype,x),o);}
void write_edge_entry(ostream& o, GenPtr& x) const {Print(ACCESS(etype,x),o);}

void read_node_entry(istream& i, GenPtr& x) { Read(X,i); x=Copy(X); }
void read_edge_entry(istream& i, GenPtr& x) { Read(Y,i); x=Copy(Y); }

void init_node_entry(GenPtr& x) { Init(X); x = Copy(X); }
void init_edge_entry(GenPtr& x) { Init(Y); x = Copy(Y); }

void print_node_entry(ostream& o, GenPtr& x)  const
     { o << "("; Print(ACCESS(vtype,x),o); o << ")"; }
void print_edge_entry(ostream& o, GenPtr& x)  const
     { o << "("; Print(ACCESS(etype,x),o); o << ")"; }

string node_type()  const { return TYPE_NAME(vtype); }
string edge_type()  const { return TYPE_NAME(etype); }

public:

int cmp_node_entry(node x, node y) const { return compare(inf(x),inf(y)); }
int cmp_edge_entry(edge x, edge y) const { return compare(inf(x),inf(y)); }

vtype  inf(node v)    const   { return ACCESS(vtype,graph::inf(v)); }
etype  inf(edge e)    const   { return ACCESS(etype,graph::inf(e)); }

vtype& operator[] (node v)    { return ACCESS(vtype,entry(v)); }
etype& operator[] (edge e)    { return ACCESS(etype,entry(e)); }
vtype  operator[] (node v) const   { return ACCESS(vtype,graph::inf(v)); }
etype  operator[] (edge e) const   { return ACCESS(etype,graph::inf(e)); }

void   assign(node v,vtype x) { operator[](v) = x; }
void   assign(edge e,etype x) { operator[](e) = x; }

node   new_node()               { return graph::new_node(); }
node   new_node(vtype a)        { return graph::new_node(Copy(a)); }

edge   new_edge(node v, node w) { return graph::new_edge(v,w); }
edge   new_edge(node v, node w, etype a)
                                { return graph::new_edge(v,w,Copy(a)); }
edge   new_edge(edge e, node w) { return graph::new_edge(e,w); }
edge   new_edge(edge e, node w, etype a)
                                { return graph::new_edge(e,w,Copy(a),0); }
edge   new_edge(edge e, node w, etype a, int dir)
                                { return graph::new_edge(e,w,Copy(a),dir); }

void   clear()                  { clear_all_entries(); graph::clear(); }

GRAPH& operator=(const GRAPH& a)
{ clear_all_entries();graph::operator=(a);copy_all_entries();return *this; }

GRAPH()  {}
GRAPH(const GRAPH& a) : graph(a) { a.copy_all_entries(); } 

/* subgraphs */
GRAPH(GRAPH& a, const list<node>& b, const list<edge>& c) : graph(a,b,c) {}
GRAPH(GRAPH& a, const list<edge>& c):graph(a,c){}

virtual ~GRAPH()   { if (parent==0) clear_all_entries(); }

};


#if defined(LEDA_CHECKING_OFF)
#define GRAPH_ARRAY_ACCESS(I,type)\
type& operator[](I x)       { return ACCESS(type,arr[index(x)]); }\
type  operator[](I x) const { return ACCESS(type,arr[index(x)]); }
#else
#define GRAPH_ARRAY_ACCESS(I,type)\
type& operator[](I x)       { return ACCESS(type,entry(x));}\
type  operator[](I x) const { return ACCESS(type,inf(x));}
#endif

//------------------------------------------------------------------------------
// node arrays
//------------------------------------------------------------------------------


template<class type>

class _CLASSTYPE node_array : public graph_array(node) {

type X;

void copy_entry(GenPtr& x) const { x=Copy(ACCESS(type,x));}
void clear_entry(GenPtr& x)const { Clear(ACCESS(type,x)); }
void init_entry(GenPtr& x)       { Init(X); x = Copy(X); }

public:
int cmp_entry(node x,node y) const 
{ return compare(ACCESS(type,arr[index(x)]),ACCESS(type,arr[index(y)])); }

GRAPH_ARRAY_ACCESS(node,type)

type& elem(int i)              { return ACCESS(type,arr[i]);}
type  elem(int i) const        { return ACCESS(type,arr[i]);}

void  init(const graph& G, int n, type i)
                                  { graph_array(node)::init(G,n,Convert(i)); }
void  init(const graph& G, type i){ graph_array(node)::init(G,Convert(i)); }
void  init(const graph& G)        { graph_array(node)::init(G); }
void  init(const node_array& A)   { graph_array(node)::init(A); }

node_array() {}
node_array(const graph& G)                 { init(G);   }
node_array(const graph& G, int n, type i)  { init(G,n,i); }
node_array(const graph& G, type i)         { init(G,i); }
node_array(const node_array& A)            { init(A);   }
node_array& operator=(const node_array& A) { init(A); return *this;}

~node_array() { clear(); }
};


//------------------------------------------------------------------------------
// edge arrays
//------------------------------------------------------------------------------


template<class type>

class _CLASSTYPE edge_array : public graph_array(edge) {

type X;

void copy_entry(GenPtr& x) const { x=Copy(ACCESS(type,x)); }
void clear_entry(GenPtr& x)const { Clear(ACCESS(type,x));  }
void init_entry(GenPtr& x)       { Init(X); x = Copy(X); }

public:
int cmp_entry(edge x,edge y) const 
{ return compare(ACCESS(type,arr[index(x)]),ACCESS(type,arr[index(y)])); }

GRAPH_ARRAY_ACCESS(edge,type)

type& elem(int i)              { return ACCESS(type,arr[i]);}
type  elem(int i) const        { return ACCESS(type,arr[i]);}
void  init(const graph& G, int m, type i)
                                  { graph_array(edge)::init(G,m,Convert(i)); }
void  init(const graph& G, type i){ graph_array(edge)::init(G,Convert(i)); }
void  init(const graph& G)        { graph_array(edge)::init(G); }
void  init(const edge_array& A)   { graph_array(edge)::init(A); }

edge_array()                               {}
edge_array(const graph& G)                 { init(G);   }
edge_array(const graph& G, int m, type i)  { init(G,m,i); }
edge_array(const graph& G, type i)         { init(G,i); }
edge_array(const edge_array& A)            { init(A);   }
edge_array& operator=(const edge_array& A) { init(A); return *this;}

~edge_array()  { clear(); }
};

//------------------------------------------------------------------------------
// Iteration   (macros)
//------------------------------------------------------------------------------

/*
#define forall_nodes(a,b) for ((b).init_node_iterator(); (b).next_node(a); )
#define forall_edges(a,b) for ((b).init_edge_iterator(); (b).next_edge(a); )
*/

#define forall_nodes(a,g) for (a=(g).first_node(); a; a=(g).succ_node(a) )

#define forall_edges(a,g) for (a=(g).first_edge(); a; a=(g).succ_edge(a) )

#define forall_adj_edges(a,b)\
for (graph_of(b)->init_adj_iterator(b); graph_of(b)->next_adj_edge(a,b); )

#define forall_adj_edges1(a,b)\
   for (a=graph_of(b)->first_adj_edge(b); a; a=graph_of(b)->adj_succ(a) )
 

#define forall_adj_nodes(a,b)\
for (graph_of(b)->init_adj_iterator(b); graph_of(b)->next_adj_node(a,b); )

#define forall_node_pairs(a,b,g)\
forall_nodes(a,g) for (b=(g).first_node(); b; b=(g).succ_node(b) )


// reverse

#define Forall_nodes(a,g) for (a=(g).last_node(); a; a=(g).pred_node(a) )

#define Forall_edges(a,g) for (a=(g).last_edge(); a; a=(g).pred_edge(a) )



//-----------------------------------------------------------------------------
// miscellaneous  (should be members or constructors)
//-----------------------------------------------------------------------------

extern void eliminate_parallel_edges(graph& G);

extern void complete_graph(graph&,int);
extern void random_graph(graph&,int,int);
extern void test_graph(graph&);

extern void complete_bigraph(graph&,int,int,list<node>&,list<node>&);
extern void random_bigraph(graph&,int,int,int,list<node>&,list<node>&);
extern void test_bigraph(graph&,list<node>&,list<node>&);

extern void random_planar_graph(graph&,int);

extern void cmdline_graph(graph&,int,char**);

#endif

