/*LINTLIBRARY*/

/*  @(#)hp.c 1.2 92/01/05
 *
 *  Popi device driver for an HP 8+ plane display.
 *  Written by Eric Haines - 3D/Eye Inc, Ithaca, NY
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#include "graphics.h"
#include <starbase.c.h>

/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c)     - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(l,y,w,c)    - to draw an image scanline.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype)      - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)   - load new colormap.
 */

extern char *getenv();

/* Set environment variables OUTDEV and OUTDRIVER or change the
 * defaults to your machine.
 */
char	default_outdev[] = "/dev/hp98721" ;
char	default_outdriver[] = "hp98721" ;
char	default_indev[] = "/dev/hil2" ;
char	default_indriver[] = "hp-hil" ;

int	fd ;

void
disp_init(argc,argv)           /* called from main at the start. */
int argc;
char *argv[];
{
    fd = devopen(OUTDEV, OverlayFlag ? NULL : INIT ) ;
    if ( fd == -1 ) {
	printf("error opening output device\n" ) ;
	exit(1) ;
    }
    vdc_extent( fd, 0.0, 0.0, 0.0, 1.25, 1.0, 1.0 ) ;
}


void
disp_finish()                  /* called from main prior to exit. */
{
    gclose( fd ) ;
}


void
disp_imgstart(width, height, ncolors)  /* called prior to drawing an image. */
int width, height, ncolors ;
{
    if ( ncolors == 1 ) {
	shade_mode( fd, CMAP_MONOTONIC | INIT, 0 ) ;
    } else {
	shade_mode( fd, CMAP_FULL | INIT, 0 ) ;
    }
    make_picture_current( fd ) ;
}


void
disp_imgend()                  /* called after drawing an image. */
{
    make_picture_current( fd ) ;
}


void
disp_putline(line, y, width, ncolors)     /* draw image scanline y. */
pixel_t **line;
int y, width, ncolors;
{
int	chan ;

    if ( ncolors == 1 ) {
	dcblock_write( fd, Xoffset,Yoffset+y,width,1,line[0],0) ;
    } else {
	for ( chan = 0 ; chan < 3 ; chan++ ) {
	    bank_switch( fd, 2-chan, 1 ) ;
	    dcblock_write( fd, Xoffset,Yoffset+y,width,1,line[chan],0) ;
	}
    }
}


disp_getchar()                 /* get next user typed character. */
{
  return(getchar());
}


disp_prompt()                  /* display popi prompt. */
{
  PRINTF("-> ");
  return 3;
}


void
disp_error(errtype, pos)            /* display error message. */
int	errtype,
	pos;
{
    extern int  errno;
    extern char *sys_errlist[];

    if (errtype & ERR_PARSE)
    {
        int     i;
 
        for (i=1; i < pos; ++i)
            PUTC('-', stderr);
        PUTC('^', stderr);
        PUTC('\n', stderr);
    }
 
    fprintf(stderr, "%s\n", ErrBuf);
    /* we assume errno hasn't been reset by the preceding output */
    if (errtype & ERR_SYS)
        fprintf(stderr, "\t(%s)\n", sys_errlist[errno]);
}

void
disp_percentdone(percent)
int	percent;
{
    static int	lastpercent = 100;

    if (!Verbose)
	return;
    if (percent == 100)
    {
	printf("\r    \n");
	return;
    }
    if (percent != lastpercent && percent % 5 == 0)
    {
	printf("\r%2d%% ", percent);
	fflush(stdout);
	lastpercent = percent;
    }
}

void
disp_resize(width, height)    /* Resize popi image window. */
int width, height ;
{
}

void
disp_colormap(n, red, green, blue)      /* Load new colormap. */
int n ;
unsigned char *red, *green, *blue ;
{
}

int devopen(dev_kind,init_mode)
int dev_kind,init_mode;
{
char	*dev, *driver;
int	fildes ;

    if ( dev_kind == OUTDEV ) {
	dev = getenv("OUTDEV");
	if (!dev) dev = default_outdev ;
	driver = getenv("OUTDRIVER");
	if (!driver) driver = default_outdriver ;
    } else {
	dev = getenv("INDEV");
	if (!dev) dev = default_indev ;
	driver = getenv("INDRIVER");
	if (!driver) driver = default_indriver ;
    }

    fildes = gopen(dev,dev_kind,driver,init_mode);

    return(fildes) ;
}
