
/*  @(#)special.c 1.4 92/03/05
 *
 *  Special transformations used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

/*  Special transformations from chapter 6 of BP.
 *
 *  The way this is done is fairly nasty at the moment, but it does work.
 */

#include <sys/types.h>
#include <ctype.h>
#include <time.h>

#ifdef      BSD
#include <sys/timeb.h>
#endif   /* BSD */

#include "popi.h"
#include "expr.h"

#define  GAMMA     7.5    /* Default gamma for matte() */
#define  TILESIZE  25     /* Default tile size for tile() */
#define  N         3      /* Default blur size for oil() */

#define  New      (CurNew->pix)

static void CloseLog P((FILE **)) ;


void
oil(t)
Tree *t ;
{
  register int color, dyoff, x, y, yoff ;
  register int dx, dy ;
  pixel_t mfp ;
  static int *histo = 0 ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  img = srcp->pix ;

  if (histo == (int *) 0 &&
      (histo = (int *)
       LINT_CAST(Emalloc((unsigned) Zsize * sizeof (int)))) == (int *) 0)
    return ;

  for (y = N; y < srcp->height-N; y++)
    {
      yoff = y * srcp->width ;
      for (x = N; x < srcp->width-N; x++)
        {
	  for (color = 0; color < colors; ++color)
	    {
	      for (dx = 0; dx < Zsize; dx++) histo[dx] = 0 ;
	      for (dy = y-N; dy <= y+N; dy++)
		{
                  dyoff = dy * srcp->width ;
                  for (dx = x-N; dx <= x+N; dx++)
		    histo[img[color][dyoff+dx]]++ ;
                }
     
	      for (dx = dy = 0; dx < Zsize; dx++)
		if (histo[dx] > dy)
		  {
		    dy = histo[dx] ;
		    mfp = (pixel_t) dx ;
		  } 
	      New[color][yoff+x] = mfp ;
	    }
        }
      disp_percentdone((y+1) * 100 / srcp->height) ;
    }

  disp_percentdone(100) ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
shear(t)
Tree *t ;
{
  register int color, dyoff, r, x, y, yoff ;
  int dx, dy ;
  static int *yshift = 0 ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ; 
  img = srcp->pix ;

  if (yshift == 0 &&
      (yshift = (int *)
      LINT_CAST(Emalloc((unsigned) srcp->width * sizeof(int)))) == 0)
    return ;

  for (x = r = 0; x < srcp->width; x++)
    {
      if (RANDOM % Zsize < 128) r-- ;
      else                      r++ ;
      yshift[x] = r ;
    }

  for (y = 0; y < srcp->height; y++)
    {
      if (RANDOM % Zsize < 128) r-- ;
      else                      r++ ;

      yoff = y * srcp->width ;
      for (x = 0; x < srcp->width; x++)
	{
	  dx = x + r ;
	  dy = y + yshift[x] ;
	  if (dx >= srcp->width || dy >= srcp->height || dx < 0 || dy < 0)
	    continue ;
	  dyoff = dy * srcp->width ;
          for (color = 0; color < colors; color++)
	    New[color][yoff+x] = img[color][dyoff+dx] ;
	}
      disp_percentdone((y+1) * 100 / srcp->height) ;
    }

  disp_percentdone(100) ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
slice(t)
Tree *t ;
{
  register int color, dyoff, r, x, y, yoff ;
  int dx, dy ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;
  static int *xshift = 0 ;
  static int *yshift = 0 ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ; 
  img = srcp->pix ;

  if (xshift == 0 &&
      (xshift = (int *)
      LINT_CAST(Emalloc((unsigned) srcp->height * sizeof (int)))) == 0)
    return ;

  if (yshift == 0 &&
      (yshift = (int *)
      LINT_CAST(Emalloc((unsigned) srcp->width * sizeof (int)))) == 0)
    return ;

  for (x = dx = 0 ; x < srcp->width; x++)
    {
      if (dx == 0)
        {
          r = (RANDOM & 63) - 32 ;
          dx = 8 + RANDOM & 31 ;
        }
      else dx-- ;
      yshift[x] = r ;
    }

  for (y = dy = 0; y < srcp->height; y++)
    {
      if (dy == 0)
        {
          r = (RANDOM & 63) - 32 ;
          dy = 8 + RANDOM & 31 ;
        }
      else dy-- ;
      xshift[y] = r ;
    }

  for (y = 0; y < srcp->height; y++)
    {
      yoff = y * srcp->width ;
      for (x = 0; x < srcp->width; x++)
	{
	  dx = x + xshift[y] ;
	  dy = y + yshift[x] ;
	  dyoff = dy * srcp->width ;
          if (dx < srcp->width && dy < srcp->height && dx >= 0 && dy >= 0)
	    for (color = 0; color < colors; color++)
	      New[color][yoff+x] = img[color][dyoff+dx] ;
	}
      disp_percentdone((y+1) * 100 / srcp->height) ;
    }

  disp_percentdone(100) ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
tile(t)
Tree *t ;
{
  register int color, dx, dy, dyoff, nyoff, x, y ;
  int ox, oy, nx, ny ;
  int TileSize = TILESIZE ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ; 
  img = srcp->pix ;

  for (y = 0; y < srcp->height-TileSize; y += TileSize)
    {
      for (x = 0; x < srcp->width-TileSize; x += TileSize)
	{
	  ox = (RANDOM & 31) - 16 ;       /* Displacement. */
	  oy = (RANDOM & 31) - 16 ;

	  for (dy = y; dy < y+TileSize; dy++)
	    {
              dyoff = dy * srcp->width ;
              for (dx = x; dx < x+TileSize; dx++)
	        {
		  nx = dx + ox ;
		  ny = dy + oy ;
		  nyoff = ny * srcp->width ;
                  if (nx >= srcp->width  || nx < 0 ||
		      ny >= srcp->height || ny < 0)
		    continue ;
		  for (color = 0; color < colors; color++)
		    New[color][nyoff+nx] = img[color][dyoff+dx] ;
	        }
            }
	}
      disp_percentdone((y+1) * 100 / srcp->height) ;
    }

  disp_percentdone(100) ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


/* Note: affects source image in situ. Buffers not swapped after processing. */

void
melt(t)
Tree *t ;
{
  register int color, k, NumPixels, x, y, yoff, yoff1 ;
  pixel_t val ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ; 
  img = srcp->pix ;

/* Copy New from img, then manipulate New */

  for (y = 0; y < srcp->height; y++)
    {
      yoff = y * srcp->width ;
      for (x = 0; x < srcp->width; x++)
        for (color = 0; color < colors; color++)
	  New[color][yoff+x] = img[color][yoff+x] ;
    }

  NumPixels = srcp->width * srcp->height ;
  for (k = 0; k < NumPixels; k++)
    {
      x     = RANDOM % srcp->width ;
      y     = RANDOM % (srcp->height - 1) ;
      yoff  =  y    * srcp->width ;
      yoff1 = (y+1) * srcp->width ;

      for (color = 0; color < colors; color++)
        while (y < srcp->height-1 &&
               New[color][yoff+x] <= New[color][yoff1+x])
          {
            val                 = New[color][yoff +x] ;
            New[color][yoff +x] = New[color][yoff1+x] ;
            New[color][yoff1+x] = val ;
            y++ ;
            yoff                =  y    * srcp->width ;
            yoff1               = (y+1) * srcp->width ;
          }
      disp_percentdone((k+1) * 100 / NumPixels) ;
    }

  disp_percentdone(100) ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
matte(t)
Tree *t ;
{
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;
  double gamma ;
  register color, x, y, yoff ;
  static pixel_t *lookup = (pixel_t *) 0 ;
  static double lastgamma = 0.0 ;

  DEBUG((Debug, "matte()\n")) ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ; 
  img = srcp->pix ;

  if (t->kids[1])
    {
      if (((Tree *) (t->kids[1]))->t == T_Fnum)
        gamma = ((Tree *) (t->kids[1]))->d ;
      else gamma = (double) ((Tree *) (t->kids[1]))->i ;
    }
  else gamma = GAMMA ;

  if (lookup == 0 &&
      (lookup = (pixel_t *) Emalloc((unsigned) Zsize)) == (pixel_t *) 0)
    return ;

  if (lastgamma != gamma)
    {
      for (x = 0; x < Zsize; ++x)
        lookup[x] = ((double) Zmax * pow(x / (double) Zmax, gamma) < 3.0)
                    ? Zmax : 0 ;
      lastgamma = gamma ;
    }

  for (y = 0; y < srcp->height; y++)
    {
      yoff = y * srcp->width ;
      for (color = 0; color < colors; color++)
        for (x = 0; x < srcp->width; x++)
          New[color][yoff+x] = lookup[img[color][yoff+x]] ;
      disp_percentdone((y+1) * 100 / srcp->height) ;
    }

  disp_percentdone(100) ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


/* Although this is set up to use one particular graphics mode
 * of a 24-pin printer, everything should be generic enough
 * that it can be changed to work with a different graphics
 * mode on an 8-pin printer, just by changing a few numbers.
 */

void
genepson(t)
Tree *t ;
{
  struct SRC *srcp ;
  char *PinVals ;
  pixel_t *PinThresh, ThreshStep ;
  FILE *ostr ;
  register int x ;
  register pixel_t **img ;
  int y, yn ;
  int pin ;
  int BytesPerChunk, PinsPerPixel, BytesPerColumn ;
  int yPixelsPerByte, xPinsPerPixel, yPinsPerPixel ;

  if (t->kids[1]) srcp = &Images[((Tree *) (t->kids[1]))->i] ;
  else            srcp = CurOld ;
  if ((ostr = EfopenW(lastfname)) == NULL) return ;

  img = srcp->pix ;

/* Printer specific */

  xPinsPerPixel  = 2 ;
  yPinsPerPixel  = 2 ;
  BytesPerColumn = 24 / BITSINBYTE ;

  PinsPerPixel   = xPinsPerPixel * yPinsPerPixel ;
  BytesPerChunk  = xPinsPerPixel * BytesPerColumn ;
  yPixelsPerByte = BITSINBYTE / yPinsPerPixel ;

/* Must be whole number of pixels (y direction) per  byte. */

  assert(yPinsPerPixel * yPixelsPerByte == BITSINBYTE) ;

/* Reallocate these each time, as changing the print mode
 * may change the sizes of these arrays.
 */

  if ((PinVals = Emalloc((unsigned)BytesPerChunk)) == 0 ||
      (PinThresh = (pixel_t *)
      Emalloc((unsigned) PinsPerPixel * sizeof(pixel_t))) == 0)
    return ;

  ThreshStep = (pixel_t) (Zsize / (PinsPerPixel + 1)) ;
  for (pin = 0; pin <  PinsPerPixel; ++pin)
    PinThresh[pin] = (pixel_t) ((pin + 1) * ThreshStep) ;

  for (y = 0;  y < srcp->height; y = yn)
    {

/* Printer specific */

/* This print line is width srcp->width pixels, and
 * (srcp->width * xPinsPerPixel) * pin positions (dots on the page).
 * No. of dots vertical  is (BytesPerColumn * BITSINBYTE)
 * which is yPinsPerPixel times  the no.  of image scanlines.
 */

      FPRINTF(ostr, "\033*%c%c%c",
              39,                             /* 180 dpi in both directions */
              (srcp->width * xPinsPerPixel) % 256,
              (srcp->width * xPinsPerPixel) / 256) ;

      for (x = 0; x <  srcp->width; ++x)
        {
          register int ycur, yoff ;
          int ByteCount, xpin ;
          char *dp ;

/* Clear the PinVals array */

          for (ByteCount = 0, dp = PinVals; ByteCount < BytesPerChunk;
               ++ByteCount)
            *dp++ = 0 ;

          dp = PinVals ;

/* For each byte-sized row of the print head, collect 1 pixel width of data. */

          for (ByteCount = 0, dp = PinVals, ycur = y;
               ByteCount < BytesPerColumn; ++ByteCount, dp  += xPinsPerPixel)
            {
              register unsigned char  bit ;

              yn = ycur + yPixelsPerByte ;
              if (yn > srcp->height) yn = srcp->height ;

/*  For the current byte row of the print-head (ie. yPixelsPerByte image
 *  scanlines), collect a pixel width of data.
 */

              for (bit = 0x80; ycur <  yn; ++ycur)
                {
                  pixel_t val ;
                  int ypin ;

                  yoff = ycur * srcp->width ;
                  if (colors == 1) val = img[0][yoff+x] ;  /* Grey level */
                  else                             /* Calculate NTSC luma */
                    val = img[0][yoff+x] * 0.299 +         /* R */
                          img[1][yoff+x] * 0.587 +         /* G */
                          img[2][yoff+x] * 0.114 ;         /* B */

/* Now use an appropriate no. of pins to simulate the greyscale value. */

                  for (ypin = 0, pin =  0; ypin < yPinsPerPixel; ++ypin)
                    {
                      for (xpin = 0; xpin < xPinsPerPixel; ++xpin, ++pin)
                        if (val < PinThresh[pin]) dp[xpin] |= bit ;
                      bit >>=  1 ;
                    }
                }
            }

/*  We have set up enough columns for a single pixel in the x direction.
 *  Now print them in the correct order.
 */

          for (xpin = 0; xpin < xPinsPerPixel; ++xpin)
            {
              for (ByteCount = 0; ByteCount < BytesPerColumn; ++ByteCount)
                PUTC(PinVals[ByteCount * xPinsPerPixel + xpin], ostr) ;
            }
        }

/* Printer specific */

      FPRINTF(ostr, "\033J%c\r", 24);
    }

  Efclose(ostr) ;
  free(PinVals) ;
  free((char *) PinThresh) ;
}


void
readimg(t)      /* :read "filename" [imagename] */
Tree *t ;
{
  char *imgname, *p ;

  if (t->kids[1]) imgname = lastimage ;
  else
    {
      for (p = imgname = lastfname; *p; ++p)
        if (*p == '/' && p[1] != '\0') imgname = p + 1 ;
    }
  getpix(lastfname, imgname, TRUE) ;
}


void
stopict(t)	/* :store [imagename] */
Tree *t ;
{
  char		*imgname ;
  char		noname[10] ;
  struct SRC	*img ;
  struct SRC	*unused = (struct SRC *) 0 ;
  int		i, ui ;

  if (t->kids[0]) imgname = lastimage ;
  else
    {
      sprintf(noname, "store_%d", nimages - 1) ;
      imgname = noname ;
    }

/* See if the named image exist, search for an unused image. */

  for (img = Images, i = ui = 0; img != &Images[nimages]; ++img, i++)
    {
      if (img->str && strcmp(img->str, imgname) == 0) break ;
      if (img->pix == (pixel_t **) NULL && unused == (struct SRC *) NULL)
        {
	  unused = img ;
	  ui = i - 1 ;
	}
    }

  if (img == &Images[nimages])
    {		                   /* Not found, copy to a new image. */
      if (ui)	                   /* Use an unused image. */
	{
	  img = unused ;
	  sprintf(imgname, "store_%d", ui) ;
	}
      else img = &Images[nimages++] ;	/* This is a really new one. */

      if ((img->pix = ImgAlloc(CurOld->width,
                               CurOld->height, CurOld->ncolors)) == 0 ||
          (img->str = (char *)
                      Emalloc((unsigned int) (strlen(imgname)+1))) == 0)
        return ;
      STRCPY(img->str, imgname) ;
    }
  else if (img->width   != CurOld->width || img->height != CurOld->height ||
           img->ncolors != CurOld->ncolors)
    {                          /* Existing image has changed size or depth. */
      ImgFree(img) ;
      if ((img->pix = ImgAlloc(CurOld->width,
                               CurOld->height, CurOld->ncolors)) == 0 ||
          (img->str = (char *)
                      Emalloc((unsigned int) (strlen(imgname)+1))) == 0)
        return ;

      STRCPY(img->str, imgname) ;
    }

  img->width   = CurOld->width ;	/* copy it */
  img->height  = CurOld->height ;
  img->ncolors = CurOld->ncolors ;
  for (i = 0; i < img->ncolors; ++i)
    {
      COPYMEM((char *) CurOld->pix[i], (char *) img->pix[i],
              img->width * img->height) ;
    }
}


void
writeimg(t)     /* :write "filename" [imagename] */
Tree *t ;
{
  struct SRC *img ;

  if (t->kids[1]) img = &Images[((Tree *) (t->kids[1]))->i] ;
  else            img = CurOld ;
  putpix(img, lastfname) ;
}


void
freeimg(t)
Tree *t ;
{
  struct SRC *srcp = (struct SRC *) 0 ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;

  if (srcp == (struct SRC *) 0) return ;
    
  if (srcp == CurOld || srcp == CurNew)
    {
      SPRINTF(ErrBuf, "Cannot free 'old' or 'new'") ;
      error(0) ;
      return ;
    }

  ImgFree(srcp) ;
}


void
displayimg(t)    /* :display [+|-image] */
Tree *t ;
{
  struct SRC *image ;

  if (t->kids[0])
    {
      switch (((Tree *) (t->kids[0]))->t)
        {
          case T_Plus  : disp_active = 1 ;
                         return ;
          case T_Minus : disp_active = 0 ;
                         return ;
          case T_Inum  : image = &Images[((Tree *) (t->kids[0]))->i] ;
			 break ;
	  default:       FPRINTF(stderr,"*** invalid tree in displayimg\n");
        }
    }
  else image = CurOld ;
  drawimg(image) ;
}


static void
CloseLog(filep)
FILE **filep ;
{
  if (*filep == NULL) return ;

  FPRINTF(*filep, "\n---\n") ;
  FCLOSE(*filep) ;
  *filep = NULL ;
}


void
OpenLog(filep)
FILE **filep ;
{
  time_t time() ;
  time_t clock ;

  CloseLog(filep) ;

  if ((*filep = fopen(LogFile, "a")) == NULL)
    {
      SPRINTF(ErrBuf,
              "Can't open log file '%s' - logging is off", LogFile) ;
      error(ERR_SYS) ;
      return ;
    }

  clock = time((time_t *) 0) ;
  FPRINTF(*filep, "==>> %s", ctime(&clock)) ;    /* Includes '\n' */
}


void
dolog(t)
Tree *t ;
{
  static char *buf = (char *) 0 ;

  if (t->kids[0])
    {
      switch (((Tree *) (t->kids[0]))->t)
        {
          case T_Plus  : OpenLog(&LogStr) ;
                         break ;
          case T_Minus : CloseLog(&LogStr) ;
                         break ;
          case T_File  : if (buf == (char *) 0 &&
                             (buf = Emalloc((unsigned int) 512)) == (char *) 0)
                           return ;
                         STRCPY(buf, lastfname) ;
                         LogFile = buf ;
                         OpenLog(&LogStr) ;
			 break ;
	  default:       FPRINTF(stderr, "*** invalid tree in dolog\n") ;
        }
    }

  if (LogStr) PRINTF("Logging is active on file '%s'\n", LogFile) ;
  else        PRINTF("Logging is off\n") ;
}


void
debug(t)
Tree *t ;
{
  static char *buf = (char *) 0 ;

  if (t->kids[0])
    {
      switch (((Tree *) (t->kids[0]))->t)
        {
          case T_Plus  : OpenLog(&Debug) ;
                         break ;
          case T_Minus : CloseLog(&Debug) ;
                         break ;
          case T_File  : if (buf == (char *) 0 &&
                             (buf = Emalloc((unsigned int) 512)) == (char *) 0)
                           return ;
                         STRCPY(buf, lastfname) ;
                         LogFile = buf ;
                         OpenLog(&Debug) ;
			 break ;
          default:       FPRINTF(stderr, "*** invalid tree in debug\n") ;
        }
    }

  if (Debug) PRINTF("Logging is active on file '%s'\n", LogFile) ;
  else       PRINTF("Logging is off\n") ;
}


static char *HelpMsg[] = {
  "binary ops:   ** * /  % + - << >> < > >= <= == != & ^ | && ||",
  "funcs:        sin(deg) cos(deg) atan(y, x) hypot(x, y)",
  "              log(val) sqrt(val) abs(val) rand()",
  "values:       x y r a X Y R A Z",
  "special functions:",
  "\t?",
  "\tq",
  "",
  "\t:color\t\t\t\t:debug      [+|-|\"filename\"]",
  "\t:display    [+|-|image]\t\t:exit",
  "\t:free       image\t\t:genepson   \"filename\" [image]",
  "\t:genps      \"filename\" [image]\t:grayscale",
  "\t:help\t\t\t\t:list",
  "\t:logfile    [+|-|\"filename\"]\t:matte      [image [gamma]]",
  "\t:melt       [image]\t\t:monochrome",
  "\t:ofmt       [+|-]\t\t:oil        [image]",
  "\t:quit\t\t\t\t:read       \"filename\" [image]",
  "\t:shear      [image]\t\t:signed     [+|-]",
  "\t:slice      [image]\t\t:store      [image]",
  "\t:tile       [image]\t\t:truncate   [+|-]",
  "\t:undo\t\t\t\t:verbose    [+|-]",
  "\t:version\t\t\t:write      \"filename\" [image]",
  (char *) 0
} ;


void
help()
{
  PrStrs(HelpMsg) ;
}


void
setcolor()
{
  if (scr_depth > 1) dtype = IS_COLOR ;
}


void
setgray()
{
  if (scr_depth > 1) dtype = IS_GRAY ;
}


void
setmono()
{
  dtype = IS_MONO ;
}


void
undo()
{
  FPRINTF(stderr, "undo.\n") ;
  SwapOldNew() ;
}


void
verbose(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  Verbose = 1 ;
      else if (op == T_Minus) Verbose = 0 ;
    }
  PRINTF("Verbose is %s\n", Verbose ? "on" : "off") ;
}


void
imtrunc(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  Truncate = 1 ;
      else if (op == T_Minus) Truncate = 0 ; 
    }
  PRINTF("Truncation is %s\n", Truncate ? "on" : "off") ;
}


void
dosigned(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  signed_io = 1 ;
      else if (op == T_Minus) signed_io = 0 ; 
    }
  PRINTF("Signed I/O is %s\n", signed_io ? "on" : "off") ;
}


void
ofmt(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  oldfmt = 1 ;
      else if (op == T_Minus) oldfmt = 0 ; 
    } 
  PRINTF("Old output format is %s\n", oldfmt ? "on" : "off") ;
}
