
/*  @(#)sparcgen.c 1.2 92/01/05
 *
 *  Popi sparc runtime assembler generation routines.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include <sys/exec.h>
#include <unistd.h>
#include <string.h>
#include "popi.h"
#include "expr.h"

#define DUMPFILE "popi.out"

typedef unsigned long Inst ;

static void gen       P((Tree *)) ;
static void binop     P((Tree *, int, int, int)) ;
static void binconst  P((Tree *, int, int, int)) ;
static void dodump    P((Inst *, int)) ;
static void clearregs P((void)) ;
static void freereg   P((int)) ;

static int findreg    P((void)) ;
static int mul        P((int, int)) ;
static int Pdiv       P((int, int)) ;
static int mod        P((int, int)) ;

static Inst *code ;
static ccset, boolean, last = 18, regs[6], spp, sppmax, result ;

#define  zero    0
#define  tmp     1
#define  base    24        /* first arg */
#define  offset  25        /* etc. */
#define  a_mul   26
#define  a_div   27
#define  a_mod   28
#define  rx      16
#define  ry      17
#define  sp      14
#define  o0      8        /* first arg out */
#define  o1      9

#define  OP(rd, op, rs, i, r_or_d)  \
                ((2 << 30) | (rd << 25) | (op << 19) | (rs << 14) | \
                 (i << 13) | (r_or_d & 017777))
#define  MOP(rd, op, rs, i, r_or_d) \
                ((3 << 30) | (rd << 25) | (op << 19) | (rs << 14) | \
                 (i << 13) | (r_or_d & 017777))
#define  SETHI(rd, op, imm)  ((rd << 25) | (op << 22) | (imm & 017777777))

#define  _ADD(a, b, c)     *code++ = OP(c, 000, a, 0, b)
#define  _ADDcc(a, b, c)   *code++ = OP(c, 020, a, 0, b)
#define  _SUB(a, b, c)     *code++ = OP(c, 004, a, 0, b)
#define  _SUBcc(a, b, c)   *code++ = OP(c, 024, a, 0, b)
#define  _XOR(a, b, c)     *code++ = OP(c, 003, a, 0, b)
#define  _XORcc(a, b, c)   *code++ = OP(c, 023, a, 0, b)
#define  _OR(a, b, c)      *code++ = OP(c, 002, a, 0, b)
#define  _ORcc(a, b, c)    *code++ = OP(c, 022, a, 0, b)
#define  _AND(a, b, c)     *code++ = OP(c, 001, a, 0, b)
#define  _ANDcc(a, b, c)   *code++ = OP(c, 021, a, 0, b)
#define  _NOR(a, b, c)     *code++ = OP(c, 006, a, 0, b)
#define  _XORI(a, b, c)    *code++ = OP(c, 003, a, 1, b)
#define  _XORIcc(a, b, c)  *code++ = OP(c, 023, a, 1, b)
#define  _ORI(a, b, c)     *code++ = OP(c, 002, a, 1, b)
#define  _ORIcc(a, b, c)   *code++ = OP(c, 022, a, 1, b)
#define  _ADDI(a, b, c)    *code++ = OP(c, 000, a, 1, b)
#define  _ADDIcc(a, b, c)  *code++ = OP(c, 020, a, 1, b)
#define  _ANDI(a, b, c)    *code++ = OP(c, 001, a, 1, b)
#define  _ANDIcc(a, b, c)  *code++ = OP(c, 021, a, 1, b)
#define  _LBU(a, b, c)     *code++ = MOP(a, 001, b, 0, c)
#define  _LBIU(a, b, c)    *code++ = MOP(a, 001, b, 1, c)
#define  _SB(a, b, c)      *code++ = MOP(a, 005, b, 0, c)
#define  _SBI(a, b, c)     *code++ = MOP(a, 005, b, 1, c)
#define  _LW(a, b, c)      *code++ = MOP(a, 000, b, 0, c)
#define  _LWI(a, b, c)     *code++ = MOP(a, 000, b, 1, c)
#define  _SW(a, b, c)      *code++ = MOP(a, 004, b, 0, c)
#define  _SWI(a, b, c)     *code++ = MOP(a, 004, b, 1, c)
#define  _SETHI(a, b)      *code++ = SETHI(a, 04, b >> 10)
#define  _SLL(a, b, c)     *code++ = OP(c, 045, a, 0, b)
#define  _SRL(a, b, c)     *code++ = OP(c, 046, a, 0, b)
#define  _SLLI(a, b, c)    *code++ = OP(c, 045, a, 1, b)
#define  _SRLI(a, b, c)    *code++ = OP(c, 046, a, 1, b)

#define  _CMP(a, b)    *code++ = OP(zero,024,a,0,b)  /* _SUBcc(a,b,zero) ; */
#define  _CMPI(a, b)   *code++ = OP(zero,024,a,1,b)  /* _SUBIcc(a,b,zero) ; */

#define  _B(cond, disp)    SETHI(cond, 02, disp)
#define  ALWAYS            010
#define  EQ                001
#define  NE                011
#define  GE                013
#define  LE                002
#define  GT                012
#define  LT                003
#define  _JAL(a)           *code++ = OP(15, 070, a, 0, zero)
#define  _SAVE(a)          *code++ = OP(sp, 074, sp, 1, a)
#define  _RET              *code++ = 0x81c7e008    /* jmpl      %o7+8,%g0  */
#define  _RESTORE          *code++ = 0x81e80000    /* restore   %g0,%g0,%g0  */
#define  _NOP              *code++ = SETHI(zero, 04, 0)

#define  _PUSH(a) { \
                    _SW(a, sp, spp) ; \
                    spp += 4 ; \
                    if (spp > sppmax) \
                      sppmax = spp ; \
                  }

#define  _POP(a) { \
                   spp -= 4 ; \
                   _LW(a, sp, spp) ; \
                 }

static void
loadimm(i, j)
int i, j ;
{
  if (j > 0x1777 || j < 0)
    {
      _SETHI(i, j) ;
      _ORIcc(i, (j & 01777), i) ;
    }
  else _ORIcc(zero, j, i) ;
}


void
compile(t)
Tree *t ;
{
  char dumpstr[MAXLINE] ;
  int color, ino ;
  Inst prog[1024], *remem1, *remem2 ;
  void (*progp)() = (void (*)()) prog ;

  code = prog ;
  clearregs() ;
  last = findreg() ;
  code = prog ;
  _SAVE(0) ;                 /* Value is or'ed in later. */
  loadimm(ry, Ysize-1) ;
  remem1 = code ;
  loadimm(rx, Xsize-1) ;
  remem2 = code ;
  gen(t) ;
  _SB(result, base, offset) ;
  _ADDI(offset, -1, offset) ;
  _CMP(rx, zero) ;
  *code = _B(NE, remem2-code) ;
  code++ ;
  _ADDI(rx, -1, rx) ;
  _CMP(ry, zero) ;
  *code = _B(NE, remem1-code) ;
  code++ ;
  _ADDI(ry, -1, ry) ;
  _RET ;
  _RESTORE ;
  *prog |= (-(sppmax + 80)) & 017777 ;
  dodump(prog, (code - prog) * sizeof (Inst)) ;
  ino = getImageNo("new") ;
  for (color = 0; color < colors; color++)
    progp(Images[ino].pix[color], (Xsize * Ysize) - 1, mul, Pdiv, mod) ;

  if (Debug)
    {
      SPRINTF(dumpstr, "echo ',-1?' | adb %s", DUMPFILE) ;
      system(dumpstr) ;
    }

  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


static int
mul(a, b)
int a, b ;
{
  return(a * b) ;
}


static int
Pdiv(a, b)
int a, b ;
{
  return(a / b) ;
}


static int
mod(a, b)
int a, b ;
{
  return(a % b) ;
}

static void
gen(t)
Tree *t ;
{
  Inst *remem ;
  int i, saved, b1, b2, c1, c2 ;

  switch (t->t)
    {
      case T_Inum   : if (t->i == 0)
                        {
                          result = zero ;
                          ccset = 0 ;
                          boolean = 1 ;
                        }
                      else
                        {
                          loadimm(result = last, t->i) ;
                          boolean = !(t->i & ~1) ;
                          ccset = 1 ;
                        }
                      break ;
      case T_Xcoord : result = rx ;
                      boolean = 0 ;
                      ccset = 0 ;
                      break ;
      case T_Ycoord : result = ry ;
                      boolean = 0 ;
                      ccset = 0 ;
                      break ;
      case T_Bang   : gen(t->kids[0]) ;
                      if (!ccset) _CMP(result, zero) ;
                      *code++ = _B(EQ, 3) ;
                      _ADDIcc(zero, 1, last) ;
                      _ADDIcc(zero, 0, last) ;
                      result = last ;
                      boolean = 1 ;
                      ccset = 1 ;
                      break ;
      case T_Not    : gen(t->kids[0]) ;
                      _XORIcc(result, -1, last) ;
                      result = last ;
                      boolean = 0 ;
                      ccset = 1 ;
                      break ;
      case T_Neg    : gen(t->kids[0]) ;
                      _SUBcc(zero, result, last) ;
                      result = last ;
                      boolean = 0 ;
                      ccset = 1 ;
                      break ;
      case T_Cond   : gen(t->kids[0]) ;                   /* test */
                      if (!ccset) _CMP(result, zero) ;
                      remem = code ;
                      *code++ = _B(EQ, 0) ;
                      _NOP ;
                      gen(t->kids[1]) ;                   /* iftrue */
                      b1 = boolean ;
                      c1 = ccset ;
                      if (result != last) _ADD(result, zero, last) ;
                      *remem |= code - remem + 1 ;
                      remem = code - 1 ;
                      *code = code[-1] ;
                      code[-1] = _B(ALWAYS, 0) ;          /* fill delay slot */
                      code++ ;
                      gen(t->kids[2]) ;                   /* iffalse */
                      b2 = boolean ;
                      c2 = ccset ;
                      if (result != last) _ADD(result, zero, last) ;
                      result = last ;
                      boolean = (b1 & b2) ;
                      ccset = (c1 & c2) ;
                      *remem |= code - remem ;
                      break ;
      case T_Land   :
      case T_Lor    : gen(t->kids[0]) ;
                      b1 = boolean ;
                           if (result != last) _ADDcc(result, zero, last) ;
                      else if (!ccset)         _CMP(result, zero) ;
                      saved = last ;
                      remem = code ;
                      if (t->t == T_Land) *code++ = _B(EQ, 0) ;
                      else                *code++ = _B(NE, 0) ;
                      _NOP ;
                      gen(t->kids[1]) ;
                      b2 = boolean ;
                      if (result != saved)
                        {
                          _ADDcc(result, zero, saved) ;
                          ccset = 1 ;
                        }
                      result = saved ;
                      *remem |= code - remem ;
                      if (!(b1 & b2))          /* "normalize" expr to {0, 1} */
                        {
                          if (!ccset) _CMP(result, 0) ;
                          *code++ = _B(NE, 3) ;
                          _ADDIcc(zero, 0, result) ;
                          _ADDIcc(zero, 1, result) ;
                          ccset = 1 ;
                        }
                      boolean = 1 ;
                      break ;
      case T_Coord  : if (t->i > nimages || t->i < 0)
                        run_error("no such buffer") ;
                      if (t->kids[0] != NULL)
                        {
                          gen(t->kids[0]) ;
                          i = last ;
                        }
                      else i = offset ;
                      loadimm(tmp, (int) Images[t->i].pix[0]) ;
                      _LBU(last, tmp, i) ;
                      result = last ;
                      ccset = 0 ;
                      boolean = 0 ;
                      break ;
      default       : gen(t->kids[0]) ;
                      saved = result ;
                      if (((Tree *) t->kids[1])->t == T_Inum &&
                          (i = ((Tree *) t->kids[1])->i) < 4096 && i > -4096)
                        {
                          binconst(t, result, i, last) ;
                        }
                      else
                        {
                          if (saved != rx && saved != ry &&
                              (last = findreg()) == -1)
                            {
                              last = saved ;
                              saved = -1 ;
                              _PUSH(last) ;
                            }
                          gen(t->kids[1]) ;
                          if (saved == -1)
                            {
                              _POP(tmp) ;
                              binop(t, tmp, result, last) ;
                            }
                          else
                            {
                              binop(t, saved, result, last) ;
                              if (saved != rx && saved != ry) freereg(saved) ;
                            }
                        }
                      result = last ;
    }
}


static void
binop(t, s1, s2, d)
Tree *t ;
int s1, s2, d ;
{
  boolean = 0 ;
  ccset = 1 ;
  switch (t->t)
    {
      default       : run_error("not a binop in binop!") ;
      case T_Add    : _ADDcc(s1, s2, d) ;
                      break ;
      case T_Sub    : _SUBcc(s1, s2, d) ;
                      break ;
      case T_Mul    : _ADD(zero, s1, o0) ;
                      _JAL(a_mul) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Div    : _ADD(zero, s1, o0) ;
                      _JAL(a_div) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Mod    : _ADD(zero, s1, o0) ;
                      _JAL(a_mod) ;
                      _ADD(zero, s2, o1) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Gt     : _CMP(s1, s2) ;
                      *code++ = _B(GT, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Lt     : _CMP(s1, s2) ;
                      *code++ = _B(LT, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ge     : _CMP(s1,s2) ;
                      *code++ = _B(GE, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Le     : _CMP(s1, s2) ;
                      *code++ = _B(LE, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Eq     : _CMP(s1, s2) ;
                      *code++ = _B(EQ, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ne     : _CMP(s1, s2) ;
                      *code++ = _B(NE, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Xor    : _XORcc(s1, s2, d) ;
                      break ;
      case T_And  : _ANDcc(s1, s2, d) ;
                      break ;
      case T_Or     : _ORcc(s1, s2, d) ;
                      break ;
      case T_Lshift : _SLL(s1, s2, d) ;
                      ccset = 0 ;
                      break ;
      case T_Rshift : _SRL(s1, s2, d) ;
                      ccset = 0 ;
    }
}


static void
binconst(t, s1, s2, d)
Tree *t ;
int s1, s2, d ;
{
  int shift ;

  boolean = 0 ;
  ccset = 1 ;
  switch (t->t)
    {
      default       : run_error("not a binop in binconst!") ;
      case T_Add    : _ADDIcc(s1, s2, d) ;
                      break ;
      case T_Sub    : _ADDIcc(s1, -s2, d) ;
                      break ;
      case T_Mul    :
      case T_Div    : if (s2 < 0 || s2 & (s2 - 1) != 0)
                        {
                          loadimm(s2, o1) ;
                          _JAL(((t->t == T_Mul) ? a_mul : a_div)) ;
                          _ADD(zero, s1, o0) ;
                          _ADDcc(zero, o0, d) ;
                        }
                      else
                        {
                          for (shift = 0 ; s2 != 0 ; shift++, s2 >>= 1)
                            continue ;
                          if (shift == 0)
                            {
                              if (t->t == T_Mul) loadimm(d, 0) ;
                              else run_error("division by zero") ;
                              return ;
                            }
                          else if (t->t == T_Mul)
                            {
                              _SLLI(s1, --shift, d) ;
                            }
                          else
                            {
                              _SRLI(s1, --shift, d) ;
                            }
                          ccset = 0 ;
                        }
                      break ;
      case T_Mod    : loadimm(s2, o1) ;
                      _JAL(a_mod) ;
                      _ADD(zero, s1, o0) ;
                      _ADDcc(zero, o0, d) ;
                      break ;
      case T_Gt     : _CMPI(s1, s2) ;
                      *code++ = _B(GT, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Lt     : _CMPI(s1, s2) ;
                      *code++ = _B(LT, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ge     : _CMPI(s1, s2) ;
                      *code++ = _B(GE, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Le     : _CMPI(s1, s2) ;
                      *code++ = _B(LE, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Eq     : _CMPI(s1, s2) ;
                      *code++ = _B(EQ, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Ne     : _CMPI(s1, s2) ;
                      *code++ = _B(NE, 3) ;
                      _ADDIcc(zero, 0, d) ;
                      _ADDIcc(zero, 1, d) ;
                      boolean = 1 ;
                      break ;
      case T_Xor    : _XORIcc(s1, s2, d) ;
                      break ;
      case T_And    : _ANDIcc(s1, s2, d) ;
                      break ;
      case T_Or     : _ORIcc(s1, s2, d) ;
                      break ;
      case T_Lshift : _SLLI(s1, s2, d) ;
                      ccset = 0 ;
                      break ;
      case T_Rshift : _SRLI(s1, s2, d) ;
                      ccset = 0 ;
                      break ;
    }
}


static void
dodump(prog, size)
Inst *prog ;
int size ;
{
  if (debug)
    {
      int fd = creat(DUMPFILE, 0644) ;
      struct exec exec ;

      exec.a_machtype = M_SPARC ;
      exec.a_magic    = OMAGIC ;
      exec.a_text     = size ;
      exec.a_data     = 0 ;
      exec.a_bss      = 0 ;
      exec.a_syms     = 0 ;
      exec.a_entry    = 0 ;
      exec.a_trsize   = 0 ;
      exec.a_drsize   = 0 ;
      if (fd < 0)
        {
          perror(DUMPFILE) ;
          exit(1) ;
        }
      write(fd, &exec, sizeof(exec)) ;
      write(fd, prog, size) ;
      close(fd) ;
    }
}


static void
clearregs()
{
  int i ;

  for (i = 0 ; i < 6 ; i++) regs[i] = 0 ;
}


static void
freereg(i)
int i ;
{
  regs[i-18] = 0 ;
}


static int
findreg()
{
  int i ;

  for (i = 0 ; i < 6 ; i++)
    if (regs[i] == 0)
      break ;
  if (i == 6) return(-1) ;
  regs[i] = 1 ;
  return(i+18) ;
}
