/**********************************************************************/
/* mprims.h :                                                         */
/*                                                                    */
/* Meshed primitive types : Keep track of geometry only, and assume   */
/* all primitives fit into unit cube                                  */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#ifndef MPRIMS_H
#define MPRIMS_H

#define NIL 0
#define X 0
#define Y 1
#define Z 2
#define W 3

#define DEG (M_PI/180.0) /* radians to degrees */
#define MIN_COORD 1.0e-6

#define MCONE     10
#define MCUBE     11
#define MCYLINDER 12
#define MSPHERE   13

/**********************************************************************/
/* Direction of normals */
/**********************************************************************/
#define NORMAL_OUT 1
#define NORMAL_IN 0
int normal_direction = NORMAL_OUT;

/**********************************************************************/
/* Meshed primitive types */
/**********************************************************************/
typedef struct { double x, y, z; } Vector;
typedef struct {
  Vector vertex[4]; /* Polygon vertices */
  Vector normal[4]; /* Polygon normals */
  int size;         /* # of vertices */
} Polygon;

typedef struct {
  double vertex[4][3]; /* Polygon vertices */
  double normal[4][3]; /* Polygon normals */
  int size;         /* # of vertices */
} CPolygon;

/**********************************************************************/
/* meshed sphere */
/**********************************************************************/
typedef struct {
  Polygon *poly;
  int num_sub;        /* Number subdivisions along lat / long         */
  int num_polys;      /* Number of patchs in mesh                     */
  double r;           /* radius */
  Vector centre;      /* Centre */
  int id;             /* Id */
  char *sid;
} MSphere;

/**********************************************************************/
/* Meshed Cone */
/**********************************************************************/
typedef struct {
  Polygon *bpoly;     /* Base polygons */
  Polygon *tpoly;     /* Top polygons (if any) */
  Polygon *spoly;     /* Side polygons */
  double rb, rt, h;   /* Radius and height */
  int num_side_polys; /* # of polys on side of cone */
  int num_disc_polys; /* # of polys on base of cone */
  int id;             /* Id */
  char *sid;
} MCone;

/**********************************************************************/
/* meshed cylinder : height is from (0,0,-1) to (0,0,1), and */
/* width is from (-1,0,0) to (1,0,0) */
/**********************************************************************/
typedef struct {
  double r;           /* Radius                                       */
  double h;           /* Height                                       */
  Polygon *spoly;     /* The mesh representing the side of the cyl.   */
  Polygon *tpoly;     /* The mesh representing the top of the cyl.    */
  Polygon *bpoly;     /* The mesh representing the top of the cyl.    */
  int num_sub;        /* Number subdivisions along h and r            */
  int num_side_polys; /* Number of patchs in mesh for sides           */
  int num_disc_polys; /* Number of patchs in mesh for top and bottom  */
  int id;             /* Id */
  char *sid;
} MCylinder;

/**********************************************************************/
/* Meshed unit cube */
/**********************************************************************/
typedef struct {
  CPolygon *faces[6]; /* 6 mesh faces                                 */
  int du, dv;         /* Amount of subdivision per face               */
  int num_polys;      /* Number of patchs in mesh                     */
  int id;             /* Id */
  char *sid;
} MCube;

/**********************************************************************/
/* Meshed octahedron */
/**********************************************************************/
typedef struct {
  float points[6][3];
  unsigned long colour[6];
  int num_polys;      /* Number of patchs in mesh                     */
  int id;             /* Id */
  char *sid;
} MOctahedron;

/**********************************************************************/
/* Meshed torus */
/**********************************************************************/
#define MAX_LATRES 16	/* maximum latitude resolution */
#define MAX_LONRES 36	/* maximum longitude resolution */

typedef struct {
  int id;
  char *sid;
  double rad;           /* ratio of major radius to minor radius */
  double scale;         /* shrink object to fit into unit square */
  int latres, longres;
  float points[MAX_LATRES+1][MAX_LONRES+1][3];  /* meshed points, normals, */
  float normals[MAX_LATRES+1][MAX_LONRES+1][3]; /* and colours */
  float colours[MAX_LATRES+1][MAX_LONRES+1][3];
  int num_polys;      /* Number of patchs in mesh                     */
} MTorus;

/**********************************************************************/
/* Surface Texture */
/**********************************************************************/
typedef struct {
  double u,v;
} TextureType;

#endif MPRIMS_H
