/**********************************************************************/
/* image.c                                                            */
/*                                                                    */
/* Mostly gathering of routines written by Paul Haeberli (1985) for   */
/* IRIS image storage and display                                     */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          *
/**********************************************************************/
#include <stdio.h>
#include <gl/gl.h>
#include <device.h>
#include <gl/image.h>
#include <sys/sysmacros.h>

/**********************************************************************/
IMAGE *image;                    /* Image */
unsigned short rbuf[4096];       /* RGB buffers for reading / writing */
unsigned short gbuf[4096];
unsigned short bbuf[4096];
unsigned char Rbuf[4096];          /* RGB buffers for display */      
unsigned char Gbuf[4096];
unsigned char Bbuf[4096];

/**********************************************************************/
/* Prototypes                                                         */
/**********************************************************************/
void Short_To_Char();
void Char_To_Short();
void RGB_To_Gray();
void Write_Image();
void Show_Image();
void Draw_Image();
void Save_Screen_Image();

/**********************************************************************/
/* Write image out in IRIS format */
/**********************************************************************/
void Write_Image(image_file, xsize, ysize)
     char *image_file;
     int xsize, ysize;
{
  int y;

  /* Open image file and write out a row at a time */
  image = iopen(image_file, "w", RLE(1), 3, xsize, ysize, 3);
  for (y = 0; y < ysize; y++) {
    putrow(image, rbuf, y, 0);	/* red row */
    putrow(image, gbuf, y, 1);	/* green row */
    putrow(image, bbuf, y, 2);	/* blue row */
  }
  iclose(image);
}

/**********************************************************************/
/* Read in and show an IRIS image */
/**********************************************************************/
void Show_Image(image_file)
     char *image_file;
{
  short val;
  int xsize, ysize, zsize;

  if ((image = iopen(image_file, "r")) == NULL) {
    printf("Can't open input file %s\n", image_file);
    exit(1);
  }
  xsize = image->xsize;
  ysize = image->ysize;
  zsize = image->zsize;
  if (xsize > (XMAXSCREEN + 1) || ysize > (YMAXSCREEN + 1)) {
    printf("Input image is too large: %d %d", xsize, ysize);
    exit(1);
  }
  prefsize(xsize, ysize);
  winopen(image_file);
  RGBmode();
  gconfig();
  RGBwritemask(0xffff, 0xffff, 0xffff);
  Draw_Image(xsize, ysize);
  while (1) {
    if (qread(&val) == REDRAW)
      Draw_Image();
  }
}

/**********************************************************************/
/* Draw the image */
/**********************************************************************/
void Draw_Image(xsize, ysize, zsize)
     int xsize, ysize, zsize;
{
  int y;

  reshapeviewport();
  viewport(0, xsize - 1, 0, ysize - 1);
  ortho2(-0.5, (float) xsize - 0.5, -0.5, (float) ysize - 0.5);

  for (y = 0; y < ysize; y++) {
    
    if (zsize < 3) {          /* Mono */
      getrow(image, rbuf, y, 0); Short_To_Char(rbuf, Rbuf, xsize);
      cmov2i(0, y);
      writeRGB(xsize, Rbuf, Rbuf, Rbuf);

    } else {                 /* Colour */
      getrow(image, rbuf, y, 0); Short_To_Char(rbuf, Rbuf, xsize);
      getrow(image, gbuf, y, 1); Short_To_Char(gbuf, Gbuf, xsize);
      getrow(image, bbuf, y, 2); Short_To_Char(bbuf, Bbuf, xsize);
      cmov2i(0, y);
      writeRGB(xsize, Rbuf, Gbuf, Bbuf);
    }
  }
}

/**********************************************************************/
/* Draw the image */
/**********************************************************************/
void Save_Screen_Image(xsize, ysize, zsize, image_file)
     int xsize, ysize, zsize;
     char *image_file;
{
  int x,y;
  unsigned long larray[4096]; /* Buffer for reading display */
  unsigned short obuf[4096];  /* Buffer for gray-scale conversion */

  if (zsize < 3) {          /* Gray */
    image = iopen(image_file, "w", RLE(1), 3, xsize, ysize, 1);
    image->colormap = CM_NORMAL;
  } else {
    image = iopen(image_file, "w", RLE(1), 3, xsize, ysize, 3);
  }

  readsource(SRC_FRONT);
  for (y = 0; y < ysize; y++) {
    lrectread(0, y, xsize-1, y, larray);
    
    /* Extract RGB values */
    for (x=0;x<xsize;x++) {
      /* printf("l[%d] = %d\n", x, larray[x]); */
      rbuf[x] = 0x0000ff & larray[x]; 
      /* printf("r[%d]=%d ", x, rbuf[x]);  */
      gbuf[x] = (0x00ff00 & larray[x]) >> 8 ; 
      /* printf("g[%d]=%d ", x, gbuf[x]);  */
      bbuf[x] = (0xff0000 & larray[x]) >> 16; 
      /* printf("b[%d]=%d\n", x, bbuf[x]); */
    }

    if (zsize == 3) { /* Store RGB colour */
      putrow(image, rbuf, y, 0); 
      putrow(image, gbuf, y, 1);
      putrow(image, bbuf, y, 2);
    } 
    else { /* Store gray-scale */
      RGB_To_Gray(rbuf,gbuf,bbuf,obuf,xsize);
      putrow(image, obuf, y, 0);
    }
  }
  iclose(image);
}

/**********************************************************************/
/* Change short -> unsigned char */
/**********************************************************************/
void Short_To_Char(sptr, bptr, n)
     register unsigned short *sptr;
     register unsigned char *bptr;
     short n;
{ while (n--) *bptr++ = *sptr++; }

/**********************************************************************/
/* Change unsigned char -> short */
/**********************************************************************/
void Char_To_Short(bptr, sptr, n)
     register unsigned short *sptr;
     register unsigned char *bptr;
     short n;
{ while (n--) *sptr++ = *bptr++; }

#define R_ratio 77
#define B_ratio 28
#define G_ratio 151
/**********************************************************************/
/* "Compress" rgb to gray-scale */
/**********************************************************************/
void RGB_To_Gray(rbuf,gbuf,bbuf,obuf,n)
     register unsigned short *rbuf, *gbuf, *bbuf, *obuf;
     int n;
{
  int i;
  
  for(i=n; i--; ) 
    *obuf++ = (R_ratio*(*rbuf++) + G_ratio*(*gbuf++) + B_ratio*(*bbuf++))>>8;
}
