/*
 * sipp.h  
 *
 * Copyright  Jonas Yngvesson
 *            Inge Wallin
 */


#ifndef _SIPP_H
#define _SIPP_H

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#ifndef FALSE
typedef unsigned char bool;
#define FALSE  0
#define TRUE   1
#endif 

/*
 * Customize for those that don't have memcpy() and friends, but
 * have bcopy() instead.
 */

#ifdef NOMEMCPY
#define memcpy(to, from, n) bcopy((from), (to), (n))
#endif


/*
 * The macro RANDOM() should return a random number
 * in the range [-1, 1].
 */
#define RANDOM()  (2.0 * drand48() - 1.0)


/*
 * Interface to shader functions.
 */
typedef void Shader();


/*
 * Colors are handled as an rgb-triple
 * with values between 0 and 1.
 */
typedef struct {
    double   red;
    double   grn;
    double   blu;
} Color;


/*
 * A transformation matrix is kept as a
 * 4x3 matrix (4x4 is only needed in the last
 * pesprctive transformation.
 * Arrays are not first class objects in C so
 * we surround it with a struct.
 */
typedef struct {
    double   mat[4][3];
} Transf_mat;


typedef struct {
    double x, y, z;
} Vector;


/*
 * Structure storing the vertices in surfaces. The vertices for a
 * surface are stored in a binary tree sorted first on x, then y and last z.
 */
typedef struct vertex_t {
    double            x, y, z;    /* vertex position */
    double            a, b, c;    /* average normal at vertex */
    double            u, v, w;    /* texture parameters (if any) */
    struct vertex_t  *big, *sml;  /* pointers to children in the tree */
} Vertex;


/*
 * Structure to keep a list of vertex references.
 */
typedef struct vertex_ref_t {
    Vertex       *vertex;
    struct vertex_ref_t *next;
} Vertex_ref;


/*
 * Polygon definition. A polygon is defined by a list of
 * references to its vertices (counterclockwize order).
 */
typedef struct polygon_t {
    Vertex_ref *vertices;     /* vertex list */
    bool        backface;   /* polygon is backfacing (used at rendering) */
    struct polygon_t  *next;
} Polygon;


/*
 * Surface definition. Each surface consists of a vertex tree, 
 * a polygon list, a pointer to a surface description and a pointer
 * to a shader function.
 */
typedef struct surface_t {
    Vertex           *vertices;          /* vertex tree */
    Polygon          *polygons;          /* polygon list */
    void             *surface;           /* surface description */
    Shader           *shader;            /* shader function */
    int               ref_count;         /* no of references to this surface */
    struct surface_t *next;              /* next surface in the list */
} Surface;


/*
 * Object definition. Object consists of one or more
 * surfaces and/or one or more subojects. Each object
 * has its own transformation matrix that affects itself
 * and all its subobjects.
 */
typedef struct object_t {
    Surface         *surfaces;       /* List of surfaces */
    struct object_t *sub_obj;        /* List of subobjects */
    Transf_mat       transf;         /* Transformation matrix */
    int              ref_count;      /* No of references to this object */
    struct object_t *next;           /* Next object in this list */
} Object;



/*
 * Lightsource definition. 
 */
typedef struct lightsource {
    double              intensity;  /* intensity, same for r, g, b */
    Vector              dir;        /* direction from origo */  
    struct lightsource *next;       /* next lightsource in the list */
} Lightsource;


/*
 * Surface description used by the basic shader. This shader
 * does simple shading of surfaces of a single color.
 */
typedef struct {
    double  ambient;       /* Fraction of color visible in ambient light */
    double  specular;      /* Fraction of colour specularly reflected */
    double  c3;            /* "Shinyness" 0 = shiny,  1 = dull */
    Color   color;         /* Colour of the surface */
} Surf_desc;



/*
 * This defines all public functions implemented in sipp.
 */
extern void     sipp_init();
extern void     vertex_push();
extern void     vertex_tx_push();
extern void     polygon_push();
extern Surface *surface_create();
extern Surface *surface_basic_create();
extern void     surface_set_shader();
extern void     surface_basic_shader();
extern Object  *object_create();
extern Object  *object_instance();
extern Object  *object_dup();
extern Object  *object_deep_dup();
extern void     object_delete();
extern void     object_add_surface();
extern void     object_add_subobj();
extern void     object_install();
extern void     object_uninstall();
extern void     object_set_transf();
extern void     object_clear_transf();
extern void     object_transform();
extern void     object_rot_x();
extern void     object_rot_y();
extern void     object_rot_z();
extern void     object_rot();
extern void     object_scale();
extern void     object_move();
extern void     lightsource_push();
extern void     view_from();
extern void     view_at();
extern void     view_up();
extern void     view_focal();
extern void     viewpoint();
extern void     render_image();
extern void     basic_shader();

#endif /* _SIPP_H */
