.\" Copyright 1990, Jonas Yngvesson, Inge Wallin
.\" This program and documentation may be distributed freely under
.\" the terms of GNU GENERAL PUBLIC LICENSE.
.TH SIPP 3X "December , 1990" 3X
.SH NAME
sipp - simple polygon processor, a 3d-graphics library
.SH SYNOPSIS
\fI#include <sipp.h>\fR
.sp
[g]cc [\fIflags\fR] \fIfiles\fR -lsipp -lm [ \fIlibraries\fR ]
.SH DESCRIPTION
\fIsipp\fR is a library for creating 3-dimensional scenes and rendering them
using a scan-line z-buffer algorithm. A scene is built up of objects
which can be transformed with rotation, translation and scaling. The
objects form hierarchies where each object can have arbitrarily many
subobjects and subsurfaces. A surface is a number of connected
polygons which are rendered with Phong interpolation of the surface
normals.
.sp
The library has an internal database for the objects that is to be
rendered. Objects can be installed in, and removed from, this database
at any time.
.sp
The library also provides 3-dimensional texture mapping with automatic
interpolation of texture coordinates. Simple anti-aliasing is performed
through double oversampling. A scene can be illuminated by an
arbitrary number of light sources. A basic shading algorithm is
provided with the library, but the user can also use his own shading
algorithms for each surface to produce special effects. Images are
produced in the Portable Pixmap format (ppm) for which many utilities
exist.

.SH USAGE
Before any other function in \fIsipp\fR is used, the function
\fIsipp_init()\fR must always be called. This call initiates the
internal database and other global variables. The rest of this section
describes the ideas behind \fIsipp\fR and outlines its usage. See the
section 
.B FUNCTIONS
for details.
.sp
All objects in \fIsipp\fR are built up of polygons with 3 or more
vertices. The polygons can either be concave or convex, but all
vertices must lie in the same plane. No test is done to assertain
this, however. A number of polygons can form a, possibly non-planar,
surface that is rendered using Phong interpolation of the surface
normals. Surfaces are grouped together into objects which can also
have other objects as subobjects. Each object or subobject can be
transformed using the standard transformations: rotation, scaling and
translation. All transformations are done with respect to the parent
object if the object is a subobject and the world if the object is a
top level object.
.sp
When an object is defined, copies can be made of it and it can be
installed into the internal database for rendering. The installation
doesn't affect the object in any way, i.e. it can still be transformed
and copied. This is useful if more pictures than one is to be created,
e.g. in an animation.
.sp
\fIsipp\fR uses a number of stacks to create a scene. First,
\fIvertices\fR of a polygon are pushed onto a \fIvertex stack\fR. When
all vertices in a polygon are pushed, a \fIpolygon\fR is created from
them and is pushed onto a \fIpolygon stack\fR. When a number of
polygons have been pushed, a call to a function in \fIsipp\fR combine
them into a \fIsurface\fR and returns a pointer to the surface. This
surface can then be inserted into an \fIobject\fR together with other
surfaces.  Objects can be transformed, used as subobjects in other
objects and inserted into the scene database.
.sp
To create a polygon, two functions are used: \fIvertex_push()\fR and
\fIpolygon_push()\fR. Variants of these functions exist, but these are
the simple forms that are the easiest to use. A call to
\fIvertex_push()\fR is done for each vertex in the polygons, going
counter clockwise around the edge. The order in which the vertices are
pushed is important, because this determines the front and the back
face of the polygon. When all polygons are pushed onto the vertex
stack, \fIpolygon_push()\fR is called. This call creates a polygon
structure out of the pushed vertices, deletes them from the vertex
stack, and pushes the newly created polygon onto the polygon stack.
.sp
When all polygons in a surface is on the polygon stack,
\fIsurface_create()\fR (or one of it's variants) will create a surface
out of them and empty the polygon stack for a new surface definition.
\fIsurface_create()\fR returns a pointer to the created surface.
.sp
An object is created with the function \fIobject_create()\fR. This
function returns a pointer to an empty object, i.e. it contains no
surfaces or subobjects. Surfaces and subobjects are added to an object
with the functions \fIobject_add_surface()\fR and \fIobject_add
subobj()\fR. When all surfaces and subobjects are added to the object
it can be installed in the rendering database, or removed from it,
with the functions \fIobject_install()\fR and \fIobject_uninstall()\fR
respectively.

.SH DATA TYPES
The \fIsipp\fR library defines a number of data types in the file
\fIsipp.h\fR. These data types are:
.IP \fIbool\fR
.br
\fIbool\fR can be used as a boolean type, with values \fITRUE\fR of
\fIFALSE\fR. These constants are also defined in \fIsipp.h\fR.

.IP \fIColor\fR
\fIColor\fR is a struct with three members, \fIred\fR, \fIgrn\fR and
\fIblu\fR.  Each member of the struct is a double that should be in
the range [0,1].

.IP \fIVector\fR
A \fIVector\fR is a struct with three members, \fIx\fR, \fIy\fR and
\fIz\fR which are all doubles.

.IP \fITransf_mat\fR
\fITransf_mat\fR is a standard 4 x 4 homogenous transformation matrix.
Actually it is stored as a 4 x 3 matrix to save memory, since the
rightmost column is only needed in the viewing calculation.
.sp
The members of a \fITransf_mat\fR should never be accessed directly, but
rather through the abstract functions described in the \fBFUNCTIONS\fR
section. 

.IP \fISurface\fR\ and\ \fIObject\fR
\fISurface\fR and \fIObject\fR are both opaque types used by \fIsipp\fR. The
actual definition of them is not important to the user.

.IP \fISurf_desc\fR
A \fISurf_desc\fR is a surface description, used by the built-in shader to
determine properties about a surface. The definition of \fISurf_desc\fR is as
follows: 
.br
\fItypedef\ struct {\fR
.br
\fI    double\  ambient;\fR\       /* Fraction of color visible in ambient light */
.br
\fI    double\  specular;\fR\      /* Fraction of colour specularly reflected */
.br
\fI    double\  c3;\fR\            /* "Shininess" 0 = shiny,  1 = dull */
.br
\fI    Color\   color;\fR\         /* Colour of the surface */
.br
\fI}\ Surf_desc;\fR

.IP \fILightsource\fR
All lightsources in the scene are kept in a linked list where the
nodes are defined like this:
.sp
.I typedef struct lightsource {
.br
\fI\    double intensity;\fR
.br
\fI\    Vector dir;\fR
.br
\fI\    struct lightsource *next;\fR
.br
.I } Lightsource;
.sp
A pointer to the head of this list is sent to the shader.

.SH SHADING FUNCTIONS
Each surface in a scene has a shading function associated with it.
This function is called for each pixel in the surface as it is
rendered. \fIsipp\fR has an internal basic shading function called
\fIbasic_shader\fR that can be used in most cases. \fIbasic_shader\fR
provides a somewhat modified and simplified version of Blinn's shading
model, taking a \fISurf_desc\fR as a description of the surface. 
.sp
If the user is not satisfied with the builtin shader, he can provide
his own shader and surface description struct. All shaders take
the same parameters and must be defined as follows:
.sp
\fIvoid\ myshader(nx,\ ny,\ nz,\ \ u,\ v,\ w,\ view_vec,\fR
.br
\fI		lights,\ surface,\ color)\fR
.br
\fI	double nx, ny, nz;\fR
.br
\fI	double u, v, w;\fR
.br
\fI	Vector\ view_vec;\fR
.br
\fI	Lightsource *lights;\fR
.br
\fI	void *surface;\fR
.br
\fI	Color\ *color;\fR
.sp
\fInx, ny\fR and \fInz\fR is the \fInon-normalized\fR surface normal at the
point that should be rendered.
.br
\fIu, v\fR and \fIw\fR are the interpolated texture coordinates at the rendered
point. If no texture coordinates have been given at some vertices these
values are undefined and contains garbage at best.
.br
\fIview_vec\fR is a normalized vector, pointing from the rendered
point at the viewpoint.
.br
\fIlights\fR is a pointer to a linked list of lightsource descriptions. See
the function \fIlightsource_push()\fR for a description of the structure of
the links.
.br
\fIsurface\fR is the same \fIsurface\fR-pointer that was sent to the
function \fIsurface_create()\fR. In the case of \fIbasic_shader\fR this is
a pointer to a \fISurf_desc\fR. If the user provides his own shader, he
can also provide his own surface description.
.br
Upon return, the shader should place the calculated rgb colour
components in the areas pointed to by \fIcolor\fR. The rgb components
must be values between 0 and 1.

.SH FUNCTIONS

.IP \fIvoid\ sipp_init()\fR
Initializes the whole library and emptys the internal database. This
function must be called before any other function in the library.

.IP \fIvoid\ vertex_push(x,\ y,\ z)\fR
.br
\fIdouble\ x,\ y,\ z;\fR
.sp
Push a vertex onto the internal vertex stack.
.br
Note: Vertices must be pushed on the vertex stack
\fIcounterclockwize\fR when looking at the "front" face of the
polygon. Otherwize the front of the surface will be defined in the
wrong direction.

.IP \fIvoid\ vertex_tx_push(x,\ y,\ z,\ u,\ v,\ w)\fR
.br
\fIdouble\ x,\ y,\ z;\fR
.br
\fIdouble\ u,\ v,\ w;\fR
.sp
Push a vertex and it's texture coordinates onto the vertex stack. Three texture
coordinates are provided to make it possible to use both 2d and solid texture.
The texture coordinates are interpolated between vertices on the same object
and are \fInot\fR affected by transformations of the object. The interpolated
coordinates will be sent to the shader. The coordinates can of course be used
to hold any attribute that one whishes to interpolate between vertices and
then use in the shader.

.IP \fIvoid\ polygon_push()\fR
Create a polygon from the vertices on the vertex stack and push it
onto the polygon stack. The vertex stack is empty and ready for a new
polygon definition after this operation.
.br
If a vertex in the polygon is already defined in a previous polygon
that belongs to the same surface, the same vertex will be referenced,
i.e. vertices shared between polygons are only stored once, but they
must be repeated when defining the polygons.

.IP \fISurface\ *surface_create(surf_desc,\ shader)\fR
.br
\fIvoid\ *surf_desc;\fR
.br
\fIShader\ *shader;\fR
.sp
Create a surface from the polygons on the polygon stack. A pointer
to the newly-created surface is returned. The polygon stack is empty
afterwards. \fIshader\fR is a pointer to the shader function that
will be called when a point on this surface is to be rendered. See the
section \fBSHADER FUNCTIONS\fR for a declaration of the shader function.
\fIsurf_desc\fR is a pointer to a static structure that contains the
surface properties of the surface. The exact representation of this
structure can be chosen freely by the user depending on the
implementation of his shader. If the internal shader,
\fIbasic_shader\fR is used, this struct is of type \fISurf_desc\fR.

.IP \fISurface\ *surface_basic_create(ambient,\ red,\ grn,\ blu,\ specular,\ c3)\fR
.br
\fIdouble\ ambient;\fR
.br
\fIdouble\ red,\ grn,\ blu;\fR
.br
\fIdouble\ specular;\fR
.br
\fIdouble\ c3;\fR
.sp
Create a surface from the polygons on the polygon stack.  A pointer to
the newly-created surface is returned.  The surface will be shaded
with the internal shader, \fIbasic_shader\fR, using the parameters as
values in a \fISurf_desc\fR struct.

.IP \fIvoid\ surface_set_shader(surface,\ surf_desc,\ shader)\fR
.br
\fISurface\ *surface;\fR
.br
\fIvoid\    *surf_desc;\fR
.br
\fIShader\  *shader;\fR
.sp
Set the surface \fIsurface\fR to be shaded with the shading function
\fIshader\fR. The shading information used by the shader is pointed at
by \fIsurf_desc\fR.

.IP \fIvoid\ surface_basic_shader(surface,\ ambient,\ red,\ grn,\ blu,\ specular,\ c3)\fR
.br
\fISurface\ *surface;\fR
.br
\fIdouble\  ambient;\fR
.br
\fIdouble\  red,\ grn,\ blu;\fR
.br
\fIdouble\  specular;\fR
.br
\fIdouble\  c3;\fR
.sp
Set \fIsurface\fR to be shaded by the internal shader and let
\fIambient\fR, \fIred\fR, \fIgrn\fR, \fIblu\fR, \fIspecular\fR and
\fIc3\fR be the values stored in the \fISurf_desc\fR struct for this
surface. 

.IP \fIObject\ *object_create()\fR
Create an empty object, i.e. an object with no surfaces or subobjects
in it. The transformation matrix in the new object will be a identity
matrix initially.

.IP \fIObject\ *object_instance(obj)\fR
.br
\fIObject\ *obj;\fR
.sp
Create a new instance of a previously defined object. The lists of
surfaces and subobjects in \fIobj\fR are not copied, but a new
reference with its own transformation matrix is created. The matrix is
set to the identity matrix. If \fIobj\fR is changed, i.e. if one of its
subobjects or surfaces are transformed, one is deleted or added, the
change will also be seen in the copy.

.IP \fIObject\ *object_dup(obj)\fR
.br
\fIObject\ *obj;\fR
.sp
Copy recursively an object and its subobjects. The
surfaces in the object tree are not copied, only new references to them
are made.

.IP \fIObject\ *object_deep_dup(obj)\fR
.br
\fIObject\ *obj;\fR
.sp
Copy the entire tree for the object \fIobj\fR, including subobjects
and all surfaces, polygons and vertices. This is a costly operation if
the object is complex.

.IP \fIvoid\ object_delete(obj)\fR
.br
\fIObject\ *obj;\fR
.sp
Delete the object \fIobj\fR, i.e. the memory used by \fIobj\fR and
all its subobjects and surfaces is recursively freed. \fIsipp\fR keeps
track of internal references so objects and surfaces referenced
from somewhere else in \fIsipp\fR will not be deleted, i.e. no dangeling
references are created in the data structures.

.IP \fIvoid\ object_install(obj)\fR
.br
\fIObject\ *obj;\fR
.sp
Install the object \fIobj\fR into the rendering database. This
function must be called on all objects that are to be visible in the
rendered image.

.IP \fIvoid\ object_uninstall(obj)\fR
.br
\fIObject\ *obj;\fR
.sp
Remove the object \fIobj\fR from the rendering database. If \fIobj\fR
is not in the database to begin with, nothing happens.

.IP \fIvoid\ object_add_surface(obj,\ surf)\fR
.br
\fIObject\  *obj;\fR
.br
\fISurface\ *surf;\fR
.sp
Add the surface \fIsurf\fR to the object \fIobj\fR.

.IP \fIvoid\ object_add_subobj(obj,\ subobj)\fR
.br
\fIObject\ *obj;\fR
.br
\fIObject\ *subobj;\fR
.sp
Add the subobject \fIsubobj\fR to the object \fIobj\fR.

.IP \fIvoid\ object_set_transf(obj,\ matrix)\fR
.br
\fIObject\     *obj;\fR
.br
\fITransf_mat\ *matrix;\fR
.sp
Set the transformation matrix of the object \fIobj\fR to \fImatrix\fR.
.br

.IP \fIvoid\ object_clear_transf(obj)\fR
.br
\fIObject\     *obj;\fR
.sp
Set the transformation matrix of the object \fIobj\fR to the unit matrix.
.br

.IP \fIvoid\ object_transform(obj,\ matrix)\fR
.br
\fIObject\     *obj;\fR
.br
\fITransf_mat\ *matrix;\fR
.sp
Post multiply the matrix \fImatrix\fR into the transformation matrix
of the object \fIobj\fR.

.br

.IP \fIvoid\ object_rot_x(obj,\ ang)\fR
.br
\fIObject\  *obj;\fR
.br
\fIdouble\  ang;\fR
.sp
Rotate the object \fIobj\fR the angle \fIang\fR about the X axis.
\fIang\fR is expressed in radians.

.IP \fIvoid\ object_rot_y(obj,\ ang)\fR
.br
\fIObject\  *obj;\fR
.br
\fIdouble\  ang;\fR
.sp
Rotate the object \fIobj\fR the angle \fIang\fR about the Y axis.
\fIang\fR is expressed in radians.

.IP \fIvoid\ object_rot_z(obj,\ ang)\fR
.br
\fIObject\  *obj;\fR
.br
\fIdouble\  ang;\fR
.sp
Rotate the object \fIobj\fR the angle \fIang\fR about the Z axis.
\fIang\fR is expressed in radians.

.IP \fIvoid\ object_rot(obj,\ point,\ vec,\ ang)\fR
.br
\fIObject\ *obj;\fR
.br
\fIVector\ *point;\fR
.br
\fIVector\ *vec;\fR
.br
\fIdouble\ ang;\fR
.sp
Rotate the object \fIobj\fR the angle \fIang\fR about the line
given by the point \fIpoint\fR and the vector \fIvec\fR starting in
that point. \fIang\fR is expressed in radians.
.br

.IP \fIvoid\ object_scale(obj,\ xscale,\ yscale,\ zscale)\fR
.br
\fIObject\ *obj;\fR
.br
\fIdouble\ xscale,\ yscale,\ zscale;\fR
.sp
Scale the object \fIobj\fR with the scaling factors \fIxscale,\
yscale\fR and \fIzscale\fR in the main directions respectively.

.IP \fIvoid\ object_move(obj,\ dx,\ dy,\ dz)\fR
.br
\fIObject\ *obj;\fR
.br
\fIdouble\ dx,\ dy,\ dz;\fR
.sp
Move (translate) the object \fIobj\ dx,\ dy\fR and \fIdz\fR in the
three main directions, respectively.

.IP \fIvoid\ lightsource_push(x,\ y,\ z,\ intensity)\fR
.br
\fIdouble\ x,\ y,\ z;\fR
.br
\fIdouble\ intensity;\fR
.sp
Create a new lightsource in the scene. All lightsources is considered
to be at an infinit distance and to emit white light. \fIx,\ y,\ z\fR
defines a vector pointing to the lightsource. \fIintensity\fR is a
double between 0 and 1 that defines the intensity of the light coming
from the lightsource.

.IP \fIvoid\ view_from(x,\ y,\ z)\fR
.br
.I double x, y, z;
.sp
Define the position of the viewpoint.

.IP \fIvoid\ view_at(x,\ y,\ z)\fR
.br
.I double x, y, z;
.sp
Define the viewing direction as going from the viewpoint to the point
.I x, y, z.

.IP \fIvoid\ view_up(x,\ y,\ z)\fR
.br
.I double x, y, z;
.sp
Define the up vector. The only constraint on this vector is that it
must not be parallel to the vector going from the viewpoint to the
viewed point.  If this function is not called, the default up vector
is the world Y axis.

.IP \fIvoid\ view_focal(ratio)\fR
.br
.I double ratio;
.sp
Define the focal ratio of the "camera". This is the ratio between the
distance from the viewpoint to the screen and half the screen height.
.sp 5
					screen
.br
					|
.br
					| d
.br
.lc \ 
	viewpoint|
.br
.lc -
	*|
.br
.lc \ 
			s|
.br
					|
.br
					|
.sp 2
.nf
        focal_ratio = d / s
.fi
.sp
Another way of describing the value is acrtan(v/2) where v is the opening
angle of the view.
.br
Note: The same focal ratio is used both horizontally and vertically.
If the picture is rendered with different resolution in x and y, the ratio
is assumed to be related to the \fIsmallest\fR of the two.

.IP \fIviewpoint(x,\ y,\ z,\ to_x,\ to_y,\ to_z,\ up_x,\ up_y,\ up_z,\ focal_ratio)\fR
.br
This function is used to set all viewpoint related values in one call. All
arguments are doubles.
.br
.I x, y, z
is the position of the viewpoint as in \fIview_from()\fR.
.br
.I to_x, to_y, to_z
defines the viewing direction as in \fIview_at()\fR.
.br
.I up_x, up_y, up_z
defines the up vector as in \fIview_up()\fR.
.br
.I focal_ratio
defines (guess what) the focal ratio as in \fIview_focal()\fR.

.IP \fIvoid\ render_image(xsize,\ ysize,\ outfile)\fR
.br
\fIint\    xsize, ysize;\fR
.br
\fIFILE\ *outfile;\fR
.sp
This function does the actual rendering of the scene. The image is
created with size (\fIxsize\fR X \fIysize\fR). \fIoutfile\fR is an
open file pointer to which the image will be written.  It can just as
well be a pipe of course.

.IP \fIvoid\ basic_shader(nx,\ ny,\ nz,\ \ u,\ v,\ w,\ view_vec,fR
.br
\fI             	lights,\ surface,\ color)\fR
.br
\fIdouble nx, ny, nz;\fR
.br
\fIdouble u, v, w;\fR
.br
\fIVector\ view_vec;\fR
.br
\fILightsource *lights;\fR
.br
\fISurf_desc *surface;\fR
.br
\fI
Color\ *color;\fR
.sp
This is the basic shader function that is provided with the library.
See the section about \fBSHADING FUNCTIONS\fR for more details.

.SH SEE ALSO
shaders(3X) - a number of shaders for \fIsipp\fR.
.br
primitives(3X) - a number of geometric primitives for \fIsipp\fR.

.SH AUTHORS
Jonas Yngvesson\	\	(jonas-y@isy.liu.se)
.br
Inge Wallin\		(ingwa@isy.liu.se)
.SH BUGS
Antialiasing is rather crude. Aliasing effects is still quite obvious. It
also ought to be possible to turn off antialiasing in order to get a faster
"preview".
.sp
The viewing vector which is sent to the shader is
actually only an approximation which assumes the viewpoint being at infinite
distance. 
.sp
No information is sent to the shader about how big interpolation steps
are in the texture coordinate system. This makes it impossible to use
any filtering techniqe for texture mapping, e.g. summed area tables.
.sp
There is no way to delete a subobject from an object.
