/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/




/*
 *                  File: MOTIF_ide_isptrak_panel.c
 *                Author: Theodossiou, Andreas
 *                  Date: 05/11/1993
 *           Description: MOTIF routines for Isotrak panel creation
 *  Modification History:
 *
 *         who?               when?           why?
 *    -----------------------------------------------------------
 *
 */

/******************************************************************/
/*                The standard C include files                    */
/******************************************************************/
# include <stdio.h>
# include <string.h>


/******************************************************************/
/*                The MOTIF include files                         */
/******************************************************************/
# include <X11/Intrinsic.h>
# include <X11/Shell.h>
# include <X11/Xlib.h>
# include <X11/Xutil.h>
# include <Xm/Xm.h>
# include <Xm/RowColumn.h>
# include <Xm/Frame.h>
# include <Xm/Form.h>
# include <Xm/ToggleBG.h>
# include <Xm/ToggleB.h>
# include <Xm/Text.h>
# include <Xm/PushB.h>
# include <Xm/PushBG.h>
# include <Xm/LabelG.h>
# include <Xm/CascadeB.h>
# include <Xm/BulletinB.h>
# include <Xm/FileSB.h>
# include <Xm/SelectioB.h>
# include <Xm/DialogS.h>
# include <Xm/MessageB.h>
#include <Xm/Scale.h>

/******************************************************************/
/*                The VolVis include files                        */
/******************************************************************/
#include "C_ide_new.h"
#include "MOTIF_windefs.h"
#include "C_isotrak.h"

/* Selection Map push button; needed for accounting on Selection Maps */
Widget Isotrak_Selection_Map; 

extern C_IDE_simulation_method C_ide_simulation_method;

void init_isotrak_operation_map();
void create_isotrak_ui();
void C_ide_create_control_panel_bottom_isotrak();
void C_ide_isotrak_create_toggles();



/*
 *        Procedure Name: ispotrak_quit
 *          Return Value: void
 *       Input Variables: Widget widget,client_data;
 *                        caddr_t call_data;
 *      Output Variables: none 
 *      Update Variables: none
 *      Global Variables: Widget Isotrak_Selection_Map
 *           Description: Closes the Isotrak Panel. If the Isotrak_Selection_Map is not
 *                        Sensitive then there is a Selection Map up. Decrement the number
 *                        of Selection Maps and reset the Sensitivity of the button
 */
void isotrak_quit(widget, client_data, call_data)
Widget widget,client_data;
caddr_t call_data;
{
   if (!XtIsSensitive(Isotrak_Selection_Map)) {
      decr_selection_maps();
      XtSetSensitive(Isotrak_Selection_Map);
   }
   XtUnmanageChild(client_data);
}


/*
 *        Procedure Name: C_ide_isotrak_allignment_toggle
 *          Return Value: void
 *       Input Variables: Widget w;
 *                        int client_data;
 *                        XmToggleButtonCallbackStruct *call_data;
 *      Output Variables: none
 *      Update Variables: none 
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method
 *                        char *isotrak_allignment_names
 *           Description: The client_data carries the appropriate integer - 0 for Set 1 for Reset.
 *                        The C_ide_simulation_mathod is updated accordingly. The printf 
 *                        statement shows the relation between the integer assigned and the strings
 *                        representing the toggles
 */
void C_ide_isotrak_allignment_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern char *isotrak_allignment_names[C_MAX_ALLIGNMENT];
  if (call_data->set) {
     C_ide_simulation_method.isotrak.allignment_simu = client_data;
     printf("Allignment is %s\n",
      isotrak_allignment_names[C_ide_simulation_method.isotrak.allignment_simu]);
  }
}


/*
 *        Procedure Name: C_ide_isotrak_boresight_toggle
 *          Return Value: void
 *       Input Variables: Widget w;
 *                        int client_data;
 *                        XmToggleButtonCallbackStruct *call_data;
 *      Output Variables: none
 *      Update Variables: none 
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method
 *                        char *isotrak_boresight_names
 *           Description: The client_data carries the appropriate integer - 0 for Set 1 for Reset.
 *                        The C_ide_simulation_mathod is updated accordingly. The printf 
 *                        statement shows the relation between the integer assigned and the strings
 *                        representing the toggles
 */
void C_ide_isotrak_boresight_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern char *isotrak_boresight_names[C_MAX_BORESIGHT];
  if (call_data->set) {
     C_ide_simulation_method.isotrak.boresight_simu = client_data;
     printf("Boresight is %s\n",
     isotrak_boresight_names[C_ide_simulation_method.isotrak.boresight_simu]);
  }
}   


/*
 *        Procedure Name: C_ide_isotrak_envelope_toggle
 *          Return Value: void
 *       Input Variables: Widget w;
 *                        int client_data;
 *                        XmToggleButtonCallbackStruct *call_data;
 *      Output Variables: none
 *      Update Variables: none 
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method
 *                        char *isotrak_envelope_names
 *           Description: The client_data carries the appropriate integer - 0 for Define.
 *                        The C_ide_simulation_mathod is updated accordingly. The printf 
 *                        statement shows the relation between the integer assigned and the strings
 *                        representing the toggles
 */
void C_ide_isotrak_envelope_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern char *isotrak_envelope_names[C_MAX_ENVELOPE];
  if (call_data->set) {
     C_ide_simulation_method.isotrak.envelope_simu = client_data;
     printf("Envelope is %s\n",
       isotrak_envelope_names[C_ide_simulation_method.isotrak.envelope_simu]);
  }
}   


/*
 *        Procedure Name: C_ide_isotrak_format_toggle
 *          Return Value: void
 *       Input Variables: Widget w;
 *                        int client_data;
 *                        XmToggleButtonCallbackStruct *call_data;
 *      Output Variables: none
 *      Update Variables: none 
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method
 *                        char *isotrak_format_names
 *           Description: The client_data carries the appropriate integer - 0 for ASCII, 1 for Binary.
 *                        The C_ide_simulation_mathod is updated accordingly. The printf 
 *                        statement shows the relation between the integer assigned and the strings
 *                        representing the toggles
 */
void C_ide_isotrak_format_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern char *isotrak_format_names[C_MAX_FORMAT];
  if (call_data->set) {
     C_ide_simulation_method.isotrak.format_simu = client_data;
     printf("Format is %s\n",
       isotrak_format_names[C_ide_simulation_method.isotrak.format_simu]);
  }
}   


/*
 *        Procedure Name: C_ide_isotrak_hemisphere_toggle
 *          Return Value: void
 *       Input Variables: Widget w;
 *                        int client_data;
 *                        XmToggleButtonCallbackStruct *call_data;
 *      Output Variables: none
 *      Update Variables: none 
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method
 *                        char *isotrak_hemishpere_names
 *           Description: The client_data carries the appropriate integer - 0 for Sensor.
 *                        The C_ide_simulation_mathod is updated accordingly. The printf 
 *                        statement shows the relation between the integer assigned and the strings
 *                        representing the toggles
 */
void C_ide_isotrak_hemisphere_toggle(w, client_data, call_data)
Widget w;
int client_data;
XmToggleButtonCallbackStruct *call_data;
{
  extern char *isotrak_hemisphere_names[C_MAX_HEMISPHERE];
  if (call_data->set) {
     C_ide_simulation_method.isotrak.hemisphere_simu = client_data;
     printf("Hemisphere is %s\n",
      isotrak_hemisphere_names[C_ide_simulation_method.isotrak.hemisphere_simu]);
  }
}   


/*
 *        Procedure Name: C_ide_isotrak_increment_callback
 *          Return Value: void
 *       Input Variables: Widget w;
 *                        caddr_t client_data;
 *                        XmScaleCallbackStruct *call_data;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method
 *           Description: The slider (increment) callback. There is a translation performed.
 *                        The slider has -100 and +100 as its min and max values, 2 decimal points.
 *                        We map -65.34 to -100 and 65.34 to 100. The values in between are 
 *                        translated accordingly. C_ide_simulation_method is assigned the translated
 *                        value.
 */
void C_ide_isotrak_increment_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
XmScaleCallbackStruct *call_data;
{
  C_ide_simulation_method.isotrak.increment_simu = call_data->value*0.6534;
  printf("Increment is set to %f\n",
      C_ide_simulation_method.isotrak.increment_simu);
}   


/*
 *        Procedure Name: create_isotrak_ui
 *          Return Value: void
 *       Input Variables: Widget top_isotrak_panel
 *                        C_Window_Defs window_defs (extern)
 *                        char *isotrak_label_names
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: none
 *           Description: It creates the panel for the Isotrak by appropriate calls to 
 *                        other functions
 */
void create_isotrak_ui(top_isotrak_panel)
Widget top_isotrak_panel;
{
   extern  char *isotrak_label_names[C_ISOTRAK_MAX_LABELS];
   extern  C_WindowDefs     window_defs;
   /***    Local Variables     ***/
   Widget isotrak_token_bulletin;
   Widget flabel[C_ISOTRAK_MAX_LABELS];
   Widget slider;
   Arg args[20];
   XmString string;
   int n,i,ypos;

   n = 0;

   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 10 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[0] = XmCreateLabelGadget(top_isotrak_panel, "Information", args, n);
   XtManageChild(flabel[0]);

   n = 0;
   XtSetArg( args[n], XmNx, 120 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   string = XmStringCreate (" ", XmSTRING_DEFAULT_CHARSET);
   XtSetArg (args[n], XmNlabelString, string); n++;
   isotrak_token_bulletin=XmCreateLabel (top_isotrak_panel, " ", args, n);
   XtManageChild(isotrak_token_bulletin);
   XmStringFree (string);

   /* create the Isotrak labels */

   ypos = 50;
   for (i=0; i<C_ISOTRAK_MAX_LABELS; i++) {
        n = 0;
        C_Set_Color_Label( args, n, window_defs );
        XtSetArg( args[n], XmNx, 10 ); n++;
        XtSetArg( args[n], XmNy, ypos ); n++;
        flabel[i] = XmCreateLabelGadget(top_isotrak_panel, 
                                        isotrak_label_names[i],
                                        args, n);
        XtManageChild(flabel[i]); 
        ypos += 60;
   }

   C_ide_isotrak_create_toggles(top_isotrak_panel); /* create the toggles */

   /* slider for the increment label */

   n = 0;
   C_Set_Color_Scale( args, n, window_defs );
   XtSetArg( args[n], XmNx, 120 );                    n++;
   XtSetArg( args[n], XmNy, 330 );                    n++;
   XtSetArg( args[n], XmNwidth, 170 );                n++;
   XtSetArg( args[n], XmNmaximum, 100);               n++;
   XtSetArg( args[n], XmNminimum, -100);              n++;
   XtSetArg( args[n], XmNvalue, -100);                n++;
   XtSetArg( args[n], XmNdecimalPoints,2);            n++; /* 100 is shown as 1.00 */
   XtSetArg( args[n], XmNscaleMultiple,1);            n++; 
   XtSetArg( args[n], XmNshowValue, TRUE );           n++;
   XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
   slider = XtCreateManagedWidget( "incr", xmScaleWidgetClass, 
                                    top_isotrak_panel,
                                    args, n );
   XtAddCallback( slider, XmNvalueChangedCallback,
                  C_ide_isotrak_increment_callback, 
                  NULL);



   n = 0;
   XtSetArg( args[n], XmNx, 10 ); n++;
   XtSetArg( args[n], XmNy, 420); n++;
   C_ide_create_control_panel_bottom_isotrak(top_isotrak_panel, n, args); /* create bottom panel */

}
    
 
/*
 *        Procedure Name: C_ide_isotrak_create_toggles
 *          Return Value: void
 *       Input Variables: Widget parent
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: C_Window_Defs window_defs
 *           Description: It creates all the toggles in the control panel
 */
void C_ide_isotrak_create_toggles(parent)
Widget parent;
{

    extern C_WindowDefs     window_defs;

    Widget allignment_frame;
    Widget allignment_radiobox;
    Widget allignment_toggles[C_MAX_ALLIGNMENT];

    Widget boresight_frame;
    Widget boresight_radiobox;
    Widget boresight_toggles[C_MAX_BORESIGHT];

    Widget envelope_frame;
    Widget envelope_radiobox;
    Widget envelope_toggles[C_MAX_ENVELOPE];

    Widget format_frame;
    Widget format_radiobox;
    Widget format_toggles[C_MAX_FORMAT];

    Widget hemisphere_frame;
    Widget hemisphere_radiobox;
    Widget hemisphere_toggles[C_MAX_HEMISPHERE];

    Arg args[20];
    XmString string;

    int n,i;

    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 50);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    allignment_frame = XmCreateFrame (parent, "allignment_frame", args, n);
    XtManageChild (allignment_frame);

    n = 0;
    C_Set_Color_RadioBox( args, n, window_defs );
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    allignment_radiobox = XmCreateRadioBox (allignment_frame, "allignment_radiobox", 
                                            args, n);
    XtManageChild (allignment_radiobox);

    /* allignment toggles */

    for (i=0; i<C_MAX_ALLIGNMENT; i++) {
        n = 0;
        string = XmStringCreate (isotrak_allignment_names[i],
                                 XmSTRING_DEFAULT_CHARSET);
        C_Set_Color_ToggleButton( args, n, window_defs );
        XtSetArg (args[n], XmNlabelString, string); n++;
        allignment_toggles[i] = XtCreateWidget 
                                  (isotrak_allignment_names[i],
                                   xmToggleButtonWidgetClass,
                                   allignment_radiobox, args, n);
        XtAddCallback(allignment_toggles[i], XmNvalueChangedCallback,
                                   C_ide_isotrak_allignment_toggle, i);
        XmStringFree (string);
    }
    XtManageChildren (allignment_toggles, C_MAX_ALLIGNMENT);

    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 110);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    boresight_frame = XmCreateFrame (parent, "boresight_frame", args, n);
    XtManageChild (boresight_frame);

    n = 0;
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    C_Set_Color_RadioBox( args, n, window_defs );
    boresight_radiobox = XmCreateRadioBox (boresight_frame, "boresight_radiobox", 
                                            args, n);
    XtManageChild (boresight_radiobox);

    /* boresight toggles */

    for (i=0; i<C_MAX_BORESIGHT; i++) {
        n = 0;
        string = XmStringCreate (isotrak_boresight_names[i],
                                 XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        boresight_toggles[i] = XtCreateWidget 
                                  (isotrak_boresight_names[i],
                                   xmToggleButtonWidgetClass,
                                   boresight_radiobox, args, n);
        XtAddCallback(boresight_toggles[i], XmNvalueChangedCallback,
                                   C_ide_isotrak_boresight_toggle, i);
        XmStringFree (string);
    }
    XtManageChildren (boresight_toggles, C_MAX_BORESIGHT);

    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 170);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    envelope_frame = XmCreateFrame (parent, "envelope_frame", args, n);
    XtManageChild (envelope_frame);

    n = 0;
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    C_Set_Color_RadioBox( args, n, window_defs );
    envelope_radiobox = XmCreateRadioBox (envelope_frame, "envelope_radiobox", 
                                            args, n);
    XtManageChild (envelope_radiobox);

    /* envelope toggles */

    for (i=0; i<C_MAX_ENVELOPE; i++) {
        n = 0;
        string = XmStringCreate (isotrak_envelope_names[i],
                                 XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        envelope_toggles[i] = XtCreateWidget 
                                  (isotrak_envelope_names[i],
                                   xmToggleButtonWidgetClass,
                                   envelope_radiobox, args, n);
        XtAddCallback(envelope_toggles[i], XmNvalueChangedCallback,
                                   C_ide_isotrak_envelope_toggle, i);
        XmStringFree (string);
    }
    XtManageChildren (envelope_toggles, C_MAX_ENVELOPE);


    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 230);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    format_frame = XmCreateFrame (parent, "format_frame", args, n);
    XtManageChild (format_frame);

    n = 0;
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    C_Set_Color_RadioBox( args, n, window_defs );
    format_radiobox = XmCreateRadioBox (format_frame, "format_radiobox", 
                                            args, n);
    XtManageChild (format_radiobox);

    /* format toggles */

    for (i=0; i<C_MAX_FORMAT; i++) {
        n = 0;
        string = XmStringCreate (isotrak_format_names[i],
                                 XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        format_toggles[i] = XtCreateWidget 
                                  (isotrak_format_names[i],
                                   xmToggleButtonWidgetClass,
                                   format_radiobox, args, n);
        XtAddCallback(format_toggles[i], XmNvalueChangedCallback,
                                   C_ide_isotrak_format_toggle, i);
        XmStringFree (string);
    }
    XtManageChildren (format_toggles, C_MAX_FORMAT);

    n = 0;
    XtSetArg (args[n], XmNx, 120);  n++;
    XtSetArg (args[n], XmNy, 290);  n++;
    C_Set_Color_Frame( args, n, window_defs );
    hemisphere_frame = XmCreateFrame (parent, "hemisphere_frame", args, n);
    XtManageChild (hemisphere_frame);

    n = 0;
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
    C_Set_Color_RadioBox( args, n, window_defs );
    hemisphere_radiobox = XmCreateRadioBox (hemisphere_frame,
                                            "hemisphere_radiobox", 
                                            args, n);
    XtManageChild (hemisphere_radiobox);

    /* hemisphere toggles */

    for (i=0; i<C_MAX_HEMISPHERE; i++) {
        n = 0;
        string = XmStringCreate (isotrak_hemisphere_names[i],
                                 XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, string); n++;
        C_Set_Color_ToggleButton( args, n, window_defs );
        hemisphere_toggles[i] = XtCreateWidget 
                                  (isotrak_hemisphere_names[i],
                                   xmToggleButtonWidgetClass,
                                   hemisphere_radiobox, args, n);
        XtAddCallback(hemisphere_toggles[i], XmNvalueChangedCallback,
                                   C_ide_isotrak_hemisphere_toggle, i);
        XmStringFree (string);
    }
    XtManageChildren (hemisphere_toggles, C_MAX_HEMISPHERE);
}
    

/*
 *        Procedure Name: C_ide_create_control_panel_bottom_isotrak
 *          Return Value: void
 *       Input Variables: Widget top_isotrak_panel;
 *                        int n; 
 *                        Arg args[20];
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: Widget Isotrak_Selection_Map
 *           Description: It creates the bottom cotrol panel for the isotrak. Positioning inforamtion
 *                        is in the arguments.
 */
void C_ide_create_control_panel_bottom_isotrak(top_isotrak_panel, n, args)
Widget top_isotrak_panel;
int n;
Arg args[20];
{
    extern C_WindowDefs     window_defs;
    extern void create_selection_read_only_ui();

    Widget frame;
    Widget rc;
    Widget button[3];
    XmString string;


    C_Set_Color_Frame( args, n, window_defs );
    frame = XmCreateFrame( top_isotrak_panel, "frame", args, n );

    n = 0;
    C_Set_Color_RowColumn( args, n, window_defs );
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL); n++;
    rc = XmCreateRowColumn(frame, "rc", args, n);
    XtManageChild (rc);

    n=0;
    C_Set_Color_Button( args, n, window_defs );
    Isotrak_Selection_Map = XmCreatePushButton( rc, "Selection Map", args, n );
    XtAddCallback(Isotrak_Selection_Map, XmNactivateCallback, 
                  create_selection_read_only_ui, NULL);
    XtManageChild(Isotrak_Selection_Map);

    n=0;
    C_Set_Color_Button( args, n, window_defs );
    button[0] = XmCreatePushButton( rc, " Simulation Panel", args, n );

    n=0;
    C_Set_Color_Button( args, n, window_defs );
    button[1] = XmCreatePushButton( rc, "Help", args, n );

    n=0;
    C_Set_Color_Button( args, n, window_defs );
    button[2] = XmCreatePushButton( rc, "Close", args, n);
    XtAddCallback(button[2], XmNactivateCallback, isotrak_quit, top_isotrak_panel);

    XtManageChildren(button,3);
    XtManageChild(frame);
}
