/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/



/*
 *                  File: MOTIF_ide_selection_panels.c
 *                Author: Theodossiou, Andreas
 *                  Date: 05/11/1993
 *           Description: MOTIF routines for Selection Map creation
 *  Modification History:
 *
 *         who?               when?           why?
 *    -----------------------------------------------------------
 *
 */

/******************************************************************/
/*                The standard C include files                    */
/******************************************************************/
# include <stdio.h>
# include <string.h>


/******************************************************************/
/*               The MOTIF include files                          */
/******************************************************************/
# include <X11/Intrinsic.h>
# include <X11/Shell.h>
# include <X11/Xlib.h>
# include <X11/Xutil.h>
# include <Xm/Xm.h>
# include <Xm/RowColumn.h>
# include <Xm/Frame.h>
# include <Xm/Form.h>
# include <Xm/ToggleBG.h>
# include <Xm/ToggleB.h>
# include <Xm/Text.h>
# include <Xm/TextF.h>
# include <Xm/PushB.h>
# include <Xm/PushBG.h>
# include <Xm/LabelG.h>
# include <Xm/CascadeB.h>
# include <Xm/BulletinB.h>
# include <Xm/FileSB.h>
# include <Xm/SelectioB.h>
# include <Xm/DialogS.h>
# include <Xm/MessageB.h>


/******************************************************************/
/*                  VolVis include files                          */
/******************************************************************/
#include "C_ide.h"
#include "C_ide_new.h"
#include "MOTIF_windefs.h"

#define FALSE 0
#define TRUE 1

void C_ide_create_bottom_selection();
void C_create_text_widgets();

void create_selection_modifiable_ui();

void create_selection_read_only_ui();

void set_callback();
void quit_selection();

extern Open_Selection_Maps selection_maps; /* data structure to save the text widgets */


/*
 *        Procedure Name: quit_read_only_selection
 *          Return Value: void
 *       Input Variables: Widget w;
 *                         Widget client_data;
 *                         caddr_t call_data;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: none
 *           Description: When a read only Selection map (one that was opened from
 *                        a mouse,isotrak,dataglove panel) closes only the total number of 
 *                        selection maps is decremented
 */
void quit_read_only_selection(w, client_data, call_data)
Widget w;
Widget client_data;
caddr_t call_data;
{
   extern void decr_selection_maps();

   decr_selection_maps();
   XtUnmanageChild(client_data);
   XtSetSensitive(XtParent(XtParent(client_data)),True);
}


/*
 *        Procedure Name: quit_modifiable_selection
 *          Return Value: void
 *       Input Variables: Widget w;
 *                         Widget client_data;
 *                         caddr_t call_data;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: none
 *           Description: When a modifiable Selection map (one that was opened from
 *                        a Virtual Device panel) closes both  the total number of 
 *                        selection maps and the number of selection maps opened from a
 *                        virtual device are decremented
 */
void quit_modifiable_selection(w, client_data, call_data)
Widget w;
Widget client_data;
caddr_t call_data;
{
   extern void decr_selection_maps();
   extern void decr_modifiable_selection_maps();

   decr_selection_maps();
   decr_modifiable_selection_maps();
   XtUnmanageChild(client_data);
   XtSetSensitive(XtParent(XtParent(client_data)),True);
}


/*
 *        Procedure Name: set_callback
 *          Return Value: void
 *       Input Variables: Widget widget,parent;
 *                        caddr_t call_data;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: int Number_Of_Open_Selection_Maps (extern)
 *           Description: When the set button is pressed on a modifiable
 *                        selection map this function is called. It loops 
 *                        from 0 to Number_Of_Open_Selection_Maps and from
 *                        0 to C_MAX_SELECTION_NUMBER succesfully updating all
 *                        the selection maps. In the future, the inner loops
 *                        upper limit should vary based on the current oparation
 *                        settings of the input devices
 */
void set_callback(widget,parent,call_data)
Widget widget,parent;
caddr_t call_data;
{
  extern int Number_Of_Open_Selection_Maps;
  int i,j;
  for (i=0; i<=Number_Of_Open_Selection_Maps; i++)
      for (j=0; j<C_MAX_SELECTION_NUMBER; j++)
          XmTextSetString(selection_maps[i][j],"try");
}


/*
 *        Procedure Name: create_selection_modifiable_ui
 *          Return Value: void 
 *       Input Variables: Widget widget;
 *                        caddr_t client_data,call_data;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: C_WindowDefs window_defs (extern)
 *           Description: It creates a Selection map that can be modified 
 *                        by the user
 */
void create_selection_modifiable_ui(widget,client_data,call_data)
Widget widget;
caddr_t client_data,call_data;
{


   extern void incr_selection_maps();

   extern  C_WindowDefs     window_defs;
   /***    Local Variables     ***/
   Widget flabel[3];
   Widget top_selection_panel;
   Arg args[20];
   XmString string;
   int n,i,ypos;

   incr_selection_maps(); /* increment total number */
   incr_modifiable_selection_maps(); /* increment modifiable number */
   n = 0;
   string = XmStringCreate( "Selection Map Panel", XmSTRING_DEFAULT_CHARSET);
   XtSetArg( args[n], XmNdialogTitle, string ); n++;
   C_Set_Color_Dialog( args, n, window_defs );
   top_selection_panel =  XmCreateBulletinBoardDialog(
                          widget, "Selection Map panel", args, n);

   n = 0;
   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 10 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[0] = XmCreateLabelGadget(top_selection_panel, "Operation", args, n);
   XtManageChild(flabel[0]);

   n = 0;
   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 140 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[1] = XmCreateLabelGadget(top_selection_panel, "   Name  ",args, n);
   XtManageChild(flabel[1]);

   n = 0;
   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 270 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[2] = XmCreateLabelGadget(top_selection_panel, "Selection Number", 
                                  args, n);
   XtManageChild(flabel[2]);

   C_create_text_widgets(top_selection_panel,FALSE); /* create the text widgets */

   n = 0;
   XtSetArg( args[n], XmNx, 10 ); n++;
   XtSetArg( args[n], XmNy, 500); n++;
   C_ide_create_bottom_selection(top_selection_panel, n, args,TRUE); 

   XtManageChild(top_selection_panel);

   XtSetSensitive(widget,False);

}


/*
 *        Procedure Name: C_create_text_widgets
 *          Return Value: void 
 *       Input Variables: Widget parent;
 *                        int cannot_be_modified;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: C_IDE_simulation_method C_ide_simulation_method (extern)
 *                        C_WindowDefs window_defs (extern)
 *                        int Number_Of_Open_Selection_Maps (extern)
 *           Description: It creates the text widgets within a selection map. If 
 *                        cannot_be_modified is TRUE then the all the widgets are read
 *                        only; if not then the last column can be modified. (NOTE : The
 *                        number of rows should be variable in the future, based on the 
 *                        current operation maps of the input device being displayed)
 */
void C_create_text_widgets(parent,cannot_be_modified)
Widget parent;
int cannot_be_modified;
{


    extern C_IDE_simulation_method C_ide_simulation_method;
    extern C_WindowDefs     window_defs;
    extern int Number_Of_Open_Selection_Maps;
  
    Arg args[20];

    Widget bb;
 
    Widget frame;

    int n,i,j,index,x,y,width;

    n = 0;
    XtSetArg( args[n], XmNx, 10); n++;
    XtSetArg( args[n], XmNy, 30); n++;
    C_Set_Color_Frame( args, n, window_defs );
    frame = XmCreateFrame(parent, "frame", args, n );
    XtManageChild( frame );


    n =0;
    C_Set_Color_RowColumn( args, n, window_defs );
    bb = XmCreateBulletinBoard(frame, "bb", args, n);
    XtManageChild (bb);

    /* first two columns are always read_only */

    x = 10;
    y = 10;
    width = 40;
    for (i=0; i<2; i++) {
        for (j=0; j<C_MAX_SELECTION_NUMBER; j++) {      
            n = 0;
            C_Set_Color_Text(args,n,window_defs);
            XtSetArg(args[n], XmNeditable, False); n++;
            XtSetArg(args[n],XmNwidth,width); n++;
            XtSetArg(args[n],XmNx,x); n++;
            XtSetArg(args[n],XmNy,y); n++;
            index = j+(i*C_MAX_SELECTION_NUMBER);
            selection_maps[Number_Of_Open_Selection_Maps][index] = 
                                          XmCreateTextField(bb,"dummy",args,n);
            XtManageChild(selection_maps[Number_Of_Open_Selection_Maps][index]);
            y += 40;
        }
        x = 100;
        y = 10;
        width = 120;
     }

    i = 0;
    if (cannot_be_modified) { /* last column read_only also */
       XtSetArg(args[i],XmNeditable,False); i++;
    }

    n = i;
    y = 10;
    for (j=0; j<C_MAX_SELECTION_NUMBER; j++) {      
         n = i;
         C_Set_Color_Text(args,n,window_defs);
         XtSetArg(args[n],XmNwidth,120); n++;
         XtSetArg(args[n],XmNx,260); n++;
         XtSetArg(args[n],XmNy,y); n++;
         index = j+20;
         selection_maps[Number_Of_Open_Selection_Maps][index] = 
                                       XmCreateTextField(bb,"dummy",args,n);
         XtManageChild(selection_maps[Number_Of_Open_Selection_Maps][index]);
         y += 40;
    }

}


/*
 *        Procedure Name: C_ide_create_bottom_selection
 *          Return Value: void
 *       Input Variables: Widget top_selection_panel;
 *                        int n,can_be_modified;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: C_WindowDefs window_defs (extern)
 *           Description: Creates the bottom of the control panel. Based on whether
 *                        the selection map is modifiable or not we have a set button
 *                        Positioning information is pased in the arguments
 */
void C_ide_create_bottom_selection(top_selection_panel, n, args,can_be_modified)
Widget top_selection_panel;
int n,can_be_modified;
Arg args[20];
{
    extern C_WindowDefs     window_defs;

    Widget frame;
    Widget rc;
    Widget button[3];
    XmString string;


    C_Set_Color_Frame( args, n, window_defs );
    frame = XmCreateFrame( top_selection_panel, "frame", args, n );

    n = 0;
    C_Set_Color_RowColumn( args, n, window_defs );
    XtSetArg( args[n], XmNorientation, XmHORIZONTAL); n++;
    rc = XmCreateRowColumn(frame, "rc", args, n);
    XtManageChild (rc);

    n=0;
    C_Set_Color_Button( args, n, window_defs );
    button[2] = XmCreatePushButton( rc, "Close", args, n);

    if (can_be_modified) {
       n=0;
       C_Set_Color_Button( args, n, window_defs );
       button[0] = XmCreatePushButton( rc, "Set", args, n );
       XtAddCallback(button[0], XmNactivateCallback, set_callback,
                     top_selection_panel);
       XtManageChild(button[0]);
       XtAddCallback(button[2], XmNactivateCallback, quit_modifiable_selection,
                     top_selection_panel);
    }
    else  XtAddCallback(button[2], XmNactivateCallback, quit_read_only_selection,
                        top_selection_panel);

    n=0;
    C_Set_Color_Button( args, n, window_defs );
    button[1] = XmCreatePushButton( rc, "Help", args, n );
    XtManageChild(button[1]);

    XtManageChild(button[2]);

    XtManageChild(frame);
}


/*
 *        Procedure Name: create_selection_read_only_ui
 *          Return Value: void
 *       Input Variables: Widget widget;
 *                        caddr_t client_data,call_data;
 *      Output Variables: none
 *      Update Variables: none
 *      Global Variables: C_WindowDefs window_defs (extern)
 *           Description: It creates a read only selection map. The total number
 *                        of selection maps is incremented
 */
void create_selection_read_only_ui(widget,client_data,call_data)
Widget widget;
caddr_t client_data,call_data;
{


   extern void incr_selection_maps();

   extern  C_WindowDefs     window_defs;
   /***    Local Variables     ***/
   Widget flabel[3];
   Widget top_selection_panel;
   Arg args[20];
   XmString string;
   int n,i,ypos;



   incr_selection_maps(); /* increment total number */
   n = 0;
   string = XmStringCreate( "Selection Map Panel", XmSTRING_DEFAULT_CHARSET);
   XtSetArg( args[n], XmNdialogTitle, string ); n++;
   C_Set_Color_Dialog( args, n, window_defs );
   top_selection_panel =  XmCreateBulletinBoardDialog(
                          widget, "Selection Map panel", args, n);

   n = 0;
   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 10 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[0] = XmCreateLabelGadget(top_selection_panel, "  Number  ",args, n);
   XtManageChild(flabel[0]);

   n = 0;
   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 140 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[1] = XmCreateLabelGadget(top_selection_panel, "  Device  ",args, n);
   XtManageChild(flabel[1]);

   n = 0;
   C_Set_Color_Label( args, n, window_defs );
   XtSetArg( args[n], XmNx, 270 ); n++;
   XtSetArg( args[n], XmNy, 10 ); n++;
   flabel[2] = XmCreateLabelGadget(top_selection_panel, "   Name   ",args, n);
   XtManageChild(flabel[2]);

   C_create_text_widgets(top_selection_panel,TRUE);


   n = 0;
   XtSetArg( args[n], XmNx, 10 ); n++;
   XtSetArg( args[n], XmNy, 500); n++;
   C_ide_create_bottom_selection(top_selection_panel, n, args,FALSE); 

   XtManageChild(top_selection_panel);
   XtSetSensitive(widget,False);
}
    
