/****************************************************************************/
/*                                                                          */
/*  VolVis is a volume visualization system for investigating, manipulating */
/*  and rendering geometric and volumetric data.                            */
/*                                                                          */
/*  Copyright (C) 1993 by the Research Foundation of the State University   */
/*                            of New York                                   */
/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 1, or (at your option)     */
/*  any later version.                                                      */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this program; if not, write to the Free Software             */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                          */
/*  For information on VolVis, contact us at:                               */
/*                                                                          */
/*                volvis@cs.sunysb.edu                         (email)      */
/*                                                                          */
/*                Lisa Sobierajski & Ricardo Avila             (US Mail)    */
/*                Department of Computer Science                            */
/*                State University of New York at Stony Brook               */
/*                Stony Brook, New York  11794-4400                         */
/*                                                                          */
/****************************************************************************/



/*
 *                      File: MOTIF_oco_modify_volume_shade.c
 *                    Author: Clifford Bohm
 *                      Date: 03/29/93
 *               Description: Motif Windowing Routines to alter obect shading.
 *      Modification History:
 *
 *              Who?            When?           Why?
 *      --------------------------------------------------------------------
 *
 */


#include <stdio.h>
#include <math.h>
#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <X11/StringDefs.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/BulletinB.h>
#include <Xm/LabelG.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/DrawingA.h>
#include <Xm/Frame.h>
#include <Xm/Scale.h>

#include "C_volvis.h"
#include "MOTIF_windefs.h"
#include "C_help.h"
#include "C_oco.h"


/* Global Variables For Motif Windowing */
extern Widget                   vv_toplevel;    /* Bulletin Board Dialog */
extern Widget                   oco_bbdialog;   /* Bulletin Board Dialog */
extern C_World                  world;
extern C_ObjectControlInfo      oco_info;
extern C_WindowDefs		window_defs;

extern void close_callback();
extern C_View                   view;
extern int C_check_for_name();          /* used to check if a name has already
                                           been used.  It is defined in
                                           C_volume_control.c */

Widget	sim_shd_frame;			/* frames which are used to hold the
					   widgets to modify either simple
					   shade, or array shade. */ 


/*
 *              Procedure Name: oco_set_simple_shade 
 *              Return Value:   none
 *              Input Variables:
 *			w                - the widget that called this
 *                                         function.
 *                      volume_index     - an int which is the index into
 *                                         world.volumes[volume_index]
 *			call_data        - generic call data given by the
 *                                         system on a call back.
 *
 *              Description:
 *			This procedure first changes the shade type of the
 *			object, and then manages the frame that contains the
 *			widgets to modify the simple shade properties of the
 *			object.
 */

void oco_set_simple_shade( w, volume_index, call_data)
Widget  w;
int     volume_index;
XtPointer *call_data;
{

	world.volume[volume_index]->shade_type = C_SIMPLE_SHADE;
        XtManageChild(sim_shd_frame);   
}

/*
 *              Procedure Name: oco_set_array_shade
 *              Return Value:   none
 *              Input Variables:
 *                      w                - the widget that called this
 *                                         function.
 *                      volume_index     - an int which is the index into
 *                                         world.volumes[volume_index]
 *			call_data        - generic call data given by the
 *                                         system on a call back.
 *              Description:
 *                      This procedure first changes the shade type of the
 *                      object, and then manages the frame that contains the
 *                      widgets to modify the array shade properties of the
 *                      object.
 */
void oco_set_no_shade( w, volume_index, call_data)
Widget  w;
int     volume_index;
XtPointer *call_data;
{
        XtUnmanageChild(sim_shd_frame);   
}


/*
 *              Procedure Name: oco_create_volume_shade_change 
 *              Return Value:   none
 *              Input Variables:
 *                      volume_index     - an int which is the index into
 *                                        world.lights[light_index]
 *
 *              Description:
 *                      In this procedure a pop up is created which contains
 *                      the name of the object, and a option menu (which allows
 *                      the user to select the shading model they wnat to use).
 *                      When a selection is made on the menu, the window is 
 *                      changed so that the new paramaters can be altered.
 *                 The fields that can be altered are:
 *			when in simple shade mode:
 *				volume_shade_diffuse_coef[C_MAX_VOLUMES];
 *				volume_shade_specular_coef[C_MAX_VOLUMES];
 *				volume_shade_specular_power[C_MAX_VOLUMES];
 *				volume_shade_transmit_coef[C_MAX_VOLUMES];
 *				volume_shade_transmissitivity[C_MAX_VOLUMES];
 *				volume_shade_index_of_refraction;
 */	
				
void oco_create_volume_shade_change(volume_index)
int volume_index;
{
	
	void close_callback();
	void help_callback();
        void oco_accept_volume_shade();
        void oco_cancel_volume_shade();

        Arg             args[20];
        int             n;


        Widget  frame;
        Widget  label;

	C_Color *volume_color;	
        Widget  rowcolumn,
		temp_button,
		button,
		pulldown,
		bboard,
		menu;

	XmString        label_string;
	char    string[16];

        n=0;
        XtSetArg( args[n],XmNx, 1000);n++;
	XtSetArg( args[n],XmNy, 500);n++; 
	XtSetArg( args[n], XmNautoUnmanage, False ); n++;
        C_Set_Color_Dialog( args, n, window_defs );
        oco_info.volume_info[volume_index]->shade.volume_shade_change =
			XmCreateBulletinBoardDialog(vv_toplevel,
                        "modify_volume_shade", args, n );
	n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 0 ); n++;
        C_Set_Color_Frame( args, n, window_defs );

        label =
                XmCreateLabelGadget(oco_info.volume_info[volume_index]->
			shade.volume_shade_change,
                                "Volume name:",
                                args, n );

        XtManageChild(label);

	n = 0;
        XtSetArg( args[n], XmNx, 300 ); n++;
        XtSetArg( args[n], XmNy, 0 ); n++;
        C_Set_Color_Frame( args, n, window_defs );
        label =
                XmCreateLabelGadget(oco_info.volume_info[volume_index]->
			shade.volume_shade_change,
                               	world.volume[volume_index]->volume_name,
                                args, n );

        XtManageChild(label);
	n = 0;
	XtSetArg( args[n], XmNy, 40 ); n++;
	C_Set_Color_RowColumn( args, n, window_defs );
        bboard =
                XmCreateBulletinBoard(oco_info.volume_info[volume_index]->
			shade.volume_shade_change
					, "bboard", args, n);
        XtManageChild (bboard);
	n = 0;
        C_Set_Color_PullDown( args, n, window_defs );
	pulldown =
                XmCreatePulldownMenu( bboard,
					 "pulldown", args, n);
	n = 0;
        label_string = XmStringCreate ("Simple Shade",XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        button =
                XmCreatePushButtonGadget(pulldown, "simple_shade", args, n);
        XtAddCallback (button, XmNactivateCallback,
                       oco_set_simple_shade, volume_index);
	XtManageChild (button);
	
	if (world.volume[volume_index]->shade_type == C_SIMPLE_SHADE)
	{
		/*** if this is the present shade type of the object then
		     we have to save the button widget, so that we can 
		     set the Option Menu so that it appears as the present
		     mode. ***/
		temp_button = button;
	}
	temp_button = button;

	n = 0;
        label_string = XmStringCreate ("No Shade",XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        button =
                XmCreatePushButtonGadget(pulldown, "no_shade", args, n);
        XtAddCallback (button, XmNactivateCallback,
                       oco_set_no_shade, volume_index);	
	XtManageChild (button);

	if (world.volume[volume_index]->shade_type == C_NO_SHADE)
        {
		/*** if this is the present shade type of the object then
                     we have to save the button widget, so that we can 
                     set the Option Menu that it appears as the present
                     mode. ***/
                temp_button = button;
        }
	n = 0;
        label_string = XmStringCreate ("Shading Model:",
                                      XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNlabelString, label_string); n++;
        XtSetArg (args[n], XmNsubMenuId, pulldown); n++;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 0 ); n++;
        C_Set_Color_OptionMenu( args, n, window_defs );


                XtSetArg (args[n], XmNmenuHistory,
                        temp_button); n++;

        menu = XmCreateOptionMenu(bboard,
                                "oco_option_menu", args, n);

        XtManageChild(menu);


        n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        XtSetArg( args[n], XmNy, 510 ); n++;
        C_Set_Color_RowColumn( args, n, window_defs );
        rowcolumn =
                XmCreateRowColumn( oco_info.volume_info[volume_index]->
			shade.volume_shade_change, "rowcolumn", args, n);
        XtManageChild (rowcolumn);
        

	n = 0;
        C_Set_Color_Button( args, n, window_defs );
        button = XtCreateManagedWidget("Accept",
                        xmPushButtonWidgetClass, rowcolumn, args, n );

        XtAddCallback( button, XmNactivateCallback,
                       oco_accept_volume_shade, volume_index);


        n = 0;
        C_Set_Color_Button( args, n, window_defs );
        button = XtCreateManagedWidget("Cancel",
                        xmPushButtonWidgetClass, rowcolumn, args, n );

        XtAddCallback( button, XmNactivateCallback,
                       oco_cancel_volume_shade,volume_index );

        n=0;
        C_Set_Color_Button( args, n, window_defs );
        button = XtCreateManagedWidget( "Close",
                xmPushButtonWidgetClass, rowcolumn, args, n );
	XtAddCallback( button, XmNactivateCallback,
                  close_callback, oco_info.volume_info[volume_index]->
			shade.volume_shade_change );

        n = 0;
        C_Set_Color_Button( args, n, window_defs );
        button = XtCreateManagedWidget("Help",
                        xmPushButtonWidgetClass, rowcolumn, args, n );
	XtAddCallback( button, XmNactivateCallback,
                  help_callback, C_OCO_MODIFY_VOLUME_SHADE_HELP ); 

	n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 90 ); n++;
        C_Set_Color_Frame( args, n, window_defs );
	sim_shd_frame = XmCreateFrame(
		oco_info.volume_info[volume_index]->
		shade.volume_shade_change, "simple shade frame", args, n );

	bboard =
                XmCreateBulletinBoard(sim_shd_frame, "sim_shd_board", args, n);
	XtManageChild(bboard);	
	/****** Shading is made up of 6 parts, the note before each is
	either the value of the slider, or the tyoe of widget used
	for inputing the value.
	
	0-1     volume_shade_diffuse_coef[C_MAX_VOLUMES];
	0-1     volume_shade_specular_coef[C_MAX_VOLUMES];
	text    volume_shade_specular_power[C_MAX_VOLUMES];
	0-1     volume_shade_transmit_coef[C_MAX_VOLUMES];
	0-1     volume_shade_transmissitivity[C_MAX_VOLUMES];
	text 	volume_shade_index_of_refraction;
	
	******/
	n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 15 ); n++;
        C_Set_Color_Frame( args, n, window_defs );

        label =
                XmCreateLabelGadget(bboard,
                                "Diffuse Coeffient:",
                                args, n );

        XtManageChild(label);

	
	n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        XtSetArg( args[n], XmNminimum, 0 ); n++;
        XtSetArg( args[n], XmNmaximum, 100 ); n++;
        XtSetArg( args[n], XmNdecimalPoints, 2 ); n++;
        XtSetArg( args[n], XmNshowValue, TRUE ); n++;
        XtSetArg( args[n], XmNvalue,
                        (int)( 100 * world.volume[volume_index]->local_shade.
			simple_shade->diffuse_coef ) ); n++;
        XtSetArg( args[n], XmNscaleMultiple, 1 ); n++;
        XtSetArg( args[n], XmNscaleWidth, 200 ); n++;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 0 ); n++;
        C_Set_Color_Scale( args, n, window_defs );
        oco_info.volume_info[volume_index]->
		shade.volume_sim_shade_diffuse_coef =
			XtCreateManagedWidget("diffuse_coef ",
                       	xmScaleWidgetClass,
			bboard, args, n );

        n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 65 ); n++;
        C_Set_Color_Frame( args, n, window_defs );

        label =
                XmCreateLabelGadget(bboard,
                                "Specular Coeffient:",
                                args, n );
        XtManageChild(label);


        n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        XtSetArg( args[n], XmNminimum, 0 ); n++;
        XtSetArg( args[n], XmNmaximum, 100 ); n++;
        XtSetArg( args[n], XmNdecimalPoints, 2 ); n++;
        XtSetArg( args[n], XmNshowValue, TRUE ); n++;
        XtSetArg( args[n], XmNvalue,
                        (int)( 100 * world.volume[volume_index]->local_shade.
                        simple_shade->specular_coef ) ); n++;
        XtSetArg( args[n], XmNscaleMultiple, 1 ); n++; 
       	XtSetArg( args[n], XmNscaleWidth, 200 ); n++;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 50 ); n++;
        C_Set_Color_Scale( args, n, window_defs );
        oco_info.volume_info[volume_index]->
		shade.volume_sim_shade_specular_coef =
                        XtCreateManagedWidget("specular_coef",
                        xmScaleWidgetClass,
                        bboard, args, n );


        n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 115 ); n++;
        C_Set_Color_Frame( args, n, window_defs );

	label=
                XmCreateLabelGadget(bboard,
                                "Specular Power:",
                                args, n );
        XtManageChild(label);

        n=0;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 115 ); n++;
        XtSetArg( args[n], XmNcolumns, 15 ); n++;
        sprintf( string, "%f",
                 world.volume[volume_index]->local_shade.
                        simple_shade->specular_power );
        XtSetArg( args[n], XmNvalue, string); n++;
        C_Set_Color_Text( args, n, window_defs );
        oco_info.volume_info[volume_index]->
	    shade.volume_sim_shade_specular_power =
                 XmCreateTextField(bboard,
                 "specular_power", args, n );
        XtManageChild( oco_info.volume_info[volume_index]->
		shade.volume_sim_shade_specular_power );



        n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 165 ); n++;
        C_Set_Color_Frame( args, n, window_defs );


        label =
                XmCreateLabelGadget(bboard,
                                "Transmit Coeffient:",
                                args, n );
        XtManageChild(label);


        n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        XtSetArg( args[n], XmNminimum, 0 ); n++;
        XtSetArg( args[n], XmNmaximum, 100 ); n++;
        XtSetArg( args[n], XmNdecimalPoints, 2 ); n++;
        XtSetArg( args[n], XmNshowValue, TRUE ); n++;
        XtSetArg( args[n], XmNvalue,
                        (int)( 100 * world.volume[volume_index]->local_shade.
                        simple_shade->transmit_coef ) ); n++;
        XtSetArg( args[n], XmNscaleMultiple, 1 ); n++; 
        XtSetArg( args[n], XmNscaleWidth, 200 ); n++;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 150 ); n++;
        C_Set_Color_Scale( args, n, window_defs );
        oco_info.volume_info[volume_index]->
	    shade.volume_sim_shade_transmit_coef =
                        XtCreateManagedWidget("transmit_coef",
                        xmScaleWidgetClass,
                        bboard, args, n );


        n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 215 ); n++;
        C_Set_Color_Frame( args, n, window_defs );


        label =
                XmCreateLabelGadget(bboard,
                                "Transmissitivity:",
                                args, n );
        XtManageChild(label);


        n = 0;
        XtSetArg( args[n], XmNorientation, XmHORIZONTAL ); n++;
        XtSetArg( args[n], XmNminimum, 0 ); n++;
        XtSetArg( args[n], XmNmaximum, 100 ); n++;
        XtSetArg( args[n], XmNdecimalPoints, 2 ); n++;
        XtSetArg( args[n], XmNshowValue, TRUE ); n++;
        XtSetArg( args[n], XmNvalue,
                        (int)( 100 * world.volume[volume_index]->local_shade.
                        simple_shade->transmissitivity ) ); n++;
        XtSetArg( args[n], XmNscaleMultiple, 1 ); n++; 
        XtSetArg( args[n], XmNscaleWidth, 200 ); n++;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 200 ); n++;
        C_Set_Color_Scale( args, n, window_defs );
        oco_info.volume_info[volume_index]->
	    shade.volume_sim_shade_transmissitivity =
                        XtCreateManagedWidget("transmissitivity",
                        xmScaleWidgetClass,
                        bboard, args, n );


        n = 0;
        XtSetArg( args[n], XmNx, 0 ); n++;
        XtSetArg( args[n], XmNy, 265 ); n++;
        C_Set_Color_Frame( args, n, window_defs );


	label =
                XmCreateLabelGadget(bboard,
                                "Index of Refraction",
                                args, n );
        XtManageChild(label);

	n=0;
        XtSetArg( args[n], XmNx, 150 ); n++;
        XtSetArg( args[n], XmNy, 265 ); n++;
        XtSetArg( args[n], XmNcolumns, 15 ); n++;
        sprintf( string, "%f",
                 world.volume[volume_index]->local_shade.
                        simple_shade->index_of_refraction );
        XtSetArg( args[n], XmNvalue, string); n++;
        C_Set_Color_Text( args, n, window_defs );
        oco_info.volume_info[volume_index]->
	    shade.volume_sim_shade_index_of_refraction =
                 XmCreateTextField(bboard,
                 "index_of_refraction", args, n );
        XtManageChild(oco_info.volume_info[volume_index]->
		shade.volume_sim_shade_index_of_refraction);


	if (world.volume[volume_index]->shade_type == C_SIMPLE_SHADE)
		/*** if the shade type of the object = simple shade when the
		     window is being created, then manage the simple shade
		     widgets ***/
                XtManageChild(sim_shd_frame);   


}


/*
 *              Procedure Name: oco_accept_volume_shade
 *              Return Value:   none
 *              Input Variables:
 *			w		 - the wiget the callback is made from
 *                      volume_index     - an int which is the index into
 *                                         world.volume[volume_index]
 *			call_data	 - This is needed because this is a
 *					   call back it is not used at all.
 *
 *              Description:
 *			The code herein updates the shading model data for the
 *		        object when the accept button is pressed.  If there is 
 *			an error in the input, the source of the error will be
 *			reported to the message window, and no update will be
 *			made.
 *		The feilds that are updated are:
 *                      when in simple shade mode:
 *                              volume_shade_diffuse_coef[C_MAX_VOLUMES];
 *                              volume_shade_specular_coef[C_MAX_VOLUMES];
 *                              volume_shade_specular_power[C_MAX_VOLUMES];
 *                              volume_shade_transmit_coef[C_MAX_VOLUMES];
 *                              volume_shade_transmissitivity[C_MAX_VOLUMES];
 *                              volume_shade_index_of_refraction;
 *
 *
 *			NOTE: the only values that are altered are the values 
 *			of the present shadeing model. ie, only the values seen 
 *			on the screen at the time that the accept is hit are
 *			updated.
 */

void oco_accept_volume_shade (w,volume_index, call_data )
Widget                  w;
int                     volume_index;
XtPointer               call_data;
{

	int      	tmp_val,error = 0;
	char            *string,*ptr;

	OcoVolumeInfo	*vinfo;

	vinfo = oco_info.volume_info[volume_index];


   if (world.volume[volume_index]->shade_type == C_SIMPLE_SHADE)
   {

/****** Check for error in input *********************************************/
/*** check specular power ***/
        string = (char *)XmTextGetString(
                        vinfo->shade.volume_sim_shade_specular_power);
        if((string != NULL) && (*string != NULL))
        {
                ptr = string;
                strtod(string,&ptr);
                if ((ptr != NULL) && (*ptr != NULL))
                {
                        C_error_message(
                           "The input for specular power is invalid.\n");
                        error = 1;
                }
        }
        else
        {
                C_error_message("There is no value for specular power.\n");
                error = 1;
        }

/*** check index of refraction ***/
        string = (char *)XmTextGetString(
                      vinfo->shade.volume_sim_shade_index_of_refraction
				);
	if((string != NULL) && (*string != NULL))
        {
                ptr = string;
                strtod(string,&ptr);
                if ((ptr != NULL) && (*ptr != NULL))
                {
                        C_error_message(
                           "The input for index of refraction is invalid.\n");
                        error = 1;
                }
        }
        else
        {
                C_error_message("There is no value for index of refraction.\n");
                error = 1;
        }

        if (!error)
        { /*** make changes because there are no errors in input ***/



	XmScaleGetValue( vinfo->shade.volume_sim_shade_diffuse_coef,
			&tmp_val);
	world.volume[volume_index]->local_shade.simple_shade->diffuse_coef =
							(float)(tmp_val)/100.0;

	

	XmScaleGetValue( vinfo->shade.volume_sim_shade_specular_coef,
                        &tmp_val);
        world.volume[volume_index]->local_shade.simple_shade->specular_coef =
                                                        (float)(tmp_val)/100.0;      


	string = (char *)XmTextGetString(
                        vinfo->shade.volume_sim_shade_specular_power);
        world.volume[volume_index]->local_shade.simple_shade->specular_power =
                        atof( string );

	
	
	XmScaleGetValue( vinfo->shade.volume_sim_shade_transmit_coef,
                        &tmp_val);      
        world.volume[volume_index]->local_shade.simple_shade->transmit_coef =
                                                        (float)(tmp_val)/100.0; 



        XmScaleGetValue( vinfo->shade.
			volume_sim_shade_transmissitivity,
                        &tmp_val);      
        world.volume[volume_index]->local_shade.simple_shade->
					transmissitivity =
                                                        (float)(tmp_val)/100.0; 

	string = (char *)XmTextGetString(
                        vinfo->shade.
			volume_sim_shade_index_of_refraction);
        world.volume[volume_index]->local_shade.simple_shade->
			index_of_refraction =
                        atof( string );

	}
   }
}

/*
 *              Procedure Name: oco_cancel_volume_shade
 *              Return Value:   none
 *              Input Variables:
 *                      w                - the wiget the callback is made from
 *                      volume_index     - an int which is the index into
 *                                         world.volume[volume_index]
 *                      call_data        - This is needed because this is a
 *                                         call back it is not used at all.
 *
 *              Description:
 *                      The code herein resets the shading model widgets for
 *                      object when the cancel button is pressed. 
 *                      This code does not have the ability to change any 
 *			values in the actual object, it reads the values from
 *			the objects data, and resets the sliders to those 
 *			values.
 *              The feilds that are reset are:
 *                      when in simple shade mode:
 *                              volume_shade_diffuse_coef[C_MAX_VOLUMES];
 *                              volume_shade_specular_coef[C_MAX_VOLUMES];
 *                              volume_shade_specular_power[C_MAX_VOLUMES];
 *                              volume_shade_transmit_coef[C_MAX_VOLUMES];
 *                              volume_shade_transmissitivity[C_MAX_VOLUMES];
 *                              volume_shade_index_of_refraction;
 *
 *
 *                      NOTE: the only values that are reset are the values
 *                      of the present shadeing model. ie, only the values seen
 *                      on the screen at the time that the cancel is hit are
 *                      updated.
 */

void oco_cancel_volume_shade(w,volume_index, call_data )
Widget                  w;
int                     volume_index;
XtPointer               call_data;
{
	        Arg             args[1];
        XColor          reset_color;
        char    string[16];
	OcoVolumeInfo	*vinfo;

	vinfo = oco_info.volume_info[volume_index];


	XmScaleSetValue( vinfo->shade.volume_sim_shade_diffuse_coef,
			(int)(100*world.volume[volume_index]->
                        local_shade.simple_shade->diffuse_coef) );

 	XmScaleSetValue( vinfo->shade.volume_sim_shade_specular_coef,
                        (int)(100*world.volume[volume_index]->
                        local_shade.simple_shade->specular_coef) );

	

	sprintf( string, "%f",world.volume[volume_index]->
			local_shade.simple_shade->specular_power);
	XtSetArg( args[0], XmNvalue, string);
	XtSetValues( vinfo->shade.volume_sim_shade_specular_power,
			args, 1 );

	XmScaleSetValue( vinfo->shade.volume_sim_shade_transmit_coef,
                        (int)(100*world.volume[volume_index]->
                        local_shade.simple_shade->transmit_coef) );

	XmScaleSetValue( vinfo->shade.volume_sim_shade_transmissitivity
						,
                        (int)(100*world.volume[volume_index]->
                        local_shade.simple_shade->transmissitivity) );
	sprintf( string, "%f",world.volume[volume_index]->
                        local_shade.simple_shade->index_of_refraction);
        XtSetArg( args[0], XmNvalue, string);
        XtSetValues( vinfo->shade.volume_sim_shade_index_of_refraction
			, args, 1 );


}

/*
 *              Procedure Name: oco_modify_volume_shade_callback
 *              Return Value:   none
 *              Input Variables:
 *                      w                - the wiget the callback is made from
 *                      volume_index     - an int which is the index into
 *                                         world.volume[volume_index]
 *                      call_data        - This is needed because this is a
 *                                         call back it is not used at all.
 *
 *              Description:
 *                      This code first makes sure that the object that the 
 *			user wishes to modify does not already have a modify
 *			shade window open.  If it does not, then it goes on
 *			to open the window.
 */

void oco_modify_volume_shade_callback(w, volume_index, call_data )
Widget                  w;
int                     volume_index;
XtPointer               call_data;
{
	OcoVolumeInfo	*vinfo;

	vinfo = oco_info.volume_info[volume_index];


	if (!vinfo->shade.volume_shade_change)

        {
		oco_create_volume_shade_change(volume_index);
	}
        else
        {
                if (XtIsManaged(
                    vinfo->shade.volume_shade_change ) )
                        XtUnmanageChild(
                                vinfo->shade.volume_shade_change );
	}
                XtManageChild( vinfo->shade.volume_shade_change );
}
	
