/*
 * misc.c	- miscellaneous utilities.
 * misc.c,v 2.3 1992/07/22 16:05:53 pete Exp
 * misc.c,v
 * Revision 2.3  1992/07/22  16:05:53  pete
 * Make it so XoConfWidget() can handle NULL widget.
 *
 * Revision 2.2  1992/07/16  17:21:34  pete
 * Added some documentation.
 *
 * Revision 2.1  1992/06/23  00:29:14  pete
 * Changed interface to _XoMenuNew and _XoMenuDone.
 *
 * Revision 2.0  1992/04/23  02:52:49  ware
 * First public release.
 *
 * Revision 1.13  1992/04/23  02:19:04  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.12  1992/03/03  17:19:43  ware
 * Added XoNameString() so there is a reentrant way of getting
 * a widget name.  Have XoConfWidget() use new function so
 * information is correct.
 *
 * Revision 1.11  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.10  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.9  1992/02/12  00:53:10  ware
 * Added _XoChildFind() to make it easier for a parent to find a
 * specific child in it's list of children.
 *
 * Revision 1.8  1992/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.7  1991/12/01  16:13:52  pete
 * Added _XoChildRoom() to make it easier for composite widgets
 * to add their own InsertChild() methods.
 *
 * Revision 1.6  1991/09/12  09:51:15  pete
 * Added XoConfWidget() to allow XtConfigureWidget() calls print info out.
 *
 * Revision 1.5  91/08/30  17:40:21  pete
 * Added XoEnablePopups.
 *
 * Revision 1.4  91/08/27  09:43:40  pete
 * Added XoStackChild() to make it easy to change the stacking mode
 * of children.
 *
 * Revision 1.3  91/08/26  17:18:18  pete
 * Fixed syntax error.
 *
 * Revision 1.2  91/08/26  13:51:08  pete
 * Added XoGeometryResultString().
 *
 * Revision 1.1  91/08/26  11:58:30  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Shell.h>
#include <X11/CoreP.h>
#include <X11/CompositeP.h>
#include <X11/Xo/XoP.h>
#include <X11/Xo/MenuBarB.h>		/* for XtNenablePopup */
#include <X11/Xo/ObjDrawP.h>
#include <X11/Xo/dbug.h>

/*
 * XoNameString -
 *	Return the name of a widget in the form Class.instance where
 *	Class is the class name of the widget and instance is the
 *	instance name of the widget.
 *
 *	Copies the value into "buffer" and also returns a pointer to
 *	the buffer.  This makes it conveniant for calls to printf
 */

char *
XoNameString (gw, buffer, len)
	Widget		gw;
	char		*buffer;
	int		len;		/* size of buffer */
{
	int             this_len;
	char           *instance;
	char           *class;

	if (!buffer || len <= 0)
		return ((char *) NULL);
	if (!gw)
	{
		instance = "None";
		class = "Bogus";
	}
	else
	{
		instance = XtName (gw);
		class = gw->core.widget_class->core_class.class_name;
	}
	--len;				/* make room for '\0' */
	buffer[len] = '\0';
	strncpy (buffer, class, len);
	if ((this_len = strlen (buffer) < len))
	{
		strncat (buffer, ".", len - this_len++);
		strncat (buffer, instance, len - this_len);
	}
	return (buffer);
}

/*
 * XoName -
 *	Return the name of a widget in the form Class.instance.  Uses
 *	XoNameString() but passes static buffer for conveniance.
 *	Since a static buffer is used only a single call at a time is
 *	supported.
 */

char *
XoName (gw)
	Widget          gw;
{
	static char	buffer[48];

	return XoNameString (gw, buffer, sizeof (buffer));
}

/*
 * geo_list -
 *	Name of the geometry results.
 */

static String   geo_list[] =
{
	"XtGeometryYes",
	"XtGeometryNo",
	"XtGeometryAlmost",
	"XtGeometryDone"
};

/*
 * XoGeometryResultString -
 *	Return the string that represents the value of the geometry result.
 *	Useful for debugging statements.
 */

char           *
XoGeometryResultString (geo)
	XtGeometryResult geo;
{
	return geo_list[geo];
}

/*
 * XoStackChild -
 *	Move the widgets window above or below its siblings.  Stack_mode is
 *	one of:
 *	Above	the widget is placed at the top of the stack
 *	Below	the widet is placed at the bottom of the stack
 *	TopIf	the widget if any sibling occludes the window; the window is
 *		placed at the top of the stack.
 *	BottomIf
 *		If the widget occludes any sibling, the window is placed
 *		at the bottom of the stack.
 *	Opposite
 *		If any sibling occludes the window, the window is placed at
 *		the top of the stack.  If the window occludes any sibling,
 *		the window is placed at the bottom of the stack.
 *	NOTE: The widget must be realized otherwise this is a no-op.
 *	NOTE: Above and Below are the only useful modes.
 *	NOTE: I'm unsure what happens with Opposite if both conditions are
 *	      satisified.
 */

void
XoStackChild (gw, stack_mode)
	Widget          gw;
	int             stack_mode;
{
	XWindowChanges  values;

	if (!gw || !XtIsRealized (gw))
		return;
	values.stack_mode = stack_mode;
	XConfigureWindow (XtDisplay (gw), XtWindow (gw),
			  CWStackMode, &values);
}

/*
 * XoMap -
 *	Map the widget if flag is True, Unmap the widget if it is False.
 *	Widget must be realized otherwise it is a no-op.
 */

void
XoMap (gw, flag)
	Widget          gw;
	int             flag;
{
	if (!gw || !XtIsRealized (gw))
		return;
	if (flag)
		XtMapWidget (gw);
	else
		XtUnmapWidget (gw);
}

/*
 * XoEnablePopups - Go through the list of children and set the
 *		enablePopup resources to the value of |enable|.
 */

void
XoEnablePopups (parent, enable)
	Widget          parent;
	int             enable;
{
	CompositeWidget w = (CompositeWidget) parent;
	Arg             args[1];
	Cardinal        cnt;
	int             i;

	if (!XtIsComposite (parent))
	{
		_XoWarn (parent, "Wrong Class", "XoEnablePopups",
			 "Must be a subclass of Composite");
		return;
	}
	cnt = 0;		/* Careful, args has room for only one
				 * argument! */
	XtSetArg (args[cnt], XtNenablePopup, enable);
	++cnt;
	for (i = 0; i < w->composite.num_children; i++)
	{
		XtSetValues (w->composite.children[i], args, cnt);
	}
}

/*
 * XoConfWidget -
 *	Identical to call to XtConfigureWidget() except it includes
 *	a debugging statement for the parameters.  The parameters are
 *	printed when "XtConfigureWidget" is enabled.  Handles
 *	NULL widget gracefully which XtConfigureWidget() does not.
 */

void
XoConfWidget (gw, x, y, width, height, border_width)
	Widget          gw;
	int             x;
	int             y;
	unsigned int	width;
	unsigned int	height;
	unsigned int	border_width;
{
	Widget		parent;
#ifndef DBUG_OFF
	char		buf1[80];
	char		buf2[80];
#endif

	DBUG_ENTER ("XoConfWidget");
	if (!gw)
		DBUG_VOID_RETURN;
	parent = XtParent (gw);
	DBUG_PRINT ("XtConfigureWidget", ("%s: Placing %s at (%d,%d)x[%d,%d] inside [%d,%d]",
					  XoNameString (parent,
							buf1, sizeof (buf1)),
					  XoNameString (gw, buf2, sizeof (buf2)),
					  x, y, width, height,
					  parent->core.width,
					  parent->core.height));
	/*
	 * Under R4, ignore the warnings about the types of these
	 * being "signed".
	 */
	XtConfigureWidget (gw, x, y, width, height, border_width);
	DBUG_VOID_RETURN;
}

/*
 * _XoChildRoom	- Increment the number of children and insure there
 *		is enough room allocated for the child to be added.
 */

void
_XoChildRoom (gw)
	Widget          gw;
{
	CompositeWidget w = (CompositeWidget) gw;

	++w->composite.num_children;
	if (w->composite.num_children > w->composite.num_slots)
	{
		w->composite.num_slots += 5;
		w->composite.children =
			(WidgetList) XtRealloc ((char *) w->composite.children,
						w->composite.num_slots *
						sizeof (Widget));
	}
}

/*
 * _XoChildFind -
 *	Find this child in the list of children and return it's index.
 *	Return -1 if it is not found.  0 is the first child.
 */
int
_XoChildFind (parent, child)
	Widget          parent;
	Widget          child;
{
	int             i;
	CompositeWidget w = (CompositeWidget) parent;

	if (!parent || !child)
		return (-1);
	/*
	 * Just a quick sanity check.  We could just as easily let the loop
	 * fall through.
	 */
	if (XtParent (child) != parent)
		return (-1);
	for (i = 0; i < w->composite.num_children; i++)
	{
		if (w->composite.children[i] == child)
			break;
	}
	return i;
}

/*
 * XoEventLocation -
 *	Return the location, if possible, contained in the event.
 *	If not interested in one of x, y pass in NULL.  If the event does
 *	not contain some of the fields, they are initialized to 0.
 */

void
XoEventLocation (event, x, y)
	XEvent         *event;
	Position       *x, *y;
{
	Position        ret_x, ret_y;

	ret_x = ret_y = 0;

	switch (event->type)
	{
	case KeyPress:
	case KeyRelease:
		ret_x = event->xkey.x;
		ret_y = event->xkey.y;
		break;
	case ButtonPress:
	case ButtonRelease:
		ret_x = event->xbutton.x;
		ret_y = event->xbutton.y;
		break;
	case MotionNotify:
		ret_x = event->xmotion.x;
		ret_y = event->xmotion.y;
		break;
	case EnterNotify:
	case LeaveNotify:
		ret_x = event->xcrossing.x;
		ret_y = event->xcrossing.y;
		break;
	case Expose:
		ret_x = event->xexpose.x;
		ret_y = event->xexpose.y;
		break;
	case GraphicsExpose:
		ret_x = event->xgraphicsexpose.x;
		ret_y = event->xgraphicsexpose.y;
		break;
	}

	if (x)
		*x = ret_x;
	if (y)
		*y = ret_y;
}

/*
 * XoEvenTime -
 *	Return the time this event occured at.  If the event type does
 *	not contain a time element, return CurrentTime
 */

Time
XoEventTime (event)
	XEvent         *event;
{
	Time            t;

	if (!event)
		return CurrentTime;
	t = CurrentTime;
	switch (event->type)
	{
	case ButtonPress:
	case ButtonRelease:
		t = ((XButtonEvent *) event)->time;
		break;
	case KeyPress:
	case KeyRelease:
		t = ((XKeyEvent *) event)->time;
		break;
	case MotionNotify:
		t = ((XMotionEvent *) event)->time;
		break;
	case EnterNotify:
	case LeaveNotify:
		t = ((XCrossingEvent *) event)->time;
		break;
	case FocusIn:
	case FocusOut:
	case KeymapNotify:
	case Expose:
	case GraphicsExpose:
	case NoExpose:
	case CirculateNotify:
	case ConfigureNotify:
	case CreateNotify:
	case DestroyNotify:
	case GravityNotify:
	case MapNotify:
	case MappingNotify:
	case ReparentNotify:
	case UnmapNotify:
	case VisibilityNotify:
	case CirculateRequest:
	case ConfigureRequest:
	case MapRequest:
	case ResizeRequest:
	case ColormapNotify:
	case ClientMessage:
		t = CurrentTime;/* no time element */
		break;
	case PropertyNotify:
		t = ((XPropertyEvent *) event)->time;
		break;
	case SelectionClear:
		t = ((XSelectionClearEvent *) event)->time;
		break;
	case SelectionRequest:
		t = ((XSelectionRequestEvent *) event)->time;
		break;
	case SelectionNotify:
		t = ((XSelectionEvent *) event)->time;
		break;
	}
	return t;
}


/*
 * _XoObjRedraw -
 *	Call the expose procedure of the given object.  Insure it is a valid
 *	object to call the expose proc for
 */

void
_XoObjRedraw (gw, event, region)
	Widget          gw;		/* should be subclassof ObjDraw */
	XEvent         *event;		/* the exposure event */
	Region          region;		/* the uncovered region */
{
	XoObjDrawWidget w = (XoObjDrawWidget) gw;
	XoObjDrawClassRec *class;

	if (!w)
		return;
	class = (XoObjDrawClassRec *) w->object.widget_class;
	if (!class)
		return;
	if (class->rect_class.expose)
	{
		(*class->rect_class.expose) (gw, event, region);
	}
}

/*
 * _XoObjPointIn -
 *	Indicate if x & y fall within the widget.
 */
Boolean
_XoObjPointIn (gw, x, y)
	Widget		gw;		/* object to check */
	int		x;		/* x & y coordinates */
	int		y;
{
	int	status = False;
	RectObj	w = (RectObj) gw;

	/*
	 * If this is actually a widget with its own window then just
	 * assume that however the point was given we are just being
	 * asked if it still within the window's borders.
	 */
	if (XtIsWidget (gw))
	{
		status = (x >= 0 && y >= 0
			  && x <= gw->core.width && y <= gw->core.height);
	}
	else if (XtIsRectObj (gw))
	{
		status = (x >= _XoObjDrawLeft (w)
			  && y >= _XoObjDrawTop (w)
			  && x <= _XoObjDrawRight (w)
			  && y <= _XoObjDrawBottom (w));
	}
	return (status);
}

/*
 * _XoRedisplaySelf -
 *	Causes a widget to redisplay itself by clearing the widget's
 *	window, without sending an expose event and then calling the
 *	\code{expose} method.  This is particular useful for actions
 *	that need to force a redraw.
 */

void
_XoRedisplaySelf (gw)
	Widget          gw;
{
	DBUG_ENTER ("_xoRedisplaSelf");
	if (XtIsRealized (gw))
	{
		XClearArea (XtDisplayOfObject (gw), XtWindowOfObject (gw), 0, 0, 0, 0, False);
		_XoObjRedraw (gw, (XEvent *) NULL, (Region) NULL);
	}
	DBUG_VOID_RETURN;
}

/*
 * XoVisualGet -
 *	Given a widget, return the Visual it uses.  It does this by
 *	finding the widgets's shell (with XoShellGet()) and getting
 *	that shells visual.  It returns NULL if unable to get a
 *	visual.
 *
 * Note: If the visual returned is CopyFromParent, is it handled
 *	 properly?
 */
Visual         *
XoVisualGet (gw)
	Widget          gw;
{
	Widget          shell;
	Visual         *v;
	Arg             args[10];
	Cardinal        cnt;

	if ((shell = XoShellGet (gw)) == NULL)
		return (Visual *) NULL;
	v = NULL;
	cnt = 0;
	XtSetArg (args[cnt], XtNvisual, &v);
	++cnt;
	XtGetValues (shell, args, cnt);
	if (v == CopyFromParent)
	{
		v = XDefaultVisualOfScreen (XtScreenOfObject (gw));
	}
	return v;
}

/*
 * XoWidgetIsColor -
 *	Return True if this widget is displayed on a color visual.  Color
 *	is defined as any visual with four or more colors (it uses
 *	XoVisualColor() to actually determine if this is a color visual).
 */

Boolean
XoWidgetIsColor (gw)
	Widget          gw;
{
	return XoVisualColor (XtDisplayOfObject (gw), XoVisualGet (gw));
}

/*
 * XoVisualColor -
 *	Decide if a given visual is for a monochrome or color screen.  If a
 *	colormap in the visual has 4 or more cells, it is declared a color
 *	visual.
 */
Boolean
XoVisualColor (dpy, visual)
	Display        *dpy;
	Visual         *visual;
{
	XVisualInfo     info;
	XVisualInfo    *info_ret;
	int             count;
	Boolean         is_color;
	static Display *last_dpy;
	static Visual  *last_visual;
	static Boolean  last_return;

	if (!dpy || !visual)
		return (False);

	if (dpy == last_dpy && visual == last_visual)
		return last_return;
	last_dpy = dpy;
	last_visual = visual;
	/*
	 * It sounds assbackward but given a visual we need to go and query
	 * the server to find a visual that matches.  We do this by setting
	 * the visualid field to uniquely identify the visual. We get back a
	 * list of matching  visuals (presumably 1) and then return True if
	 * _a_ colormap on that visual has 4 or more cells.
	 */
	info.visualid = XVisualIDFromVisual (visual);
	info_ret = XGetVisualInfo (dpy, VisualIDMask, &info, &count);
	if (!count || !info_ret)
	{
		is_color = False;
	}
	else
	{
		is_color = info_ret->colormap_size >= 4;
	}
	XFree ((char *) info_ret);
	last_return = is_color;
	return (is_color);
}

/*
 * _XoGravityAction -
 *	Return the gravity as specified by the parameters to an action
 *	function.  If none are specified then return the specified default.
 */

XoGravity
_XoGravityAction (params, num, gravity_default)
	String		*params;
	Cardinal	*num;
	XoGravity	gravity_default;
{
	int		i;
	XoGravity	gravity_return = gravity_default;

	for (i =  0; i < *num; i++)
	{
		if (params && params[i])
		{
			gravity_return = XoGetGravityFromString (params[i]);
			if (gravity_return >= 0)
				break;
			else
				gravity_return = gravity_default;
		}
	}
	return (gravity_return);
}
