;;;
;;; This file contains the scheme code which implements
;;; a left-to-right substitution model.
;;;
;;;
;;; Derek Lindner  buddha@theory.lcs.mit.edu
;;; Justin Liu     dondon@theory.lcs.mit.edu
;;; Brian So       brianso@theory.lcs.mit.edu
;;;

;;;;;;;;;;;;;;;;;;
;;;            ;;;
;;; CONSTAPPLY ;;;
;;;            ;;;
;;;;;;;;;;;;;;;;;;

; CONSTAPPLY takes an application whose operator
; and operands have all been reduced to values as
; defined above and does the approriate application
; in Scheme

(define (constapply exp)                ;exp is a combination (i.e. application) of values
  (cond
   ;((letc-application? exp let?)                       ; This should be moved
   ;	 (set! display-variable #t)                  ; out of constapply
   ;	 (letc-application exp let? make-let))       ; and redone.
   ((letc-application? exp letrec?)             ;
    (set! display-variable #f)                  ; See the letrec-out note below.
    (record-step "OUT")
    (letc-application exp letrec? make-letrec)) ;
   ((rule-defined-listop? exp)
    (set! display-variable #t)
    (record-step "CNST: " (car exp))
    (apply-rule-defined-listop exp))
   ((and (definable-op? exp)
         (expand-op? (car exp)))
    (record-step "CNST" (car exp))
    (definable-op-expand exp))
   ((known-constant? (car exp))              ;;REVISED
        (record-step "CNST" (car exp))
        (magic-eval exp))
      (else (sm-error "Apply unknown object" exp))))

;;; let-apply and letrec-apply
;;; this section handles the "bubbling-out" of lets and letrecs

(define (letc-application? exp expected-letc?)
  (cond ((expected-letc? exp)
	 (letc-application? (body-letc exp) expected-letc?))
	((pair? exp)
	 (if (letc-in-exp? exp expected-letc?)
	     #t
	     (or (letc-application? (car exp) expected-letc?)
		 (letc-application? (cdr exp) expected-letc?))))
	(else #f)))

(define (letc-application exp expected-letc? constructor)
  (cond ((expected-letc? exp)
         (garbage-collect
          (constructor (bindings-letc exp)
                       (letc-apply-rule (body-letc exp) expected-letc? constructor))))
        ((pair? exp)
	 (if (letc-in-exp? exp expected-letc?)
	     (letc-apply-rule exp expected-letc? constructor)
	     (cons (car exp)
		   (letc-apply-rule (cdr exp) expected-letc? constructor))))
	(else exp)))

(define (letc-in-exp? exp expected-letc?)
  (if (not (pair? exp)) 
      #f
      (if (expected-letc? (car exp))
	  #t
	  (letc-in-exp? (cdr exp) expected-letc?))))

(define (letc-apply-rule exp expected-letc? constructor)
  (let* ((let-pos (first-letc-position exp expected-letc?))
	 (letexp (list-ref exp let-pos)) 
	 (fresh-z-list (completely-fresh-zs (-1+ (length exp)))))
    (define (helper exp pos zlist)
      (cond ((null? exp)
	     '())
	    ((= pos let-pos)
	     (cons (body-letc letexp)
		   (helper (cdr exp) (1+ pos) zlist)))
	    (else (cons (car zlist)
			(helper (cdr exp)
				(1+ pos)
				(cdr zlist))))))
    (substitute (make-substlist
		 fresh-z-list
		 (remove-first-letc exp expected-letc?))
		(constructor
		 (bindings-letc letexp)
		 (helper exp 0 fresh-z-list)))))
		
(define (first-letc-position exp expected-letc?)
  (define (helper exp n)
    (if (expected-letc? (car exp))
	n
	(helper (cdr exp) (1+ n))))
  (helper exp 0))

(define (remove-first-letc exp expected-letc?)
  (cond ((expected-letc? (car exp))
	 (cdr exp))
	(else (cons (car exp)
		    (remove-first-letc (cdr exp) expected-letc?)))))



;;;;;;;;;;;;;;;;;;;;;;;
;;;                 ;;;
;;; reduction rules ;;;
;;;                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;

; variable-name is a placeholder for the name of the variable 
; currently being looked up.

(define variable-name '())

(define fresh
  (generate-uninterned-symbol 'fresh-variable-with-unlikely-name))

; NOTE: letrec-out should be done more in this style, but I haven't
; the time to deal now.
;
;(define (letrec-free-values-followed-by-letrec exp)
;  (if (not(pair? exp))
;      #f
;      (let ((hd (car exp)))
;	(or (letrec? hd)
;	    ((and (letrec-free-value? hd)
;		  (letrec-free-values-followed-by-letrec (cdr exp))))))))
;
;(define (letrec-out-redex? exp)
;  (if (not(pair? exp))
;      #f
;      (let ((operator (car exp)))
;	(or (and (non-binding-keyword? operator)
;		 (letrec-free-values-followed-by-letrec (cdr exp)))
;	    (letrec-free-values-followed-by-letrec exp)))))
;
;(define (reduce-letrec-out-redex exp) ...)


(define (left-red exp)              ;exp should arrive garbage-collected and
                                    ;(left-red exp) will be returned garbage-collected 
  (cond ((value? exp) exp)
	((variable? exp)
	 (set! variable-name exp)
         (record-step "INST" exp)
	 fresh)
	((and (beta-redex? exp) (all-values? (randb exp)))
	 (reduce-lam-app exp))
;	((letrec-out-redex? exp) ;handles letrec-out, so must
;	                         ;appear before:
;	                         ;and or if cond begin
;	 (reduce-letrec-out exp))
	((all-values? exp) (constapply exp))
	((if? exp) (reduce-if exp))
	((cond? exp) (reduce-cond exp))
	((or? exp) (reduce-or exp))
	((and? exp) (reduce-and exp))
	((let? exp) (if (named-let? exp)
			(reduce-named-let exp)
			(reduce-let exp)))
	((letrec? exp) (reduce-letrec exp))
	((sequence? exp) (reduce-sequence exp))
	((combination? exp) (reduce-combination exp))
	(else (sm-error "LEFT-RED: can't reduce expression" exp))))

(define (left-reduce exp0)              ;;COMMENT: handles def-<var>'s of previous
  (set! variable-name ())               ;;top-level defines
  (let* ((exp1 (left-red exp0))
	 (exp2 (cond ((null? variable-name) exp1)
		     (else (let ((bind (lookup variable-name)))
			     (set! variable-name ())
			     (substitute
			      (list (list fresh
					  bind))
			      exp1))))))
    exp2))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                      ;
; Implement the lam-app rule.  This is not beta reduction!             ;
;                                                                      ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (reduce-lam-app beta-redex)
  (let* ((params (paramb beta-redex))
         (rands (randb beta-redex))
         (FVrands (mapunion free-variables rands))
         (body (bodyb beta-redex))
         (newparams (freshvlist-letc params FVrands (free-variables body)))
         (substlist (extend-sub '() newparams params)))
    (cond ((= (length params)
              (length rands))
           (set! display-variable #f)
           (record-step "LAM")
           (garbage-collect
            (make-letrec
             (make-substlist
              (substitute substlist params)
              rands)
             (substitute substlist body))))
          (else
           (sm-error "ERROR: wrong number of arguments passed to procedure")))))

(define (reduce-if ifexp)               ;REVISED
  (let ((test (garbage-collect (test-if ifexp))))
    (cond ((value? test)
           (set! display-variable #f)
           (record-step "IF")
           (if (false? test)
               (if (= (length ifexp) 4)
                   (garbage-collect (alternative-if ifexp))
                   undefined-value)
               (garbage-collect (consequent-if ifexp))))
          (else (make-if (cons (left-red test)
                               (cddr ifexp))))))) ;REVISED

(define (reduce-cond condexp)           ;REVISED
  (let ((clauses (clauses-cond condexp)))
    (if (null? clauses)
        undefined-value
        (let* ((clause1 (first-clauses clauses))
               (test (garbage-collect (test-clause clause1))))
          (cond ((false? test)
                 (set! display-variable #f)
                 (record-step "COND")
                 (make-cond (rest-clauses clauses)))
                ((or (eq? test 'else) (value? test))
                 (set! display-variable #f)
                 (record-step "COND")
                 (if (= (length clause1) 1)
                     test
                     (consequent-clause clause1)))
                (else (make-cond
                       (cons
                        (if (= (length clause1) 1)
                            (make-clause-cond (left-red test))
                            (make-clause-cond (left-red test) (consequent-clause clause1)))
                        (rest-clauses clauses)))))))))

(define (reduce-or orexp)
  (let ((clauses (cdr orexp)))
    (set! display-variable #t)
    (record-step "OR")
    (if (null? clauses)
        #f
        (let* ((first (garbage-collect (car clauses)))               ;REVISED
               (rest  (cdr clauses)))
          (cond ((null? rest) first)
                ((value? first)
                 (if (false? first) (make-or rest) first))     ;REVISED
                (else
                 (set! display-variable #f)
                 (make-or (cons (left-red first) rest))))))))

(define (reduce-and andexp)
  (let ((clauses (cdr andexp)))
    (set! display-variable #t)
    (record-step "AND")
    (if (null? clauses)
        #t
      (let* ((first (garbage-collect (car clauses)))
             (rest  (cdr clauses)))
          (cond ((null? rest) first)
                ((value? first)
                  (if (false? first) first (make-and rest)))     ;REVISED
                (else
                 (set! display-variable #f)
                 (make-and (cons (left-red first) rest))))))))

; Maybe let should be sugar?
(define (reduce-let letexp)
  (record-step "LET")
  (cons (make-lambda (binding-variable-list (bindings-letc letexp))
		     (body-letc letexp))
	(binding-init-list (bindings-letc letexp))))

(define (reduce-named-let letexp)
  (let ((loopname (cadr letexp)) 
	(binds (caddr letexp)) 
	(body (cadddr letexp)))
    (record-step "NMDLET")
    (garbage-collect
     (make-letrec (make-bindings-letc
		  (cons loopname (binding-variable-list binds))
		  (cons (make-lambda (binding-variable-list binds) body)
			(binding-init-list binds)))
		 body))))

; REDUCE-BINDINGS-ONE reduces the first non-value
; binding in a LET declaration one step

(define (reduce-bindings-one bindings)  ;REVISE delete?: unused and anyway it's wrong 
  (cond ((null? bindings)
	 '())
	((value? (first-binding-init bindings))
	 (cons (first-binding bindings)
	       (reduce-bindings-one (rest-binding bindings))))
	(else (cons (list (first-binding-var bindings)
			  (left-red
			   (first-binding-init bindings))) ;REVISE: first-init must be inside
                                                           ;letrec bindings
		    (rest-binding bindings)))))

(define (reduce-letrec letexp)
  (let ((body (body-letc letexp))
        (bind (bindings-letc letexp)))
    (cond ((not (all-bindings-letrec-free-values? bind))
           (garbage-collect
            (make-letrec (reduce-bindings-one-letrec
                          bind
                          (binding-variable-list bind)
                          (free-variables body))
                         body)))
          ((not (value? body))
           (let ((new-body (left-red (garbage-collect body))))
             (if (null? variable-name)
                 (garbage-collect
                  (make-letrec bind new-body))
                 (let ((assn (assq variable-name bind)))
                   (cond ((null? assn) (make-letrec bind new-body))
                         (else (set! display-variable #t)
                               (set! variable-name ())
                               (garbage-collect
                                (make-letrec
                                 bind
                                 (substitute
                                  (list (list fresh (cadr assn)))
                                  new-body))))))))))))


; REDUCE-BINDINGS-ONE-LETREC reduces the first non-value
; binding in a LETREC one step and also checks for 
; unassigned variables

(define (reduce-bindings-one-letrec bindings all-bind-vars FVbody)
    (if (null? bindings)                ;;REVISED: null? test BEFORE let
        '()
        (let ((first-var (first-binding-var bindings))
              (first-init (first-binding-init bindings)))
          (cond ((value? first-init)
                 (if (letrec-free-value? first-init)
               (cons (first-binding bindings)
                     (reduce-bindings-one-letrec (rest-binding bindings)
                                                 all-bind-vars
                                                 FVbody))
               (let ((newbinds (append (bubble-binding
                                        first-var
                                        first-init
                                        all-bind-vars
                                        FVbody)
                                       (rest-binding bindings))))
                 (record-step "FLAT")
                 newbinds)))
                      ;;; Improve to return fewer UNASSIGNED errors:
          ((and (intersect (free-variables first-init) all-bind-vars)
                (not (lambda? first-init)))
           (sm-error "ERROR: Unassigned variable(s)" (intersect (free-variables
                                                                first-init)
                                                               all-bind-vars)))
          (else (let* ((var (first-binding-var bindings))
                       (var-body (first-binding-init bindings)))
                  (cons (list var (left-red var-body))
                        (rest-binding bindings))))))))

(define (bubble-binding var init all-bind-vars FVbody)
  (let* ((letbinds (bindings-letc init))
         (vars-to-avoid (union all-bind-vars FVbody))
         (var-list (binding-variable-list letbinds))
         (all-free-vars (union vars-to-avoid (binding-variable-list letbinds)))
         (Zn (freshvlist-letc var-list vars-to-avoid all-free-vars)))
    (set! display-variable #t)
    (append (make-bindings-letc
             Zn
             (map garbage-collect (binding-init-list letbinds))) ;REVISED
            (list (list var (substitute (smart-make-substlist var-list Zn)
                                        (body-letc init)))))))

(define (reduce-combination exp)
  (let ((fe (garbage-collect (first exp))) ;REVISED
        (re (rest exp)))
    (if (value? fe)
        (if (null? re)
            fe
            (cons fe (reduce-combination re)))
        (cons (left-red fe) re))))

(define (reduce-sequence seqexp)
  (let* ((exps (expression-sequence seqexp))
         (firstexp (garbage-collect (first exps)))
         (restexps (rest exps)))
    (cond ((value? firstexp)
           (set! display-variable #f)   ;REVISED
           (record-step "BEG")        ;REVISED
           (if (null? restexps)
               firstexp
               (make-sequence restexps)))
          (else (make-sequence
                 (cons (left-red firstexps)
                       restexps))))))

(define (define-output var)
  (newline)
  (display (string-append "The variable `" 
	    (symbol->string var)
	    "' has been defined."))
  (newline)
  undefined-value)

; REDUCE-UNTIL-VALUE reduces an expression to a value
; without displaying the reduction.

;(define (reduce-until-value exp) 
;  (cond ((value? exp)
;	 (set! display-variable #t)
;	 exp)
;	(else (reduce-until-value (left-reduce exp)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                      ;
; The main evaluation loop                                             ;
;                                                                      ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (driver-loop)
  (newline)
  (set! display-variable #f)
  (cond (exit-loop-flag
         (set! exit-loop-flag #f)
         (display "Exiting SUB-EVAL")
         (newline)
         'done)
        (else (display "SUB-EVAL==> ")
              (newline)
              (set! display-variable #t)
              (set! numbered-variable-list '())
              (set! line-count 0)
              (set! record-of-steps ())
              (let* ((exp0 (read))
                     (exp1 (top-level-desugar exp0))
                     (exp2 (garbage-collect exp1)))
                (display-reductions exp2)
                (eval-loop exp2)
                (driver-loop)))))

(define (eval-loop exp)  ;;exp should be garbage-collected already
  (cond ((equal? exp '(driver-loop))
         (newline)
         (display "Already within driver-loop. C-c C-c to exit")
         (driver-loop))
        ((interface-command? exp) (dispatch-interface-command exp))
        ((define? exp)
         (let* ((def-variable (cadr exp))
                (def-exp (garbage-collect (caddr exp)))
                (def-value (eval-loop0 def-exp)))
           (do-bindings def-variable def-value)
           (display-reductions def-value)
           (define-output def-variable)))
        (else
         (let ((final-val (eval-loop0 exp)))
           (if (zero? line-count) (display-reductions final-val 'force-display))
         (display-reductions final-val)))))

(define (eval-loop0 exp)   ;;exp should be garbage-collected already
  (let loop ((exp exp))
    (cond ((value? exp) exp)
	  (else (set! line-count (1+ line-count))
		(let ((exp1 (left-reduce exp)))
		  (cond ((or display-variable
			     verbose)
			 (display-reductions exp1)
			 (set! display-variable #f)))
		  (loop exp1))))))


;;; GLOBAL VARIABLES REVISE: Separate system globals from user-option globals (update?)

;SYSTEM
;USER-OPTIONS
;TABLES
                                     
; definable-ops                       a list
; display-variable                    #t/#f
; exit-loop-flag                      #t/#f
; expand-assocs                       assoc list binding definable ops to #t/#f expand flag value
; fresh                               uninterned symbol for INST rule
; functional-constants                a list
; keywords                            a list
; line-count                          0,1,...
; line-of-first-step                  0
; numbered-variable-list              assoc list
; record-of-steps                     a list
; rule-defined-listops                a list
; scheme-listops                      #t/#f
; show-rule-names                     #t/#f
; side-effect-constants               a list
; the-global-environment              a list
; undefined-value                     '*undefined-value*
; variable-name                       used by lookup
; verbose                             #t/#f
