;; Copyright (c) 1990-1994 The MITRE Corporation
;; 
;; Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;;   
;; The MITRE Corporation (MITRE) provides this software to you without
;; charge to use, copy, modify or enhance for any legitimate purpose
;; provided you reproduce MITRE's copyright notice in any copy or
;; derivative work of this software.
;; 
;; This software is the copyright work of MITRE.  No ownership or other
;; proprietary interest in this software is granted you other than what
;; is granted in this license.
;; 
;; Any modification or enhancement of this software must identify the
;; part of this software that was modified, by whom and when, and must
;; inherit this license including its warranty disclaimers.
;; 
;; MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;; OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;; OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;; FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;; SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;; SUCH DAMAGES.
;; 
;; You, at your expense, hereby indemnify and hold harmless MITRE, its
;; Board of Trustees, officers, agents and employees, from any and all
;; liability or damages to third parties, including attorneys' fees,
;; court costs, and other related costs and expenses, arising out of your
;; use of this software irrespective of the cause of said liability.
;; 
;; The export from the United States or the subsequent reexport of this
;; software is subject to compliance with United States export control
;; and munitions control restrictions.  You agree that in the event you
;; seek to export this software or any derivative work thereof, you
;; assume full responsibility for obtaining all necessary export licenses
;; and approvals and for assuring compliance with applicable reexport
;; restrictions.
;; 
;; 
;; COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994

(defun filename-basename ()
  (let ((fn (if (buffer-file-name)
		(file-name-nondirectory (buffer-file-name))
	      (buffer-name))))
    (substring
     fn
     0
     (string-match "\\." fn))))

(defun create-schemeweb-file ()
  (let ((path
	 (format "%s%s.sw"
		 (file-name-directory (buffer-file-name))
		 (filename-basename))))
    (and (file-exists-p path)
	 (or (y-or-n-p
	      (format "Overwrite file [%s]? " path))
	     (error "Schemeweb file already exists")))
    (write-file path)))

(defconst sw-special-characters '(?^ ?_ ?$ ?# ?%))


(defun sw-protect-special-characters (str)
  (let ((l nil)
	(i 0)
	(len (length str)))
    (while (< i len)
      (let ((ch (aref str i)))
	(if (memq ch sw-special-characters)
	    (setq l (cons ch (cons ?\\ l)))
	  (setq l (cons ch l)))
	(setq i (1+ i))))
    (concat (reverse l))))

(defun sw-omit-special-characters (str)
  (let ((l nil)
	(i 0)
	(len (length str)))
    (while (< i len)
      (let ((ch (aref str i)))
	(and (not (memq ch sw-special-characters))
	     (setq l (cons ch l)))
	(setq i (1+ i))))
    (concat (reverse l))))
  

(defvar sw-omit-def-form-pattern
  "^(def-\\(renamer\\|theory-ensemble-overloadings\\|parse-syntax\\|print-syntax\\|overloading\\)"
  "Defuns starting with a match to this pattern are removed from the Schemeweb file.")

(defvar sw-make-figure-pattern
  "^(def-\\(atomic-sort\\|constant\\|recursive-constant\\|quasi-constructor\\|theorem\\)"
  "Figures are made for defuns starting with a match to this pattern.
The figure is constructed by xviewing the result of ")

(defvar sw-omit-proofs t)

(defun sw-omit-trivial-def-forms ()
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward sw-omit-def-form-pattern nil 1)
    (goto-char (match-beginning 0))
    (insert "\\iffalse%\n")
    (goto-char (scan-sexps (point) 1))
    (insert "\n\\fi%\n")))

(defun sw-omit-proofs ()
  (interactive)
  (goto-char (point-min))
  (save-restriction
    (while (re-search-forward "^(def-theorem" nil t)
      (let ((start (match-beginning 0))
	    (end   (scan-sexps (match-beginning 0) 1)))
	(and (search-forward "(proof" end t)
	     (goto-char (match-beginning 0))
	     (insert "\\iffalse%\n  ")
	     (goto-char (scan-sexps (match-beginning 0) 1))
	     (insert "\n\\fi%\n"))))))


(defvar sw-figure-index 0
  "Number to put in place of NN in /tmp/$USER-imps-NN.tex when requesting Imps construct 
the next figure.")

(defvar sw-figure-markers '()
  "Alist of figure indices and markers to insert resulting imps tex output.")


(defun sw-get-kind-name-text-and-theory-name (here)
  "Return list (kind name text theory-name) for the current standard def-form."
  (let ((parse-sexp-ignore-comments t)
	kind name text theory-name)
    (save-excursion 
      (goto-char here)
      (and
       (re-search-forward "^(def-" nil t)
       (progn
	 (goto-char (match-beginning 0))
	 (looking-at sw-make-figure-pattern))
       (let ((def-form-end (scan-sexps (match-beginning 0) 1)))
	 (setq kind (buffer-substring (match-beginning 1)(match-end 1)))
	 (goto-char (match-end 1))
	 (cond ((looking-at "\\s *\\((\\s *)\\)")
		(setq name "Anonymous"))
	       (t
		(setq name (next-symbol-string (point)))))
	 (forward-sexp 2)		;skip name and text
	 (backward-sexp 1)		;beginning of text
	 (and
	  (= (char-after (point)) 34)	; Ascii for double quote char
	  (setq	
	   text				;omit quotes 
	   (buffer-substring (1+ (point))
			     (1- (scan-sexps (point) 1)))))
	 (forward-sexp 1)		;skip contents
	 (save-excursion
	   (and (search-forward "(theory" def-form-end)
		(setq
		 theory-name
		 (sw-protect-special-characters (next-symbol-string (point))))))
	 (while (and (string= kind "theorem")
		     (not (last-list-item-p (point))))
	   (if (string= (next-sexp-as-string) "lemma")
	       (setq kind "lemma")
	     (forward-sexp 1)))
	 (list kind name text theory-name))))))

(defun sw-process-standard-def-form (here)
  "Replace the def-form following HERE with a latex theorem-like environment."
  (interactive "d")
  (let ((kind-name-text-and-theory-name 
	 (sw-get-kind-name-text-and-theory-name here))
	(new-marker (make-marker)))
    (and 
     kind-name-text-and-theory-name
     (let ((kind        (nth 0 kind-name-text-and-theory-name))
	   (name        (nth 1 kind-name-text-and-theory-name))
	   (text        (nth 2 kind-name-text-and-theory-name))
	   (theory-name (nth 3 kind-name-text-and-theory-name)))
       (if (not text)
	   (goto-char (scan-sexps here 1))
	 (insert
	  (format "
\\begin{%s}
{\\bf (%s)}
\\label{%s:%s}
Theory: %s

%%%% Contents of %s to be inserted here.

"
		  (sw-omit-special-characters kind)
		  (sw-protect-special-characters name)
		  (sw-omit-special-characters kind)
		  (sw-omit-special-characters name)
		  theory-name
		  (substitute-in-file-name
		   (format
		    "/tmp/$USER-imps-%d.tex"
		    sw-figure-index))))
	 (set-marker new-marker (point)(current-buffer))
	 (setq sw-figure-markers
	       (cons
		(cons new-marker sw-figure-index)
		sw-figure-markers))
	 (insert
	  (format
	   "

\\end{%s}
"
	   (sw-omit-special-characters kind)))
	 (sw-send-string theory-name text)
	 (re-search-forward "^(def-" nil t)
	 (goto-char (match-beginning 0))
	 (insert "
\\iffalse% 
")
	 (forward-sexp 1)
	 (insert "
\\fi% 
"
		 ))))))


(defun sw-process-theory-def-form (here)
  "Construct a latex figure of the axioms of the theory def-form after HERE,
modifying the def-form to refer to the figure."
  (interactive "d")
  (and
   (re-search-forward "^(def-" nil t)
   (progn
     (goto-char (match-beginning 0))
     (looking-at "^(def-theory"))
   (let* ((theory-start (point))
	  (theory-end (scan-sexps (point) 1))
	  (theory-end-marker
	   (set-marker
	    (make-marker)
	    theory-end)))
     (down-list 1)
     (forward-sexp 2)
     (backward-sexp 1)			;point at start of theory name
     (let ((theory-name (next-symbol-string (point)))
	   (new-marker (make-marker)))
       (and
	(search-forward "(axioms" theory-end t)
	(progn
	  (beginning-of-line)
	  (insert
	   (format 
	    "  ;; Axioms---see Figure \\ref{fig:theory:%s}\\iffalse \n"
	    (sw-omit-special-characters theory-name)))
	  (forward-sexp 1)		;skip the axioms
	  (insert "\n  ;; \\fi\n")
	  (sw-send-theory-axioms theory-name)
	  (goto-char here)
	  (insert
	   (format
	    "\n\\begin{figure}
\\begin{center}
\\fbox{\\begin{minipage}{4.5in}
%%%% Contents of %s to be inserted here.\n\n"
	    (substitute-in-file-name
	     (format
	      "/tmp/$USER-imps-%d.tex"
	      sw-figure-index))))
	  (set-marker new-marker (point)(current-buffer))
	  (setq sw-figure-markers
		(cons
		 (cons new-marker sw-figure-index)
		 sw-figure-markers))
	  (insert
	   (format
	    "\n\n\\end{minipage}}\n\\end{center}
\\caption{Components and axioms for %s}\n\\label{fig:theory:%s}
\\end{figure}\n\n"
	    (sw-protect-special-characters theory-name)
	    (sw-omit-special-characters theory-name)))
	  (sw-send-theory-axioms theory-name)
	  (goto-char theory-end-marker)))))))
	       

(defun sw-send-theory-axioms (theory-name)
  (process-send-string
   tea-process
   (format
    "(xview-theory-for-figure '%s \"%s\")\n"
    theory-name  
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))

(defun sw-get-axiom-names-and-text-w-boundaries (here)
  "Return list (name text theory-name) lists for the axioms 
of a theory."
  (let ((parse-sexp-ignore-comments t)
	return-vals)
    (save-excursion 
      (goto-char here)
      (and
       (looking-at "^(def-theory")
       (search-forward "(axioms" (scan-sexps (match-beginning 0) 1) t)
       (while (not (last-list-item-p (point)))
	 (let (name text name-boundaries text-boundaries)
	   (down-list 1)
	   (forward-sexp 1)
	   (backward-sexp 1)		;position at first axiom-spec item
	   (if (= (char-after (point))
		  34)			;Ascii for double quote char
	       (setq text-boundaries (next-sexp-boundaries (point))
		     text (buffer-substring (1+ (car text-boundaries))
					    (1- (cdr text-boundaries))))
	     (setq name-boundaries (next-symbol-boundaries (point))
		   name (next-symbol-string (point)))
	     (forward-sexp 2)
	     (backward-sexp 1)		;position at second axiom-spec item
	     (setq text-boundaries (next-sexp-boundaries (point))
		   text (buffer-substring (1+ (car text-boundaries))
					  (1- (cdr text-boundaries)))))
	   (setq return-vals
		 (cons (list name text name-boundaries text-boundaries)
		       return-vals))
	   (up-list 1)))
       (reverse return-vals)))))


(defun sw-make-figure-for-def-form ()
  "Check whether the current def-form needs a figure for its second (string) argument,
and request T to create it in the file /tmp/$USER-imps-NN.tex."
  (and
   (looking-at sw-make-figure-pattern)
   (let ((kind (buffer-substring (match-beginning 1)(match-end 1))))
     (save-restriction
       (let ((start (point))
	     (end (scan-lists (point) 1 0)))
	 (down-list 1)
	 (forward-sexp 1)		; Move beyond "def-xx"
	 (or
	  (looking-at "\\s (")
	  (let* ((name (next-symbol-string (point)))
		 (pr-name (sw-protect-special-characters name))
		 (om-name (sw-omit-special-characters name)))		       
	    (delete-region		;Delete unprotected name 
	     (next-symbol-start (point))
	     (next-symbol-end (point)))
	    (insert (concat " " pr-name)) ; Insert proteceted name 
	    (forward-sexp 1)		; move beyond string (or other arg) 
	    (backward-sexp 1)		; Move to beginning of string (or other arg)
	    (and
	     (= (char-after (point)) 34) ; Ascii for double quote char
	     (let* ((str-start (point))
		    (str-end   (scan-sexps (point) 1))
		    (str (buffer-substring (1+ str-start) (1- str-end))))
	       (delete-region str-start str-end)
	       (insert
		(format ";; Contents to be found in Figure~\\ref{fig:%s:%s}"
			kind om-name))
	       (search-forward "(theory " end)
	       (let ((theory-name (next-symbol-string (point)))
		     (new-marker  (make-marker)))
		 (end-of-defun)
		 (insert
		  (format "
\\begin{figure}
\\begin{center}
\\fbox{\\begin{minipage}{4.5in}
\\iffalse%% Text of %s to be inserted here\\fi%%
"
			  (substitute-in-file-name
			   (format
			    "/tmp/$USER-imps-%d.tex"
			    sw-figure-index))))
		 (set-marker new-marker (point)(current-buffer))
		 (setq sw-figure-markers
		       (cons
			(cons new-marker sw-figure-index)
			sw-figure-markers))
		 (insert
		  (format "
\\end{minipage}}
\\end{center}
\\caption{Contents of %s %s} \\label{fig:%s:%s}
\\end{figure}
"
			  kind pr-name kind om-name))
		 (sw-send-string theory-name str)))))))))))

(defun sw-send-string (theory-name str)
  (process-send-string
   tea-process
   (format
    "(xview-figure '%s \"%s\" \"%s\")\n"
    theory-name str 
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))

(defun sw-insert-imps-tex-output ()
  (interactive)
  (let ((buff (current-buffer)))
    (mapcar
     (function
      (lambda (marker-index)
	(and
	 (eq buff (marker-buffer (car marker-index)))
	 (goto-char (marker-position (car marker-index)))
	 (insert-file-contents
	  (substitute-in-file-name
	   (format
	    "/tmp/$USER-imps-%d.tex"
	    (cdr marker-index))))
	 (set-marker (car marker-index) nil))))
     sw-figure-markers)
    (setq sw-figure-markers nil)))

(defun sw-make-envs-for-all-standard-def-forms (&optional arg)
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward sw-make-figure-pattern nil t)
    (goto-char (match-beginning 0))
    (sw-process-standard-def-form (point))
    (sit-for 1))
  (or arg
      (message
       "Standard environments dispatched to Imps; please wait for retrieval..."))
  (sit-for 3)
  (sw-insert-imps-tex-output)
  (or arg
      (message "Figures retrieved from Imps; Schemeweb done.")))
  
(defun sw-make-figures-for-all-theory-def-forms (&optional arg)
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward "^(def-theory" nil t)
    (goto-char (match-beginning 0))
    (sw-process-theory-def-form (point))
    (sit-for 1))
  (or arg
      (message
       "Theory figures dispatched to Imps; please wait for retrieval..."))
  (sit-for 3)
  (sw-insert-imps-tex-output)
  (or arg
      (message "Figures retrieved from Imps; Schemeweb done.")))
;; 
;; (defun sw-process-file ()
;;   (interactive)
;;   (if (not (y-or-n-p "Have you already loaded this file into Imps? "))
;;       (error "Please load the file into Imps.")
;;     (setq sw-figure-markers nil)
;;     (sw-make-figures-for-all-def-forms)
;;     (message "Figures dispatched to Imps.")
;;     (sit-for 3)
;;     (sw-insert-imps-tex-output)
;;     (save-buffer)))

(defun sw-process-file ()
  (interactive)
  (if (not (y-or-n-p "Have you already loaded this file into Imps? "))
      (error "Please load the file into Imps.")
    (message "Starting Schemeweb on file...")
    (sw-omit-trivial-def-forms)
    (sw-make-figures-for-all-theory-def-forms 'no-message)
    (sw-make-envs-for-all-standard-def-forms 'no-message)
    (save-buffer)
    (message "Starting Schemeweb on file... done")))

(defun sw-un-process-file ()
  (interactive)
  (if (y-or-n-p
       "Really undo Schemeweb on current buffer? ")
      (save-excursion
	(sw-undo-iffalses)
	(sw-undo-envs)
	(sw-undo-axioms)
	(message "Schemeweb undone"))
    (error "Schemeweb not undone")))

(defun sw-undo-iffalses ()
  (goto-char (point-min))
  (while (re-search-forward "\\\\iffalse%\\|\\\\fi%" nil t)
    (beginning-of-line)
    (kill-line)
    (delete-char 1)))

(defconst sw-envs
  '(constant theorem lemma recursive-constant atomic-sort figure)
  "Symbols naming environments Schemeweb creates, and should delete to undo.")

(defun sw-undo-envs ()
  (goto-char (point-min))
  (while (re-search-forward "\\\\begin{\\([^}]+\\)}" nil t)
    (let ((start (match-beginning 0))
	  (kind (intern (buffer-substring (match-beginning 1) (match-end 1)))))
      (and (memq kind sw-envs)
	   (search-forward (format "\\end{%s}" kind))
	   (delete-region start (match-end 0))))))

(defun sw-undo-axioms ()
  (goto-char (point-min))
  (while (re-search-forward "^\\s-*;; Axioms---see Figure.*$" nil t)
    (delete-region (match-beginning 0)(1+ (match-end 0)))
    (re-search-forward "^\\s-*;; \\\\fi.*$" nil nil)
    (delete-region (match-beginning 0)(1+ (match-end 0)))))
		 
(defun hide-lemmas ()
  (goto-char (point-min))
  (while (search-forward "\\begin{lemma}" nil t)
    (comment-region
     (match-beginning 0)
     (progn
       (search-forward "\\end{lemma}")
       (match-end 0)))))

(defun unhide-lemmas ()
  (goto-char (point-min))
  (while (search-forward "\\begin{lemma}" nil t)
    (beginning-of-line)
    (let ((here (point)))
      (search-forward "\\end{lemma}")
      (save-restriction
	(narrow-to-region here (match-end 0))
	(goto-char here)
	(while (re-search-forward "^\\s<+\\s-*" nil t)
	  (replace-match "")
	  (end-of-line))))))
     
