;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald reduce)

;;; REDUCE.T  7/17/89  dpr
;;; Reduce a set of linear equations to a non-redundant subset.

;;; REMOVE-REDUNDANCY accepts a list of conditions and returns a list
;;; of non-redundant conditions.  The set is first checked for
;;; feasibility.  Then the set is checked with each condition, one at a time,
;;; replaced by it's negation, non-feasibility indicating that the
;;; negated condition is redundant.

(define (remove-redundancy condition-list)
  (if (not (feasible-conds? condition-list))
      (error "Not feasible")
      (iterate loop ((index (-1+ (length condition-list)))
		     (c-list condition-list))
	       (if (>= index 0)
		   (if (feasible-conds?
			(list-replace
			 c-list index (vneg (list-ref c-list index))))
		       (loop (-1+ index) c-list)
		       (loop (-1+ index) (list-delete c-list index)))
		   c-list))))

;;; REMOVE-REDUNDANCY2 does the same thing, but accepts an A matrix and
;;; a B vector as input, and returns a list: (NEW-A NEW-B).

(define (remove-redundancy2 A B)
  (if (not (feasible? A B))
      (error "Not feasible")
      (iterate loop ((index (-1+ (vector-length B))))
	(if (>= index 0)
	    (if (feasible? (negate-row A index)
			   (negate-elm B index))
		(loop (-1+ index))
		(block (zero-row! A index)
		       (zero-elm! B index)
		       (loop (-1+ index))))
	    (remove-0-rows A B)))))


(define (number-of-0s vector)
  (iterate loop ((ans 0) (index (-1+ (vector-length vector))))
	   (if (>= index 0)
	       (if (= (vref vector index) 0)
		   (loop (1+ ans) (-1+ index))
		   (loop ans (-1+ index)))
	       ans)))

(define (remove-0-rows matrix vector)
  (let* ((length (- (num-rows matrix) (number-of-0-rows matrix)))
	 (new-m (make-vector length))
	 (new-v (make-vector length)))
    (iterate loop ((old-index 0) (new-index 0))
	     (if (< new-index length)
		 (if (not (zero-vector? (vref matrix old-index)))
		     (block (vset new-m new-index (vref matrix old-index))
			    (vset new-v new-index (vref vector old-index))
			    (loop (1+ old-index) (1+ new-index)))
		     (loop (1+ old-index) new-index))
		 (list new-m new-v)))))

(define (number-of-0-rows matrix)
  (iterate loop ((ans 0) (index (-1+ (vector-length matrix))))
	   (if (>= index 0)
	       (if (zero-vector? (vref matrix index))
		   (loop (1+ ans) (-1+ index))
		   (loop ans (-1+ index)))
	       ans)))

;;; FEASIBLE-CONDS? accepts a list of conditions and returns a boolean
;;; indicating if they form a feasible system.

(define (feasible-conds? c-list)
  (let* ((AB (conds->AB c-list))
	 (A (car AB))
	 (B (cadr AB)))
    (feasible? A B)))


;;; CONDS->AB accepts a list of m conditions:
;;;   a11x1 + a12x2 + ... + a1nxn <= b1   through
;;;   am1x1 + am2x2 + ... + amnxn <= bm   in the form:
;;;   #(a11 a12 ... a1n b1) through
;;;   #(am1 am2 ... amn bm)
;;; and returns a list:  (A B)

(define (conds->AB c-list)
  (let* ((m (length c-list))
	 (n (-1+ (vector-length (car c-list))))
	 (A (make-matrix m n))
	 (B (make-vector m)))
    (iterate loop ((i 0) (rest c-list))
	     (vset B i (vref (car rest) n))
	     (if (< i (-1+ m))
		 (loop (1+ i) (cdr rest))))
    (iterate loop ((i 0) (j 0) (rest c-list))
	     (mset A i j (vref (car rest) j))
	     (if (< j (-1+ n))
		 (loop i (1+ j) rest)
		 (if (< i (-1+ m))
		     (loop (1+ i) 0 (cdr rest))
		     (list A B))))))
