;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald MACETE-CONSTRUCTORS)


(define-operation (COMPOSITE-MACETE? x)
  (imps-enforce macete? x) '#f)

(define-operation (MACETE-COMPONENTS x)
  (imps-enforce macete? x) '())

(define-operation (MACETE-CONSTRUCTOR-NAME x))
(define-operation (MACETE-CONSTRUCTOR x) '#f)
(define-operation (CONSTRUCTOR-MACETE->STRATEGY-CONVERTER soi))
(define-operation (CONSTRUCTOR-KEY-AMALGAMATOR soi))

(let ((macete-constructor-table (make-table)))
  
  
  (define (MAKE-MACETE-CONSTRUCTOR
	   proc
	   soundness-predicate
	   bidirectionality-predicate
	   
	   macete->strategy-converter
	   key-amalgamator

	   constructor-name)
    ;;proc is procedure of three args: CONTEXT EXPR MACETES 
    ;;which should return an expression EXPR
    
    ;;returns a procedure of one argument: MACETES
    ;;This procedure when applied returns a macete
    ;;whose effect is some combination of the macetes
    ;;in the list MACETES.
    (enforce symbol? constructor-name)
    (set (table-entry macete-constructor-table constructor-name)
	 (labels ((constructor
		   (object
		       (lambda (macetes)
			 (enforce proper-list? macetes)
			 (walk (lambda (x) (enforce macete? x)) macetes)
			 (let ((macete
				(join
				  (object nil
				    ((composite-macete? soi) '#t)
				    ((macete-constructor soi) constructor)
				    ((macete-components x) macetes)
				    ((macete-constructor-name soi) constructor-name) 
				    ((macete-sound-in-theory? soi theory)
				     (soundness-predicate macetes theory)))
	     
				  (build-macete
				   (lambda (context expr)
				     (proc context expr macetes))
				   (bidirectionality-predicate macetes)
				   '()))))
			   macete))
		     ((constructor-key-amalgamator soi)
		      key-amalgamator)
		     ((constructor-macete->strategy-converter soi)
		      macete->strategy-converter))))
	   constructor)))
  
  (define (CONSTRUCTOR-NAME->MACETE-CONSTRUCTOR constructor-name)
    (table-entry macete-constructor-table constructor-name)))
  
;;some basic constructors:

(define (DEFAULT-BIDIRECTIONALITY-PREDICATE macetes)
  (every? bidirectional-macete? macetes))

(define (DEFAULT-SOUNDNESS-PREDICATE macetes theory)
  (every? (lambda (x) (macete-sound-in-theory? x theory)) macetes))

;;basic macete->strategy-converters

(define (DEFAULT-MACETE->STRATEGY-CONVERTER macete)
  (lambda (sqn)
    (deduction-graph-apply-macete sqn macete)))

(define (SERIES-MACETE->STRATEGY-CONVERTER macete)
  (series-strategy-constructor
   (map macete->strategy (macete-components macete))))
  
(define (REPEAT-MACETE->STRATEGY-CONVERTER macete)
  (repeat-strategy-constructor
   (map macete->strategy (macete-components macete))))
  
(define (SEQUENTIAL-MACETE->STRATEGY-CONVERTER macete)
  (sequential-strategy-constructor
   (map macete->strategy (macete-components macete))))

(define (PARALLEL-MACETE->STRATEGY-CONVERTER macete)
  (parallel-strategy-constructor
   (map macete->strategy (macete-components macete))))	

;;basic key amalgamators

(define (UNION-AMALGAMATOR key-lists)
  (big-u key-lists))

(define (GUARD-AMALGAMATOR key-lists)
  (car key-lists))

(define SERIES-MACETE-CONSTRUCTOR
  (make-macete-constructor
 
 
   ;;execute them in sequence.
   ;;Failure of one does not stop subsequent macetes from acting.
 
   (lambda (context expr macetes)
     (iterate loop ((expr expr) (macetes macetes))
       (cond ((null? macetes) expr)
	     (else (loop (apply-macete (car macetes) context expr) (cdr macetes))))))
 
   default-soundness-predicate
   default-bidirectionality-predicate
   series-macete->strategy-converter
   union-amalgamator

   'series))

(define WITHOUT-MINOR-PREMISES-MACETE-CONSTRUCTOR
  (make-macete-constructor
 
   (lambda (context expr macetes)
     
     (imps-enforce (lambda (x) (= (length x) 1)) macetes)

     (bind (((accumulate-undischarged-minor-premises?) '#f))
       (apply-macete (car macetes) context expr)))
 
   default-soundness-predicate
   default-bidirectionality-predicate
   default-macete->strategy-converter
   union-amalgamator

   'without-minor-premises))

(define HANDLING-MINOR-PREMISES-MACETE-CONSTRUCTOR
  (make-macete-constructor
 
   (lambda (context expr macetes)
     
     (imps-enforce (lambda (x) (= (length x) 2)) macetes)
     (let ((main-macete (car macetes))
	   (minor-macete (cadr macetes)))
       (receive (result new-minors)
	 (bind (((accumulate-undischarged-minor-premises?) '#t)	      
		(undischarged-minor-premises '()))
	   (let ((result (apply-macete main-macete context expr)))
	     (return result undischarged-minor-premises)))
	 (set undischarged-minor-premises
	      (set-union 
	       (map (lambda (seq)
		      (build-sequent
		       (sequent-context seq)
		       (apply-macete
			minor-macete
			(sequent-context seq)
			(sequent-assertion seq))))
		    new-minors)
	       undischarged-minor-premises))
	 result)))
	 
 
   default-soundness-predicate
   default-bidirectionality-predicate
   default-macete->strategy-converter
   union-amalgamator

   'handling-minor-premises))

(define SEQUENTIAL-MACETE-CONSTRUCTOR
  (make-macete-constructor
 
 
   ;Failure inhibits subsequent macetes from acting.
   ;For example, succesful application of induction might be a
   ;prerequisite for applying other macetes.
 
 
   (lambda (context expr macetes)
     (iterate loop ((expr expr) (macetes macetes))
       (cond ((null? macetes) expr)
	     (else (let ((new-expr (apply-macete (car macetes) context expr)))
		     (if (eq? new-expr expr) 
			 expr
			 (loop new-expr (cdr macetes))))))))
 
   default-soundness-predicate
   default-bidirectionality-predicate
   sequential-macete->strategy-converter
   guard-amalgamator

   'sequential))

(define PARALLEL-MACETE-CONSTRUCTOR
  (make-macete-constructor
 
   (lambda(context expr macetes)
     (iterate loop ((expr1 expr) (macetes macetes))
       ;;keep doing until something changes
       (cond ((null? macetes) expr1)
	     ((eq? expr1 expr)
	      (loop (apply-macete (car macetes) context expr) (cdr macetes)))
	     (else expr1))))
 
   default-soundness-predicate
   default-bidirectionality-predicate
   parallel-macete->strategy-converter
   union-amalgamator
 
   'parallel))

;something changed, so stop.

(make-macete-constructor
 
 (lambda(context expr macetes)
   (iterate loop ((macetes macetes))
     ;;keep doing until something changes
     (if (null? macetes) expr
	 (let ((expr1 (apply-macete (car macetes) context expr)))
	   (if (truth? expr1) expr1
	       (loop (cdr macetes)))))))
 
 default-soundness-predicate
 default-bidirectionality-predicate
 parallel-macete->strategy-converter
 union-amalgamator
 
 'any-true?)

(define REPEAT-MACETE-CONSTRUCTOR
  (make-macete-constructor
 
   (lambda (context expr macetes)
     (labels ((series
	       (lambda (expr macetes)
		 (cond ((null? macetes) expr)
		       (else 
			(series (apply-macete (car macetes) context expr) (cdr macetes))))))
	      (repeat
	       (lambda (expr)
		 (let ((new-expr (series expr macetes)))
		   (if (alpha-equivalent? new-expr expr) 
		       expr
		       (repeat new-expr))))))
       (repeat expr)))

   default-soundness-predicate
   default-bidirectionality-predicate
   repeat-macete->strategy-converter
   union-amalgamator

   'repeat))

(define SOUND-MACETE-CONSTRUCTOR
  (make-macete-constructor
   (lambda (context expr macetes)
   
     (enforce (lambda (x) (= (length x) 3)) macetes)
     
     (let ((macete (car macetes))
	   (source-macete (cadr macetes))
	   (target-macete (caddr macetes)))
       (imps-enforce bidirectional-macete? target-macete)
       (let ((new-expr (apply-macete macete context expr)))
	 (cond ((and (macete-sound-in-theory? source-macete (context-theory context))
		     (macete-sound-in-theory? target-macete (context-theory context))

		     ;;first insure SOURCE-MACETE, TARGET-MACETE
		     ;;which are used to verify the soundness of
		     ;;MACETE are themselves sound. For this reason, resulting
		     ;;macete is always sound.
		     
		     (alpha-equivalent?
		      (apply-macete source-macete context expr)
		      (apply-macete target-macete context new-expr)))
		new-expr)
	       (else expr)))))

;;;for formulas, since  target-macete is bidirectional
;;;new-expr iff
;;;(apply-macete target-macete context new-expr) which is eq?
;;;(apply-macete source-macete context expr) implies
;;;expr

   (lambda (x theory) (ignore x) (ignore theory) '#t)
   (lambda (macetes) (bidirectional-macete? (cadr macetes)))

   default-macete->strategy-converter
   guard-amalgamator

   'sound))
 

(define (BUILD-ANONYMOUS-MACETE sexp)
  (cond ((symbol? sexp) (cond ((name->macete sexp))
			      (else (imps-error "There is no macete ~a." sexp))))
	((list? sexp)
	 (cond ((constructor-name->macete-constructor (car sexp))
		=>
		(lambda (constructor)
		  (constructor
		   (map (lambda (x) (build-anonymous-macete x)) (cdr sexp)))))

	       (else (imps-error "~a is not a macete constructor" (car sexp)))))
	(else (imps-error "~a is not a valid form for a macete" sexp))))

(define (BUILD-MACETE-FROM-SEXP sexp name)
  (let ((macete (build-anonymous-macete sexp)))
    (set (macete-name macete) name)
    macete))

(define (BUILD-AND-INSTALL-MACETE-FROM-SEXP sexp name . dont-rename)
  (let ((macete (build-anonymous-macete sexp)))
    (or dont-rename (set (macete-name macete) name))
    (add-macete macete)
    macete))

;;;(define (BUILD-AND-INSTALL-MACETE-FROM-SEXP sexp name)
;;;  (let ((macete (build-macete-from-sexp sexp name)))
;;;    (add-macete macete)
;;;    macete))

;;;(define (MACETE-KEYS macete)
;;;  (cond ((elementary-macete? macete)
;;;	 (let ((expr (elementary-macete-lhs macete)))
;;;	   (list (cons (expression-quasi-constructor-or-constructor expr)
;;;		       (expression-lead-constant expr)))))
;;;	((composite-macete? macete)
;;;	 ((constructor-key-amalgamator (macete-constructor macete))
;;;	  (map macete-keys (macete-components macete))))
;;;	(else '())))
;;;
;;;


;;;conversion of macetes to strategies:

(define (MACETE->STRATEGY macete)
  (if (composite-macete? macete)
      ((constructor-macete->strategy-converter (macete-constructor macete))
       macete)
      (default-macete->strategy-converter macete)))

(define (BUILD-STRATEGY-FROM-MACETE macete)
  (build-universal-command
   (macete->strategy macete)
   (if (macete-name macete)
       (symbol-append 'converted% (macete-name macete))
       '())
   (always '#t)))

