/******************************************************************************
  anf_div.c                                                           

  This file contains:
  anf_div
  anf_div_z
 
******************************************************************************/
                 
#include "kant.h"
#include "integer.e"
#include "mat.h"
#include "anf.h"


anf_elt
anf_div WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************
 
Description:
 
	Divides alpha by beta.
	This is done via computing the representation matrix M of beta. 
	M multiplied by the (unknown) coefficient vector of the quotient yields 
	the coefficient	vector b of beta. So we have to compute the nullspace 
	of the matrix M with additional column -b. At the moment this is done 
        by using mat_z_null_space which calls Hermite normal form. In future 
	this also can be done by calling the modified LLL algorithm.
 
	The coefficient order must be Z.
                                                        
 
Calling sequence:
 
	gamma = anf_div(ord, alpha, beta);
 
      	order       ord   = t_handle of order alpha, beta are given referring to
        anf_elt     alpha = algebraic number (numerator)
        anf_elt     beta  = algebraic number (denominator)
        anf_elt     gamma = algebraic number (quotient, simplified)
     
History:
 
	92-03-31 JS    integer_anf_elt_lift if alpha is an integer; simplify
	91-10-01 JS    first version

*******************************************************************************/
{       
	block_declarations;
 
	integer_small	deg, degp1;
        order		ordcoef;
	matrix		repmat, mat, mat1, mat2, nulls;
	anf_elt		gamma, temp, alocal, gamma_sim;
	integer_big	denalpha, denbeta;
	integer_small	i;
                                   
  
	order_must_be_over_z(ord);
 
        ordcoef = order_coef_order(ord);
	deg 	= order_rel_degree(ord);
	degp1 	= deg + 1;
 
        alocal  = anf_elt_is_integer(alpha)
                        ? integer_anf_elt_lift(ord, alpha)
                        : anf_elt_incref(alpha);
 
	denalpha = anf_elt_den(alocal);
	denbeta  = anf_elt_is_integer(beta) ? 1 : anf_elt_den(beta);
 
/*
    We construct the matrix of which we want to compute the nullspace
*/
	mat1   = mat_new(deg, degp1);
 
	repmat = anf_rep_mat(ord, beta);
 
	for (i=1; i<=deg; ++i)
	{
		mat_elt(mat1, i, degp1) = 
		anf_negate(ordcoef, anf_elt_coef(alocal, i));
	}
 
	mat2 = mat_ring_insert(ordcoef, mat1, repmat, 1, 1);
  
/*
    unfortunately the nullspace program works on rows...
*/                         
	mat   = mat_ring_trans(ordcoef, mat2);
	nulls = mat_z_null_space(ordcoef, mat);
 
/*
    The row dimension must be one! We get the quotient.
*/
	anf_elt_alloc(gamma, deg);
	for (i=1; i<=deg; ++i)
		anf_elt_coef(gamma, i) = anf_elt_incref(mat_elt(nulls, 1, i));
 
/*
    Now for the denominator
*/ 
	anf_elt_den(gamma) = integer_mult(denalpha, mat_elt(nulls, 1, deg+1));
 
	if(denbeta != 1)
	{
		temp  = gamma;
		gamma = anf_mult_z(ord, temp, denbeta);
		anf_elt_delete(ord, &temp);
	}
/*
    we are done! Simplifying the result
*/
        gamma_sim = anf_elt_simplify(ord, gamma);
 

	mat_delref(ordcoef, &repmat);
	mat_delref(ordcoef, &mat);
	mat_delref(ordcoef, &mat1);
	mat_delref(ordcoef, &mat2);
	mat_delref(ordcoef, &nulls);
	anf_elt_delete(ord, &alocal);
	anf_elt_delete(ord, &gamma);
 
	return gamma_sim;
} 
 
 
 

anf_elt
anf_div_z WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	integer_big,	a
)
/*******************************************************************************
 
Description:
 
	Division of an algebraic number by a rational integer. 
	The division is done componentwise without checking for any rest.
 
 
Calling sequence:
 
	gamma = anf_div_z(ord, alpha, a);
 
      	order       ord   = t_handle of order alpha, beta are given referring to
        anf_elt     alpha = algebraic number (numerator)
        integer_big a     = denominator
        anf_elt     gamma = algebraic number (quotient)
 
History:
 
	91-09-01 JS    first version

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		beta;
	integer_small	deg;
        order		ordcoef;
        integer_big	temp;	
	integer_small	i;
 
 
/*
     trivial case:
*/
	if (anf_elt_is_integer(alpha)) return integer_div(alpha, a);
 
	deg = order_rel_degree(ord);
	anf_elt_alloc(beta, deg);
                                                   
	temp = anf_elt_den(alpha);
        anf_elt_den(beta) = integer_incref(temp);	
        
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z)
        {	
        	for (i=1; i<=deg; ++i)
			anf_elt_coef(beta,i) = 
                        integer_div(anf_elt_coef(alpha, i), a);
	}
	else
	{
        	for (i=1; i<=deg; ++i)
			anf_elt_coef(beta,i) = 
			anf_div_z(ordcoef, anf_elt_coef(alpha, i), a);
	}
         
	return beta;	
}

