/******************************************************************************
  anf_ideal_2_assure.c
******************************************************************************/
 
#include "stdio.h"        
#include "kant.h"
#include "anf.h"

t_void
anf_ideal_2_assure WITH_2_ARGS(
	order,		ord,
	anf_ideal,	id
)
/*******************************************************************************
 
Description:
             
	Tries to find a 2-Element-Presentation of id with a
	positive(!) integer as first generator.
	Notice: if id is a principal ideal, you will not be able
		to see this from the new Presentation.
  
Calling sequence:
 
        anf_ideal_2_assure(ord, id);
 
        order   	ord:        order over which id is defined
        anf_ideal       id:	    ideal to be in 2-Element-Presentation
 
History:

	93-01-04 CO	take care that first generator is > 0
			also in trivial case ( l. 54 pp )
	92-06-16 CO     first version

*******************************************************************************/
{       
	block_declarations;
 
	integer_big	m, idnorm, id1norm, dum1;
	anf_elt		hlp, alpha;
	anf_ideal	id1;
	vector		loop_vec, temp;
	matrix		mat;
	order		Z;                 
	t_int		i, comp;
 
	order_must_be_over_z(ord);


	if( !anf_ideal_is_integral( ord, id ))
		error_internal("anf_ideal_2_assure: Ideal is not integral.");

	
	if( anf_ideal_is_2( id ))
	    if( anf_elt_is_integer( anf_ideal_gen1(id) ))
	    {
		comp = integer_compare( anf_ideal_gen1(id), 0 );
		if( comp == 0 )
		{
		    if( anf_elt_is_zero( ord, anf_ideal_gen2(id) ))
			error_internal("anf_ideal_2_assure: Ideal is 0.");
		    else
			anf_ideal_gen1(id) = anf_elt_incref(
						anf_ideal_gen2(id) );
		}
		else
		{
		    if( comp == -1 )  /* gen1 < 0: mult. id with -1 */
		    {
			dum1 = integer_negate( anf_ideal_gen1(id));
			integer_delref( anf_ideal_gen1(id));
			anf_ideal_gen1(id) = dum1;

			hlp = anf_elt_negate( ord, anf_ideal_gen2(id));
			anf_elt_delete( ord, &anf_ideal_gen2(id));
			anf_ideal_gen2(id) = hlp;
		    }
		    return ;
		}
	    }

/******* Now our ideal has either no 2-Element-Presentation *******/
/******* or the first generator is not an integer.          *******/

/******* As first generator for a new 2-Elem.-Pres. we take *******/
/******* the minimum of positive integers in id.            *******/

	anf_ideal_min_assure( ord, id );
	anf_ideal_2_delete( ord, id );
	m = anf_ideal_min( id );                  

	anf_ideal_norm( ord, id, &idnorm, &dum1 );
	integer_delref( dum1 );
	mat = mat_incref( anf_ideal_tran( id ));
	Z = m_z_str_incref(structure_z);

/******* To find a second generator we use a probabilistic  *******/
/******* method: for the algebraic numbers                  *******/
/*******   alpha := b(1)*a(1) + ... + b(n)*a(n),            *******/
/******* (  -1 <= b(i) <= 1,  a(1), ..., a(n) ideal basis ) *******/
/******* we test if the norm of the ideal id1 generated by  *******/
/******* m and alpha is equal to norm( id ). In this case   *******/
/******* we have id = id1.				    *******/

	loop_vec = vector_index_init( order_abs_degree( ord ), -1 );
	for( i = -1; (vector_index_inc( loop_vec, -1, 1 )) 
		&& (!anf_ideal_is_2( id )); i++ )
	{          
		temp = mat_vector_col_mult( Z, mat, loop_vec );
		alpha = vec_to_anf_elt( ord, temp );
		id1 = anf_ideal_2_create( ord, m, alpha );
		anf_ideal_norm( ord, id1, &id1norm, &dum1 );
		integer_delref( dum1 );
        	if( integer_compare( idnorm, id1norm ) == 0 )
		{
			anf_ideal_gen1( id ) = integer_incref( m );
			anf_ideal_gen2( id ) = anf_elt_incref( alpha );
			anf_ideal_gen_g( id ) = 1;
		}
		integer_delref( id1norm );
		vec_delete( Z, &temp );
		anf_elt_delete( ord, &alpha );               
		anf_ideal_delete( ord, &id1 );
	}                                                        
	if( anf_print_level > 5 )
		printf("number of false trials in anf_ideal_2_assure: %d",i++);
	integer_delref( idnorm );
	vec_delete( Z, &loop_vec );           
	mat_delref( Z, &mat );
	ring_delete( &Z );
                                   
	if ( anf_ideal_is_2( id))
		return;
	else
		error_internal("anf_ideal_2_assure: No 2-Element-Presentation found.");
}

