/******************************************************************************
  mat_ring_gram_chol.c
******************************************************************************/
#include "kant.h" 
#include "real.e"
#include "mat.h"
 
#define	STD_PREC_QFORM 20

matrix 
mat_ring_gram_chol WITH_3_ARGS(
	t_handle,		gram_ring,
	matrix,		gram,
	t_handle*,	chol_ring
)       
/******************************************************************************
 
Description:	Computes the positive quadratic form from a gram matrix.
		The coefficients of the quadratic form are computed with by the 
		modificated Cholesky-method.
 
Calling sequence:
 
	qform = mat_ring_gram_chol(gram_ring,gram,&chol_ring)

	ring	gram_ring	= t_handle of ring
	matrix	gram		= t_handle of matrix
	ring	chol_ring	= t_handle of a real ring. 
				  If chol_ring is not initialized chol_ring
				  will become a real ring with the default
				  precision STD_PREC_QFORM (see above).				
History:
 
	92-03-05 AJ    minor changes		
	92-01-09 KW    minor changes
	91-12-?? AJ    written
 
******************************************************************************/
{        
	block_declarations;

	matrix		chol; 
	integer_small	i, j, l, k; 
	t_real		h1, h2, h3,zero;
	integer_small	row, col;

	row = mat_row(gram);
	col = mat_col(gram);

 	chol = mat_new( row, col );
    
	if (*chol_ring == MEM_NH)
	{   
		error_internal("mat_ring_gram_chol: The chol_ring is not defined.");
	}  
	else
	{
		if (ring_type(*chol_ring) != RING_R)
                   error_internal("mat_ring_gram_chol: chol_ring has wrong ring type.");
	}

	zero = ring_zero(*chol_ring);

	/*  chol <--- gram */

	for (i=1; i<=row; i++)
	{
		for (j=i; j<=col; j++) 
		{
			if (ring_type(gram_ring) == RING_R)
			{
				mat_elt(chol, i, j) = conv_real_to_real(mat_elt(gram,i,j),gram_ring,*chol_ring);
			}
			else
			{
				mat_elt(chol, i, j) = conv_int_to_real(*chol_ring,mat_elt(gram,i,j));
			}
		}
	}

	/*  This loops are exactly described in Fincke/Pohst  */

	for (i=1; i< row; i++)
	{
		h2 = real_incref(mat_elt(chol,i,i));

		for (j=i+1; j<=col; j++) 
		{
			mat_elt(chol, j, i) = real_incref(mat_elt(chol,i,j));
			h1 = mat_elt(chol,i,j);
			mat_elt(chol, i, j) = real_divide(*chol_ring,h1,h2);
			real_delete(&h1);
		}

		for (k=i+1; k<=row; k++) 
		{
			for (l=k; l<=col; l++) 
			{
				h1=  real_mult(*chol_ring,mat_elt(chol,k,i),mat_elt(chol,i,l));
				h3=  mat_elt(chol, k, l); 
				mat_elt(chol,k,l)= real_subtract(*chol_ring,h3,h1);
				real_delete(&h1);
				real_delete(&h3);
			}
		}

		for (k=1; k<i; k++) 
		{
			real_delete(&mat_elt(chol,i,k));
			mat_elt(chol,i,k) = real_incref(zero);
		}
		real_delete(&h2);
	}

	for (i=1; i<=col; i++)
	{
	        for (k=1; k<i; k++) 
		{
			real_delete(&mat_elt(chol,i,k));
			mat_elt(chol,i,k) = real_incref(zero);
		}
	}
	real_delete(&zero);

	return chol;
}    

