/*******************************************************************************
  mat_z_kernel.c 


  This file contains :

    mat_z_kernel

    _gj_pivelement
    _gj_tauschen
    _gj_io

********************************************************************************/


#include <stdio.h>             
#include "kant.h"     
#include "integer.e"
#include "mat.h"
#include "mat.e"                         


                      
void 
_gj_pivelement WITH_7_ARGS (t_handle          ,  cring       ,
                            matrix          ,  mat         ,
                            matrix          ,  zeile       ,
                            matrix          ,  spalte      ,
                            integer_small * ,  piv_zeile   ,
                            integer_small * ,  piv_spalte  ,
                            integer_big   * ,  piv_el       )
/*******************************************************************************
Description:               
   
   Searches the absolute largest element in mat = a(i,j) with the conditions
                1) Zeile (i)  < 0 
                2) Spalte (j) > 0                                

   Since this routine is only written for the function mat_z_kernel NO 
   checking is done whether cring is the ring of integers.
 

Calling sequence:            
    
        t_handle            cring       : t_handle to a ring-structure of ZZ.
        matrix            mat         : a standard Cayley-matrix. Within this
                                        matrix all searching is done.
        matrix            zeile       : indicates whether or not a certain row
                                        is admissible for searching.
        matrix            spalte      : dito for column
        integer_small     *piv_zeile  : the row - index of the maximum
        integer_small     *piv_spalte : the column - index of the maximum
        integer_big       *piv_el     : the value of the maximum.
                                                                          

     _gj_pivelement  (cring,mat,zeile,spalte,piv_zeile,piv_spalte,piv_el);

                                                                     
History:  
                                                               
       MD 91-11-01         : first version                
  
********************************************************************************/
{
       block_declarations;

       integer_small     k,l,n,m;
       integer_big       help;

                         
  n = mat_row (mat);
  m = mat_col (mat);

  *piv_el  = 0;
  
  for (k=1;k<=n;k++)
    if (mat_elt (zeile,k,1) < 0 )
    {
      for (l=1;l<=m;l++)
      {
        help = integer_abs (mat_elt (mat,k,l));
        if ((mat_elt (spalte,1,l) >0 ) && (integer_compare (*piv_el,help)  < 0))
        {  
          *piv_zeile   = k;
          *piv_spalte  = l;
          integer_delref (*piv_el);
          *piv_el      = help;
        }
      }
    }
}




integer_small 
_gj_tauschen WITH_1_ARG (matrix , zeile)
/*******************************************************************************
Description:               
   
   Checks, if it is still necassary to search for another pivot - element.
   The return value will be :

            0    <=>   1<= j <= n : zeile (j) > 0
            1    <=>   ex. j      : zeile (j) > 0


Calling sequence:            
    
        matrix            zeile       : indicates whether or not a certain row
                                        is admissible for searching.
                                                                          

     _gj_tauschen  (zeile);

                                                                     
History:  
                                                               
       MD 91-11-01         : first version                
  
********************************************************************************/
{
      block_declarations;

      integer_small ok,n,i;


  ok = 0;            
  n = mat_row (zeile);

  for (i=1;i<=n;i++)
  {
    if (mat_elt (zeile,i,1) < 0)
    {
      ok = 1;
      break;
    }
  }
  
  return ok;

}


void
_gj_io WITH_4_ARGS (t_handle  , cring,
                    matrix  , a,
                    matrix  , zeile,
                    matrix  , spalte)
/*******************************************************************************
Description:               
   
   An io - routine with some special features for the gauss - jordan - alg.

   The matrix mat is send to output with the flag-vectors zeile and spalte 
   surrounding it.


Calling sequence:            
        
       t_handle            cring   : the ring of integers
       matrix            a       : n x m matrix ; the output matix 
       matrix            zeile   : n x 1 matrix ; the row - flag matrix
       matrix            spalte  : 1 x m matrix ; thw column - flag matrix
                                       

     _gj_io  (cring,a,zeile,spalte);

                                                                     
History:  
                                                               
       MD 91-11-01         : first version                
  
********************************************************************************/
{   
     block_declarations;

     integer_small i,j,n,m;


  if (ring_type (cring) == RING_Z)
  {
    n = mat_row (a);
    m = mat_col (a);
     
    printf (" (%d x %d)\n\n",n,m);
                    
    printf ("        ");
    for (i=1;i<= m;i++)  
        cay_print ("%d         ",mat_elt (spalte,1,i));
    puts ("");

    for (i=1;i<= n;i++)  
    {
      cay_print ("%d    | ",mat_elt (zeile,i,1));
      for (j=1;j<=m; j++)
        cay_print ("%d         ",mat_elt (a,i,j));
      puts ("");
    }
  }
}



   
matrix
mat_z_kernel WITH_2_ARGS ( t_handle , cring,
                           matrix , mat   )
/*******************************************************************************
Description:               

    Calculates the full kernel of a given linear map over the rational integers.
    The computations are down using the Gauss - Jordan - Alg. . 
    The return value again is a matrix. There are two cases :

               1) the kernel is not trivial : Each column of the return matrix
                  represents one basis - vector of the kernel.
               2) the kernel is trivial     : The return matrix is just one 
                  column and this column is a zero - vector.

                            
Calling sequence:            

           t_handle       cring : The ring corresponding to the matrix mat.
                                Cring has to be a reference to ZZ.
           matrix       mat   : The matrix representing the linear map.
           
           matrix       kernel: The kernel of mat a described above.

    
        kernel = mat_z_kernel (cring,mat);

                                                                     
History:  
                                                               
       MD 91-12-100        : first version                
  
********************************************************************************/
{         
        block_declarations;


        matrix        zeile,spalte;
        matrix        a,temp_mat;

        integer_big   nenner,piv_el;
        integer_big   temp,temp1,temp2;
        integer_small n,m,i,j; 
        integer_small piv_zeile,piv_spalte,lsg_dim;




  if (ring_type (cring) !=  RING_Z) 
    error_internal ("MAT_Z_KERNEL : COEF. RING IST NICHT Z");

  n = mat_row (mat);
  m = mat_col (mat);

  if (n > m) 
    error_internal ("MAT_Z_KERNEL : SYSTEM IST UEBERBESTIMMT");


  a = mat_ring_copy (cring,mat);


  nenner = 1;                      
  zeile  = mat_new (n,1);
  spalte = mat_new (1,m);
  for (i=1;i<=n;i++)
    mat_elt (zeile,i,1) = -i;

  for (i=1;i<=m;i++)
    mat_elt (spalte,1,i) = i;

/*
  _gj_io (cring,mat,zeile,spalte);
*/  

  do                              
  {
    _gj_pivelement (cring,a,zeile,spalte,&piv_zeile,&piv_spalte,&piv_el);

/*      
    cay_print (" Pivotelement : %d (%d,%d)\n\n",piv_el,piv_zeile,piv_spalte);
*/

    if (piv_el != 0)
    {
      temp = mat_elt (zeile,piv_zeile,1);
      mat_elt (zeile,piv_zeile,1) = mat_elt (spalte,1,piv_spalte);
      mat_elt (spalte,1,piv_spalte) = temp;
             
      temp_mat = mat_new (n,m);
  

/* Pivotelement aendern */
      mat_elt (temp_mat,piv_zeile,piv_spalte) = integer_mult (nenner,nenner);
                          
/* Pivotzeile aendern  */
      for (i=1;i<=n;i++)
      { 
        if (i != piv_zeile)
          mat_elt (temp_mat,i,piv_spalte) = integer_mult (mat_elt (a,i,piv_spalte),nenner);
      }
             
/* Pivotspalte aendern  */
      for (i=1;i<=m;i++)
      { 
        if (i != piv_spalte)
        {
          temp =  integer_mult (mat_elt (a,piv_zeile,i),nenner);
          temp1 = integer_negate (temp);
          integer_delref (temp);
          mat_elt (temp_mat,piv_zeile,i) = temp1;
        }
      }
                          
/* Die restl. Elemente aendern */

      for (i=1;i<=n;i++)
        if (i != piv_zeile) 
        
          for (j=1;j<=m;j++)
            if (j != piv_spalte)
            {
              temp  = integer_mult (mat_elt (a,piv_zeile,piv_spalte),
                                  mat_elt (a,i,j)                  );
              temp1 = integer_mult (mat_elt (a,i,piv_spalte),
                                    mat_elt (a,piv_zeile,j)  );
              temp2 = integer_subtract (temp,temp1);
              integer_delref (temp);
              integer_delref (temp1);
              mat_elt (temp_mat,i,j) = temp2;
            }

      temp = nenner;
      nenner = integer_mult (mat_elt (a,piv_zeile,piv_spalte),temp);
      integer_delref (temp);
   
    

/* In der Matrix temp_mat steht nun die umgeformte neue Matrix */
/* Diese wird nun auf die Matrix a umgespeichert.              */

      mat_delref (cring,&a);
      a = temp_mat;         /* Auf die Matrix temp_mat zeigt nur ein t_handle.     */
                            /* Da die Matrix temp_mat im Moment nicht mehr       */
                            /* ben. wird reicht es also aus den t_handle umzulegen */
                            /* um temp_mat auf a zu kopieren.                    */
                                                                                
      mat_z_simplify (cring,a,&nenner); 
/*  
      cay_print ("Nenner : %d\n",nenner);
      _gj_io (cring,a,zeile,spalte);
      puts ("\n\n\n\n");
*/

    }                  
  }    
  while ((piv_el != 0) && (_gj_tauschen (zeile) != 0));

/*           
  puts (" Ende Vertauschen ");
*/

/* Bestimmung des Loesungsunterraumes */
   

  temp_mat = mat_ring_create_zero (cring,m);                           
  lsg_dim = 0;
  for (i=1;i<=m;i++)          
  {             
/*
    printf (" spalte(1,%d) :%d \n",i,mat_elt (spalte ,1,i));
*/
    if (mat_elt (spalte,1,i) >0) 
    {
      
      lsg_dim++;
      mat_elt (temp_mat,mat_elt (spalte,1,i),lsg_dim) = integer_incref (nenner);
      for (j=1;j<=n;j++)
        if (mat_elt (zeile,j,1) >0)
        { 
          mat_elt (temp_mat,mat_elt (zeile,j,1),lsg_dim) = integer_incref (mat_elt (a,j,i));
        }
/*                                  
     printf (" lsgdim : %d\n",lsg_dim);
     mat_anf_write (cring,temp_mat);
*/
    }               
  }   

  mat_delref (cring,&a);
  mat_delref (cring,&zeile);
  mat_delref (cring,&spalte);

  integer_delref (piv_el);

  if (lsg_dim >0) 
    a = mat_ring_submat (cring,temp_mat,1,1,m,lsg_dim);
  else
    a = mat_ring_submat (cring,temp_mat,1,1,m,1);                                                    

/* 
  mat_anf_write (cring,a);
*/





/*
  _gj_kuerzen (cring,a,&nenner); 
*/
                         
  mat_delref (cring,&temp_mat);
  integer_delref (nenner);

  return a;

}

