#include "kant.h"
#include "integer.e"
#include "poly.h"
#include "mat.h"
#include "anf.h"
 
/*
 
This file contains:
 
order_mult_table_create           (generic)
order_mult_table_create_pure      (internal)
order_mult_table_create_power     (internal)
order_mult_table_create_trans     (internal)
 
*/
 

void
order_mult_table_create WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
order_mult_table_create.c
 
JS September 1991                                               
Last modification: 21.10.91
 
Creates the multiplication table of the basis elements of the order.
Depending on what we know about the order different functions are used.
 
Until now is implemented:
   - pure equation orders
   - equation orders
   - order given via transition matrix of a suborder 
 
*******************************************************************************/
{
	void		order_mult_table_create_pure();
	void		order_mult_table_create_power();
	void		order_mult_table_create_trans();
 
	integer_small	mode;
 
	if(order_basis_is_pure(ord))
	{
		order_mult_table_create_pure(ord);
	}
	else if (order_basis_is_power(ord))
	{
		order_mult_table_create_power(ord);
	}
	else if (order_basis_is_rel(ord))
	{
		order_mult_table_create_trans(ord);
	}
	else
	{	
		error_internal("order_mult_table_create: Unknown situation.");
	}
 	
	if (anf_print_level > 3)
	{
		mode = (order_rel_degree(ord) < 30) ? 2 : 1;
		order_mult_table_write(ord, mode);
	}
		
	return;
}
  

void
order_mult_table_create_pure WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
order_mult_table_create_pure.c
 
JS September 1991
Last modification: 01.10.91
 
Initialization of the multiplication table of ord assuming that the basis 
is power basis of a pure polynomial. The generator is given implicitly via ord.
 
In future the flag order_pure_gen should be exploited by arithmetic routines
like anf_mult.
 
*******************************************************************************/
{
	block_declarations;
             
	integer_small	deg;
	integer_small	j, k, jsw, k1, k2;
	matrix		mat;
	anf_elt		alpha;
  
                            
	deg = order_rel_degree(ord);
 
	alpha = order_pure_gen(ord);
 
	order_mult_table_alloc(ord);
 
	for (k=1; k<=deg; ++k)
	{
		k1  = k - 1;
		k2  = k - 1 - deg;
		jsw = deg - k + 2;
 
		mat = order_mult_matrix(ord, k);
                
		for (j=1; j<jsw; ++j) 
				mat_elt(mat, j+k1, j) = 1;
 
		for (j=jsw; j<=deg; ++j) 
				mat_elt(mat, j+k2, j) = anf_elt_incref(alpha);
 	}
 
	return;
}



void
order_mult_table_create_power WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
order_mult_table_create_power.c
 
JS September 1991                                               
Last modification: 21.10.91
 
Creates the multiplication table of the basis elements of the order ord
assuming that the basis is power basis.
   
*******************************************************************************/
{
	block_declarations;
	
 
	integer_small	deg;
        order		ordcoef;
	integer_small	i, ilow, j, k, khalf, kup, ind, nterms;
	anf_elt		beta, gamma, temp0, temp1, mult;
	t_poly	pol;
	t_handle		polh;
 
 
	pol  = order_poly(ord);
	polh = m_poly_poly_to_handle(pol);
 	
        ordcoef = order_coef_order(ord);
	deg   = order_rel_degree(ord);

	order_mult_table_alloc(ord);
 
/*
    Now: We multiply b.e. (= basis element) i by b.e. j. In case of
    k:=i+j-1<=deg this yields b.e. k.
    We first deal with this easy case, exploiting symmetry.
*/               
	for (k=1; k<=deg; ++k)
	{               
		khalf = k / 2 + k % 2;
		for (i=1, j=k; i<=khalf; ++i, --j)
		{
   			mat_elt(order_mult_matrix(ord, i), k, j) = 1;
			if (i!=j) 
			mat_elt(order_mult_matrix(ord, j), k, i) = 1;
		}
	}
  
/*
    Now the case where b.e. i times b.e. j does not yield a pure basis
    element any more: We have to reduce modulo the minimal polynomial.
    The anf_elt gamma will successively contain subsequent
    powers of a root of the polynomial, starting with power deg.
    beta will at any time contain power deg of a root.
*/
	nterms = m_poly_nterms(polh)-1; 
 
        anf_elt_alloc(beta, deg);
        anf_elt_alloc(gamma, deg);
	anf_elt_den(gamma) = anf_elt_den(beta) = 1;
 
	for (i=1; i<=deg; ++i) anf_elt_coef(gamma, i) = 0;
 
	for (i=0; i<nterms; ++i)
	{
		anf_elt_coef(gamma, m_poly_expt(polh, i) + 1) = 
		anf_negate(ordcoef, m_poly_coefft(polh, i));
	}

	for (i=1; i<=deg; ++i)
 	    anf_elt_coef(beta, i)  = anf_elt_incref(anf_elt_coef(gamma, i));
 
	kup = deg + deg - 1;
 
	for (k=deg+1, ilow=2;; ++k, ++ilow)
	{
		khalf = k / 2 + k % 2;
		for (i=ilow, j=deg; i<=khalf; ++i, --j)
		{
			anf_elt_to_mat_order_col(ord, gamma, 
					order_mult_matrix(ord, i), j);
			if (i!=j) 
			anf_elt_to_mat_order_col(ord, gamma, 
					order_mult_matrix(ord, j), i);
		}
/*  
    now update of gamma: Multiplication by the root by shifting all
    coefficients down one step and reducing modulo the polynomial
    (which is in beta)
*/               
		if(k==kup) break;
 
		mult = anf_elt_incref(anf_elt_coef(gamma, deg));
 
		for (j=deg; j>1; --j)
		{
			temp0 = anf_elt_coef(gamma, j);
			temp1 = anf_mult(ordcoef, mult, anf_elt_coef(beta, j));
			
			anf_elt_coef(gamma, j) = 
		             anf_add(ordcoef, temp1, anf_elt_coef(gamma, j-1));
 
			anf_elt_delete(ordcoef, &temp0);
			anf_elt_delete(ordcoef, &temp1);
		}
 
		temp0 = anf_elt_coef(gamma, 1);
		anf_elt_coef(gamma, 1) = 
			     anf_mult(ordcoef, mult, anf_elt_coef(beta, 1));
 
		anf_elt_delete(ordcoef, &temp0);
		anf_elt_delete(ordcoef, &mult);
	}
	anf_elt_delete(ord, &beta);
	anf_elt_delete(ord, &gamma);
 	
	return;
}
 

void
order_mult_table_create_trans WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
order_mult_table_create_trans.c
 
JS September 1991  written                                             
MD April     1992  integer_delref (densq)   before return added
  
Last modification: 03.04.92
 
Creates the multiplication table of the basis elements of the order ord.
The basis must be given via a transformation matrix referring to another
order. In this suborder the multiplication must be possible.
If the suborder is an equation order and no multiplication table for that
order is given the table is computed automatically.
The same is tried if the suborder is given by another suborder.
   
 
*******************************************************************************/
{
	block_declarations;
 
	integer_small	deg;
        order		ordcoef;
        order		subord;
	integer_small	i, j;
	integer_big	den, densq;
	anf_elt		alpha, beta, temp0, temp1, gamma;
	matrix		trans, transinv;
 	
        ordcoef  = order_coef_order(ord);
	deg      = order_rel_degree(ord);
	trans    = order_tran(ord);
	transinv = order_invtran(ord);
	den      = order_tran_den(ord);
	densq    = integer_mult(den, den);
	subord   = order_suborder(ord);
 
	if (!order_mult_known(subord))	order_mult_table_create(subord);

	order_mult_table_alloc(ord);
 
/*
    We multiply basis element i by basis element j using the fact that
    the transition matrix gives us the representations of the basis 
    elements referring to the suborder basis.
    We also exploit the fact that the multiplication table is symmetric.
*/

	for (i=1; i<=deg; ++i)
	{
  
		alpha = mat_order_col_to_anf_elt(subord, trans, i);
 
		for (j=i; j<=deg; ++j)
		{
 
			beta  = mat_order_col_to_anf_elt(subord, trans, j);
 
			temp0 = anf_mult(subord, alpha, beta);
/*                      
     temp0 now contains the product. We convert it back and divide by
     the square of the denominator. This division must be possible.
*/
  
			temp1 = anf_elt_trans(subord, temp0, transinv, 1);
			gamma = anf_div_z(ord, temp1, densq);
/*
    now we have the product of the basis elements!
*/
			anf_elt_to_mat_order_col(ord, gamma, 
						 order_mult_matrix(ord, i), j);
			if (i!=j) 
			anf_elt_to_mat_order_col(ord, gamma, 
						 order_mult_matrix(ord, j), i);
 
			anf_elt_delref(subord, &beta);
			anf_elt_delref(subord, &temp0);
			anf_elt_delref(subord, &temp1);
			anf_elt_delref(subord, &gamma);
		}
		anf_elt_delref(subord, &alpha);
	}
 
	integer_delref (densq);
	return;
}
