/******************************************************************************
  order_units_indep_calc.c
******************************************************************************/
#include "kant.h"

t_void
order_units_indep_calc WITH_1_ARG (
	order,		ord
)
/******************************************************************************
 
Description:	Computes a maximal system of independent units and
		calculates its regulator.
		    
Calling sequence:
 
	order_units_indep_calc(ord);

	ord	: order		= t_handle of order

History:

	93-02-11 KW	new calling sequence
	92-10-13 KW	computation of regulator corrected
	92-08-12 KW	minor changes
	92-06-22 KW	written
 
******************************************************************************/
{
	block_declarations;

	anf_elt			eta;
	t_handle		R,Z;
	integer_small		i,j,k,ii,jj,iii,jjj;
	integer_small		r,r2,r12;
	matrix			echo,mat,ltrans,rtrans,inv_rtrans;
	t_real			tempu,tempv,tempw,tempx;
	vector			v,w;


	if (anf_print_level > 0)
		printf("Compute a maximal system of independent units...\n");

	if (order_units_are_maximal(ord))
	{
		order_reg_assure(ord);
		return;
	}

	r = order_r(ord);
	k = order_units_count(ord);
	R = order_reals(ord);
	Z = structure_z;

	r2 = order_r2(ord);
	r12 = order_r1(ord) + r2;

/*
**	We already know k independent units eta1,...,etak. Build
**		
**	    ( L(eta1) )
**          (         )
**	A = (    :    )
**	    ( 	      )
**	    ( L(etak) )
**
**	where L(eta) = (log|eta(1)|,.....,log|eta(r1+r2)|).
**	eta(m) denotes the m-th conjugate of eta.
*/

	if (k)
	{
		mat = mat_new(k,r12);
		for(i=1;i<=k;i++)
		{
			v = anf_elt_logs(ord,order_unit(ord,i));
			for (j=1;j<=r12;j++)
			{
				mat_elt(mat,i,j) = real_incref(vec_entry(v,j));
			}
			vec_delete(R,&v);
		}

/*
**	Echelonize A
*/
	        mat_real_echelon(R,mat,&echo,&ltrans,TRUE,&rtrans);
		mat_delref(R,&ltrans);
/*
**	Cast the right transformation matrix into an integer matrix
**	and compute its inverse.
*/
		mat_delref(R,&mat);
		mat = mat_real_to_mat_z(R,rtrans);
		mat_delref(R,&rtrans);
		rtrans = mat;
		inv_rtrans = mat_ring_inverse(Z,rtrans);


/*
**	Extend A up to r rows
*/
		mat = mat_new(r,r12);
		for (i=1;i<=k;i++)
		{
			for (j=1;j<=r12;j++)
			{
				mat_elt(mat,i,j) = real_incref(mat_elt(echo,i,j));
			}
		}
		mat_delref(R,&echo);
	}
	else
	{
		mat = mat_new(r,r12);
		rtrans = mat_ring_create_id(Z,r12);
		inv_rtrans = mat_incref(rtrans);
	}

/*
**
**
**	Compute r-k additional independent units to get a maximal
**	system of independent units.
**
**
*/
	order_units_count_set(ord,r);
	for(j=k+1;j<=r;j++)
	{
/*
**		Initialize conjugate set
*/
		v = vec_new(r12);
		for (i=1;i<j;i++) vec_entry(v,i) = -real_sign(R,mat_elt(mat,i,j));
		vec_entry(v,j) = 1;
		for (i=j+1;i<=r12;i++) vec_entry(v,i) = 0;
/*
**		Consider col swaps
*/
		w = mat_ring_mult(Z,v,inv_rtrans);
/*
**		Compute a unit eta
*/
		eta = order_unit_con_construct(ord,w);
		order_unit(ord,j) = eta;
		vec_delete(Z,&v);
		vec_delete(Z,&w);

/*
**		Update A (j-th row gets L(eta))
*/
		v = anf_elt_logs(ord,eta);
		w = mat_real_mat_z_mult(R,v,rtrans);
		for (jj=1;jj<=r12;jj++)
		{
			mat_elt(mat,j,jj) = real_incref(vec_entry(w,jj));
		}
		vec_delete(R,&v);
		vec_delete(R,&w);

/*
**		Compute row echolon form (see above)
**
**		A(j,i) = 0 for 1<=i<j 
*/
		for (jj=1;jj<j;jj++)
		{
			tempv = real_divide(R,mat_elt(mat,j,jj),mat_elt(mat,jj,jj));
			tempw = real_negate(R,tempv);
/*			mat_ring_row_add(R,mat,jj,j,tempw,j,r12);
*/			mat_ring_row_add(R,mat,jj,j,tempw,1,r12);
			real_delete(&tempv);
			real_delete(&tempw);
		}
/*
**		A(i,j) = 0 for 1<=i<j
*/
		tempu = real_inverse(R,mat_elt(mat,j,j));
		for (jj=1;jj<j;jj++)
		{
			tempv = real_mult(R,mat_elt(mat,jj,j),tempu);
			tempw = real_negate(R,tempv);
/*			mat_ring_row_add(R,mat,j,jj,tempw,j+1,r12);
*/			mat_ring_row_add(R,mat,j,jj,tempw,1,r12);
			real_delete(&tempv);
			real_delete(&tempw);
		}
		real_delete(&tempu);
	}

/*
**	Compute regulator
*/
	tempu = ring_one(R);
	for (i=1;i<=r;i++)
	{
		tempv = tempu;
		tempu = real_mult(R,tempu,mat_elt(mat,i,i));
		real_delete(&tempv);
	}
        i = (r2 < 2) ? 0 : r2-1;
        tempv = conv_int_to_real(R,2);
        tempw = real_power(R,tempv,i);
	order_reg(ord) = real_mult(R,tempu,tempw);
	real_delete(&tempu);
	real_delete(&tempv);
	real_delete(&tempw);

	mat_delref(R,&mat);
	mat_delref(Z,&rtrans);
	mat_delref(Z,&inv_rtrans);

	return;
}
