#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"


#ifdef USE_DOUBLE_QUOT_REM

/*
ib_quot_rem is a macro using doubles which needs the following constant.
*/

double DOUBLE_BETA = BETA;

#else

void
ib_quot_rem_func	WITH_5_ARGS(
    integer_small,	a1,
    integer_small,	a0,
    integer_small,	b,
    integer_small  *,	qptr,
    integer_small  *,	rptr
)
/*
The following macro, given *positive* beta-digits a1, a0 and b with a1 < b,
so that b is nonzero and

	a = a1 * BETA + a0

is a two-digit (or one-digit if a1 is zero) beta-representation of an integer
a with a < BETA * b, calculates the quotient q and dividend-signed remainder r
on dividing a by b, that is, the unique integers q and r such that 

	a = q * b + r  with  0 <= r < b (with q >= 0).

It returns q in *qptr and r in *rptr.

Note that a < BETA * b implies (a/b) < BETA, so that q really
can be represented as a single beta-digit.
*/
{
    register t_word		i;
    register unsigned long	a1t, a0t, bt, q;

    if (a1 == 0)
    {
	/*
	a is itself a small integer
	*/

	*qptr = a0 / b;
	*rptr = a0 - *qptr * b;
	return;
    }

    /*
    a is a genuine two-digit integer: take absolute values (signs
    will be adjusted later)
    */

    a1t = a1;
    a0t = a0;
    bt = b;

    /*
    Build up q = floor(at/bt) (where at = a1t*BETA + a0t) one binary
    digit at a time.  Note that to start with, 0 <= at < BETA*bt.  We shall
    show by induction that this inequality still holds at the end of the
    loop.
    */

    q = 0;

    for (i = 0; i < ZETA; i++)
    {
	/*  multiply at by 2 (overflow is impossible by choice of BETA)  */

	a1t <<= 1;
	a0t <<= 1;
	if (a0t >= BETA)
	{
	    a0t -= BETA;
	    a1t++;
	}

	/*  at this point, 0 <= at < 2*BETA*bt  */

	/*  multiply q by 2  */

	q <<= 1;

	/*
	if at >= BETA*bt, decrement at by BETA*bt and set low-order
	bit in q
	*/

	if (a1t >= bt)
	{
	    a1t -= bt;
	    q++;
	}

	/*
	If we did not have to decrement above, we still have

	    0 <= at < BETA*bt

	and if atold >= BETA*bt so we did decrement, we have

	    0 <= atnew = atold - BETA*bt < 2*BETA*bt - BETA*bt = BETA*bt

	completing the induction.
	*/
    }

    /*
    At this point, since BETA is 2 to the power ZETA,

	    at = BETA * abs(a) - q * BETA * bt

    so BETA divides at whence  a1t = abs(a) - q*bt  and  a0t = 0 .
    But by the induction argument we also have

	    0 <= BETA*a1t = at < BETA*bt

    whence 0 <= a1t < bt.

    Hence q is the quotient and a1t the remainder on dividing a by b.

    We now have a1t = a - q*bt with abs(a1t) < abs(b) and a1t*a >= 0.
    */

    *qptr = q;
    *rptr = a1t;
}
#endif
