#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"

void
integer_root    WITH_4_ARGS(
    integer_big,       a,
    integer_small,     n,
    integer_big *,     root,
    integer_small *,   sign
)
/*
 * Given a positive general integer a and a single precision
 * integer n >= 2, return root as the nth root of a, and sign as
 * sign(a - root^n).
 */
{
    block_declarations;

    register integer_small  k;
    register integer_small  ndash;
    register integer_small  h;
    register integer_small  u;
    register integer_big    b;
    register integer_big    c;
    integer_big             r;
    integer_big             d;
    register integer_big    e;
    register integer_big    f;
    register integer_big    adash;
    register integer_big    bdash;
    register integer_small  s;
    register integer_small  t;

    /*
     * Compute the first approximation
     */

    k = integer_log2(a);
    ndash = n - 1;
    h = k / n;
    r = k - h * n;
    u = 4 * r / n;
    if (h >= 2)
    {
	bdash = integer_power(2, h - 2);
	b = integer_mult(u + 5, bdash);
	integer_delref( bdash );
    }
    else
	b = 4;
    /*
     * Iterate the modified Newton method
     */

    for (;;)
    {
	c = integer_power(b, ndash);
	integer_quot_rem(a, c, &d, &r);
	integer_delref( c );

	s = integer_compare(b, d);
	if (s <= 0)
	{
	    integer_delref( d );
	    break;
	}

	integer_delref( r );

	e = integer_mult(b, ndash);
	integer_delref( b );

	f = integer_add(e, d);
	integer_delref( e );
	integer_delref( d );

	b = integer_div(f, n);
	integer_delref( f );
    }

    /*
     * Test whether result is too small
     */

    if (s == 0)
	t = integer_compare(r, 0);
    else
    {
	bdash = integer_add(b, 1);

	adash = integer_power(bdash, n);

	t = integer_compare(a, adash);
	integer_delref( adash );

	if (t >= 0)
	{
	    integer_delref( b );
	    b = bdash;		/* transfer reference from bdash to b */
	}
	else
	{
	    integer_delref( bdash );
	    t = 1;
	}
    }

    /*
     * Finished
     */

    integer_delref( r );

    *root = b;			/* transfer reference from b to *root */
    *sign = t;
}
