#include "defs.h"
#include "mp.e"
#include "mp.h"


/* Maximum length (in characters) of the exponent printed in decimal */
#define MAX_EXPT_LENGTH		32


void
mp_break_expt	WITH_5_ARGS(
	mp_float,	x,
	mp_base_type,	outbase,
	mp_length,	width,
	mp_length,	places,
	void_func,	act /* (char) */
)
/*
Breaks x into characters for output in exponent form, and sends
them through act().  The exponent is printed in decimal.  The parameters
width and places are the same as for mp_break().
Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_expt_type	expt, int_expt, output_exponent = 0;
    mp_base_type	b;
    mp_bool		leading = TRUE;
    mp_acc_float	frac_temp;
    mp_length		t, i, nmax, out_index, frac_t;
    mp_int		x_int_digits, limit, multiplier,
			next_digits, max_mult, j;
    char		*out_buf;


    DEBUG_BEGIN(DEBUG_BREAK);
    DEBUG_PRINTF_1("+break {\n");
    DEBUG_1("x = ", xp);

    if (mp_is_zero(xp))
    {
	/*
	Use mp_break()'s formatted output of 0.
	*/

	mp_break(x, outbase, width, places, act);
	return;
    }


    /*
    Copy parameters of x into temporary variables.
    */

    t = mp_t(xp);
    b = mp_b(xp);
    expt = mp_expt(xp);


    places = int_abs(places) + 1;

    nmax = mp_change_base(outbase, b, t) - 1;

    if (nmax > places)
	nmax = places;

    DEBUG_PRINTF_2("nmax = %d\n", nmax);


    /*
    Compute the highest power of outbase which we can multiply by.
    If b is a multiple of outbase, then use it since multiplication
    by b is trivial.
    */

    limit = MAX_INT / outbase;
    max_mult = outbase;

    while (max_mult <= limit && max_mult != b)
	max_mult *= outbase;


    /*
    Compute the digit precision for the temporary float variable frac_temp
    which will hold the fractional part of the output.  Allocate the
    context and float, copy x into the float, and remove the integer part.
    */

    frac_t = t + 1 + mp_guard_digits(int_max(50, max_mult), b);
    DEBUG_PRINTF_2("frac_t = %d\n", frac_t);

    mp_acc_float_alloc(b, frac_t, frac_temp);
    out_buf = (char *)mem_malloc(nmax + 5 + MAX_EXPT_LENGTH);

    mp_move(x, frac_temp);

    /*
    Make frac_temp positive since the sign has already been accounted for.
    */

    mp_set_sign(mp_acc_float_ptr(frac_temp), 1);

    /*
    Add rounding constant if necessary.
    */

    if (round == MP_TRUNC || round == MP_RND ||
	    mp_fix_directed(mp_sign(mp_ptr(x)), round) == MP_RND_UP)
    {
	mp_acc_float		rnd_temp;
	mp_round_type		save_round = round;


	mp_acc_float_alloc(b, frac_t, rnd_temp);

	if (round == MP_TRUNC || round == MP_RND)
	    mp_q_to_mp(1, 2, rnd_temp);

	else
	{
	    round = MP_RND_UP;

	    mp_int_to_mp(1, rnd_temp);
	}

	if (expt < 0)
	    mp_scale(rnd_temp, b, expt);

	if (nmax > 0)
	    mp_scale(rnd_temp, outbase, -nmax);


	mp_add_eq(frac_temp, rnd_temp);

	round = save_round;

	mp_acc_float_delete(rnd_temp);
    }

    if (expt > 0)
    {
	output_exponent = mp_change_base(outbase, b, expt);
	mp_scale(frac_temp, outbase, -output_exponent);
    }

    mp_to_frac(frac_temp, frac_temp);

    multiplier = 1;
    i = 0;

    out_index = 0;
    mp_change_up();

    while (i < nmax)
    {
	if (multiplier <= 1)
	{
	    mp_mul_int(frac_temp, max_mult, frac_temp);
	    next_digits = mp_to_int(frac_temp);
	    mp_to_frac(frac_temp, frac_temp);
	    multiplier = max_mult;
	}

	multiplier /= outbase;
	j = next_digits / multiplier;

	if (leading)
	    if (j)
	    {
		out_buf[out_index++] = j + '0';
		out_buf[out_index++] = '.';
		leading = FALSE;
		i++;
	    }
	    else
		output_exponent--;
	else
	{
	    out_buf[out_index++] = j + '0';
	    i++;
	}

	next_digits %= multiplier;
    }

    out_buf[out_index++] = 'e';
    sprintf(out_buf + out_index, "%d", output_exponent - 1);

    mp_break_out(out_buf, mp_sign(mp_ptr(x)), width, act);

    mp_change_down();
    mp_acc_float_delete(frac_temp);
    mem_free(out_buf);

    DEBUG_PRINTF_1("-} break\n");
    DEBUG_END();
}
